<?php
/**
 * Payment Reminder Manager
 * Handles automated payment reminders, warnings, and suspension logic
 */

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/functions.php';
require_once __DIR__ . '/email_functions.php';
require_once __DIR__ . '/sms_functions.php';

class PaymentReminderManager {
    
    private $pdo;
    private $academy_reference;
    
    public function __construct($academy_reference = null) {
        $this->pdo = getDB();
        $this->academy_reference = $academy_reference;
    }
    
    /**
     * Get tenant payment settings
     */
    public function getTenantSettings($academy_reference = null) {
        $academy_reference = $academy_reference ?? $this->academy_reference;
        
        $stmt = $this->pdo->prepare("
            SELECT * FROM tenant_payment_settings 
            WHERE academy_reference = ?
        ");
        $stmt->execute([$academy_reference]);
        return $stmt->fetch();
    }
    
    /**
     * Initialize payment schedule for a new student
     */
    public function initializeStudentSchedule($student_id, $academy_reference) {
        try {
            $stmt = $this->pdo->prepare("CALL sp_init_student_payment_schedule(?, ?)");
            $stmt->execute([$student_id, $academy_reference]);
            
            $this->logAudit([
                'action' => 'initialize_payment_schedule',
                'entity_type' => 'student',
                'entity_id' => $student_id,
                'academy_reference' => $academy_reference,
                'description' => 'Payment schedule initialized for new student'
            ]);
            
            return ['success' => true];
        } catch (Exception $e) {
            error_log("Failed to initialize payment schedule: " . $e->getMessage());
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Get students who need payment reminders
     */
    public function getStudentsNeedingReminders() {
        $stmt = $this->pdo->query("
            SELECT * FROM v_students_need_reminder
            ORDER BY days_since_payment DESC
        ");
        return $stmt->fetchAll();
    }
    
    /**
     * Get students who need warnings
     */
    public function getStudentsNeedingWarnings() {
        $stmt = $this->pdo->query("
            SELECT * FROM v_students_need_warning
            ORDER BY days_since_reminder DESC
        ");
        return $stmt->fetchAll();
    }
    
    /**
     * Get students to be suspended
     */
    public function getStudentsToSuspend() {
        $stmt = $this->pdo->query("
            SELECT * FROM v_students_to_suspend
            ORDER BY days_overdue DESC
        ");
        return $stmt->fetchAll();
    }
    
    /**
     * Send payment reminder to student
     */
    public function sendPaymentReminder($student_id, $reminder_type = 'initial_reminder') {
        try {
            // Get student details
            $stmt = $this->pdo->prepare("
                SELECT s.*, sps.*, tps.*,
                       DATEDIFF(CURDATE(), sps.last_payment_date) AS days_overdue
                FROM students s
                INNER JOIN student_payment_schedules sps ON s.id = sps.student_id
                INNER JOIN tenant_payment_settings tps ON s.academy_reference COLLATE utf8mb4_general_ci = tps.academy_reference COLLATE utf8mb4_general_ci
                WHERE s.id = ?
            ");
            $stmt->execute([$student_id]);
            $student = $stmt->fetch();
            
            if (!$student) {
                throw new Exception("Student not found");
            }
            
            // Generate payment link
            $payment_data = $this->generatePaymentLink($student_id, $student['total_amount_due']);
            
            // Create reminder record
            $stmt = $this->pdo->prepare("
                INSERT INTO payment_reminders (
                    student_id, academy_reference, schedule_id, reminder_type,
                    due_date, amount_due, payment_link, payment_token, token_expires_at
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $student_id,
                $student['academy_reference'],
                $student['id'], // schedule_id
                $reminder_type,
                $student['next_due_date'],
                $student['total_amount_due'],
                $payment_data['payment_link'],
                $payment_data['token'],
                $payment_data['expires_at']
            ]);
            
            $reminder_id = $this->pdo->lastInsertId();
            
            // Send notifications
            $notification_result = $this->sendNotifications($student, $reminder_type, $payment_data, $reminder_id);
            
            // Update schedule
            if ($reminder_type === 'initial_reminder') {
                $this->pdo->prepare("
                    UPDATE student_payment_schedules 
                    SET reminder_sent_at = NOW(), reminder_count = reminder_count + 1,
                        last_notification_at = NOW()
                    WHERE id = ?
                ")->execute([$student['id']]);
            } else if ($reminder_type === 'warning') {
                $this->pdo->prepare("
                    UPDATE student_payment_schedules 
                    SET warning_sent_at = NOW(), warning_count = warning_count + 1,
                        last_notification_at = NOW()
                    WHERE id = ?
                ")->execute([$student['id']]);
            }
            
            $this->logAudit([
                'action' => 'send_payment_reminder',
                'entity_type' => 'reminder',
                'entity_id' => $reminder_id,
                'academy_reference' => $student['academy_reference'],
                'description' => "Payment reminder ({$reminder_type}) sent to student {$student['full_name']}"
            ]);
            
            return [
                'success' => true,
                'reminder_id' => $reminder_id,
                'notifications' => $notification_result
            ];
            
        } catch (Exception $e) {
            error_log("Failed to send payment reminder: " . $e->getMessage());
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Generate secure payment link with token
     */
    private function generatePaymentLink($student_id, $amount) {
        // Generate token with embedded data
        $data = [
            'student_id' => $student_id,
            'amount' => $amount,
            'timestamp' => time(),
            'expires' => time() + (48 * 3600) // 48 hours
        ];
        
        // Encrypt token
        $token = $this->encryptToken($data);
        
        // Generate URL
        $base_url = defined('PUBLIC_BASE_URL') ? PUBLIC_BASE_URL : 
                   (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http') . 
                   '://' . $_SERVER['HTTP_HOST'];
        
        $payment_link = $base_url . "/pay_reminder.php?token=" . urlencode($token);
        
        return [
            'payment_link' => $payment_link,
            'token' => $token,
            'expires_at' => date('Y-m-d H:i:s', $data['expires'])
        ];
    }
    
    /**
     * Encrypt payment token
     */
    private function encryptToken($data) {
        $json = json_encode($data);
        $key = substr(hash('sha256', DB_NAME . DB_USER), 0, 32);
        $iv = substr(hash('sha256', 'payment_token'), 0, 16);
        
        $encrypted = openssl_encrypt($json, 'AES-256-CBC', $key, 0, $iv);
        return base64_encode($encrypted);
    }
    
    /**
     * Decrypt payment token
     */
    public function decryptToken($token) {
        try {
            $key = substr(hash('sha256', DB_NAME . DB_USER), 0, 32);
            $iv = substr(hash('sha256', 'payment_token'), 0, 16);
            
            $decrypted = openssl_decrypt(base64_decode($token), 'AES-256-CBC', $key, 0, $iv);
            $data = json_decode($decrypted, true);
            
            // Check expiry
            if ($data['expires'] < time()) {
                return ['success' => false, 'error' => 'Token expired'];
            }
            
            return ['success' => true, 'data' => $data];
        } catch (Exception $e) {
            return ['success' => false, 'error' => 'Invalid token'];
        }
    }
    
    /**
     * Send notifications (Email and SMS)
     */
    private function sendNotifications($student, $reminder_type, $payment_data, $reminder_id) {
        $settings = $this->getTenantSettings($student['academy_reference']);
        $results = [];
        
        // Send Email
        if ($settings['send_email_notifications']) {
            $email_result = $this->sendEmailNotification($student, $reminder_type, $payment_data);
            $results['email'] = $email_result;
            
            // Update reminder record
            $this->pdo->prepare("
                UPDATE payment_reminders 
                SET email_sent = ?, email_sent_at = NOW(), email_error = ?
                WHERE id = ?
            ")->execute([
                $email_result['success'] ? 1 : 0,
                $email_result['error'] ?? null,
                $reminder_id
            ]);
        }
        
        // Send SMS
        if ($settings['send_sms_notifications'] && !empty($student['phone'])) {
            $sms_result = $this->sendSMSNotification($student, $reminder_type, $payment_data);
            $results['sms'] = $sms_result;
            
            // Update reminder record
            $this->pdo->prepare("
                UPDATE payment_reminders 
                SET sms_sent = ?, sms_sent_at = NOW(), sms_error = ?
                WHERE id = ?
            ")->execute([
                $sms_result['success'] ? 1 : 0,
                $sms_result['error'] ?? null,
                $reminder_id
            ]);
        }
        
        return $results;
    }
    
    /**
     * Send email notification
     */
    private function sendEmailNotification($student, $reminder_type, $payment_data) {
        try {
            $subject = $this->getEmailSubject($reminder_type);
            $body = $this->getEmailBody($student, $reminder_type, $payment_data);
            
            $result = sendEmail($student['email'], $student['full_name'], $subject, $body);
            
            return ['success' => $result];
        } catch (Exception $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Send SMS notification
     */
    private function sendSMSNotification($student, $reminder_type, $payment_data) {
        try {
            $message = $this->getSMSBody($student, $reminder_type, $payment_data);
            
            require_once __DIR__ . '/sms_functions.php';
            $result = sendSMS($student['phone'], $message, 'PAYMENT_REMINDER', $student['academy_reference']);
            
            return $result;
        } catch (Exception $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Get email subject based on reminder type
     */
    private function getEmailSubject($reminder_type) {
        switch ($reminder_type) {
            case 'initial_reminder':
                return 'Payment Reminder - Your Account Fee is Due';
            case 'warning':
                return 'URGENT: Payment Overdue - Action Required';
            case 'final_warning':
                return 'FINAL WARNING: Account Suspension Imminent';
            case 'suspension_notice':
                return 'Account Suspended - Payment Required';
            default:
                return 'Payment Notice';
        }
    }
    
    /**
     * Get email body template
     */
    private function getEmailBody($student, $reminder_type, $payment_data) {
        $school_name = $student['academy_name'] ?? APP_NAME;
        $amount = number_format($student['total_amount_due'], 2);
        $days_overdue = $student['days_overdue'] ?? 0;
        
        $html = '
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <style>
        body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; background: #f4f4f4; margin: 0; padding: 0; }
        .container { max-width: 600px; margin: 20px auto; background: #ffffff; border-radius: 10px; overflow: hidden; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        .header { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 30px; text-align: center; }
        .content { padding: 30px; }
        .amount-box { background: #f8f9fa; border-left: 4px solid #667eea; padding: 20px; margin: 20px 0; }
        .amount { font-size: 32px; font-weight: bold; color: #667eea; }
        .button { display: inline-block; background: #667eea; color: white !important; padding: 15px 40px; text-decoration: none; border-radius: 5px; font-weight: bold; margin: 20px 0; }
        .button:hover { background: #5568d3; }
        .warning { background: #fff3cd; border-left: 4px solid #ffc107; padding: 15px; margin: 20px 0; }
        .danger { background: #f8d7da; border-left: 4px solid #dc3545; padding: 15px; margin: 20px 0; }
        .footer { background: #f8f9fa; padding: 20px; text-align: center; font-size: 12px; color: #666; }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1 style="margin: 0;">' . htmlspecialchars($school_name) . '</h1>
            <p style="margin: 10px 0 0;">Payment Notice</p>
        </div>
        
        <div class="content">
            <p>Dear ' . htmlspecialchars($student['full_name']) . ',</p>';
        
        if ($reminder_type === 'initial_reminder') {
            $html .= '
            <p>This is a friendly reminder that your payment for the current period is due.</p>
            
            <div class="amount-box">
                <p style="margin: 0 0 10px; color: #666;">Amount Due:</p>
                <div class="amount">SZL ' . $amount . '</div>
                <p style="margin: 10px 0 0; color: #666;">Due Date: ' . date('F j, Y', strtotime($student['next_due_date'])) . '</p>
            </div>
            
            <p>Please make your payment at your earliest convenience to avoid any interruption to your account.</p>';
        } else if ($reminder_type === 'warning') {
            $html .= '
            <div class="warning">
                <strong>⚠️ Payment Overdue</strong>
                <p style="margin: 10px 0 0;">Your payment is now ' . $days_overdue . ' days overdue.</p>
            </div>
            
            <div class="amount-box">
                <p style="margin: 0 0 10px; color: #666;">Outstanding Balance:</p>
                <div class="amount">SZL ' . $amount . '</div>
            </div>
            
            <p><strong>Action Required:</strong> Please make your payment within 3 days to avoid account suspension.</p>';
        } else if ($reminder_type === 'suspension_notice') {
            $html .= '
            <div class="danger">
                <strong>🚫 Account Suspended</strong>
                <p style="margin: 10px 0 0;">Your account has been temporarily suspended due to non-payment.</p>
            </div>
            
            <div class="amount-box">
                <p style="margin: 0 0 10px; color: #666;">Amount Required to Reactivate:</p>
                <div class="amount">SZL ' . $amount . '</div>
            </div>
            
            <p>Your access to the learning platform has been suspended. Please make payment immediately to restore access.</p>';
        }
        
        $html .= '
            <div style="text-align: center;">
                <a href="' . htmlspecialchars($payment_data['payment_link']) . '" class="button">
                    💳 Pay Now - SZL ' . $amount . '
                </a>
            </div>
            
            <p style="font-size: 14px; color: #666; margin-top: 30px;">
                <strong>Payment Methods Available:</strong><br>
                • MTN Mobile Money<br>
                • Credit/Debit Card
            </p>
            
            <p style="font-size: 12px; color: #999; margin-top: 20px;">
                This payment link is valid for 48 hours. If you have already made payment, please disregard this notice.
            </p>
        </div>
        
        <div class="footer">
            <p><strong>' . htmlspecialchars($school_name) . '</strong></p>
            <p>If you have any questions, please contact school administration.</p>
            <p style="margin-top: 15px; color: #999;">© ' . date('Y') . ' All rights reserved.</p>
        </div>
    </div>
</body>
</html>';
        
        return $html;
    }
    
    /**
     * Get SMS body template (160 characters max)
     */
    private function getSMSBody($student, $reminder_type, $payment_data) {
        $amount = number_format($student['total_amount_due'], 2);
        $school = $student['academy_name'] ?? APP_NAME;
        
        if ($reminder_type === 'initial_reminder') {
            return "Hi {$student['full_name']}, payment of SZL {$amount} is due. Pay now: {$payment_data['payment_link']} - {$school}";
        } else if ($reminder_type === 'warning') {
            return "URGENT: Payment SZL {$amount} overdue. Pay within 3 days to avoid suspension: {$payment_data['payment_link']} - {$school}";
        } else if ($reminder_type === 'suspension_notice') {
            return "Account suspended. Pay SZL {$amount} to reactivate: {$payment_data['payment_link']} - {$school}";
        }
        
        return "Payment reminder: SZL {$amount} due. Pay: {$payment_data['payment_link']} - {$school}";
    }
    
    /**
     * Process successful payment
     */
    public function processPaymentSuccess($transaction_id) {
        try {
            $stmt = $this->pdo->prepare("CALL sp_process_payment_success(?)");
            $stmt->execute([$transaction_id]);
            
            // Get student details
            $stmt = $this->pdo->prepare("
                SELECT pt.*, s.full_name, s.email, s.phone, s.academy_reference
                FROM payment_transactions pt
                INNER JOIN students s ON pt.student_id = s.id
                WHERE pt.id = ?
            ");
            $stmt->execute([$transaction_id]);
            $transaction = $stmt->fetch();
            
            // Send confirmation notifications
            $this->sendPaymentConfirmation($transaction);
            
            // Unsuspend in Moodle if was suspended
            $this->unsuspendInMoodle($transaction['student_id']);
            
            $this->logAudit([
                'action' => 'process_payment_success',
                'entity_type' => 'payment',
                'entity_id' => $transaction_id,
                'academy_reference' => $transaction['academy_reference'],
                'description' => "Payment processed successfully for {$transaction['full_name']}"
            ]);
            
            return ['success' => true];
        } catch (Exception $e) {
            error_log("Failed to process payment success: " . $e->getMessage());
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Send payment confirmation
     */
    private function sendPaymentConfirmation($transaction) {
        // Email confirmation
        $subject = "Payment Received - Thank You!";
        $body = $this->getPaymentConfirmationEmail($transaction);
        sendEmail($transaction['email'], $transaction['full_name'], $subject, $body);
        
        // SMS confirmation
        if (!empty($transaction['phone'])) {
            $message = "Payment of SZL " . number_format($transaction['amount'], 2) . " received. Thank you! Your account is now active.";
            sendSMS($transaction['phone'], $message, 'PAYMENT_CONFIRMATION', $transaction['academy_reference']);
        }
    }
    
    /**
     * Get payment confirmation email
     */
    private function getPaymentConfirmationEmail($transaction) {
        $amount = number_format($transaction['amount'], 2);
        
        return '
<!DOCTYPE html>
<html>
<head>
    <style>
        body { font-family: Arial, sans-serif; line-height: 1.6; }
        .success-box { background: #d4edda; border-left: 4px solid #28a745; padding: 20px; margin: 20px 0; }
    </style>
</head>
<body>
    <div class="success-box">
        <h2 style="color: #28a745; margin: 0 0 10px;">✅ Payment Received!</h2>
        <p>Thank you for your payment of <strong>SZL ' . $amount . '</strong></p>
    </div>
    
    <p>Dear ' . htmlspecialchars($transaction['full_name']) . ',</p>
    
    <p>We have received your payment. Your account is now active and in good standing.</p>
    
    <p><strong>Payment Details:</strong></p>
    <ul>
        <li>Amount: SZL ' . $amount . '</li>
        <li>Reference: ' . htmlspecialchars($transaction['transaction_reference']) . '</li>
        <li>Date: ' . date('F j, Y g:i A', strtotime($transaction['paid_at'])) . '</li>
        <li>Method: ' . strtoupper($transaction['payment_method']) . '</li>
    </ul>
    
    <p>Your next payment will be due on ' . date('F j, Y', strtotime('+30 days')) . '</p>
    
    <p>Thank you for your continued trust in our services.</p>
</body>
</html>';
    }
    
    /**
     * Suspend student in Moodle
     */
    public function suspendInMoodle($student_id, $reason = 'payment_overdue') {
        try {
            require_once __DIR__ . '/moodle_suspension_handler.php';
            $handler = new MoodleSuspensionHandler();
            return $handler->suspendUser($student_id, $reason);
        } catch (Exception $e) {
            error_log("Moodle suspension failed: " . $e->getMessage());
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Unsuspend student in Moodle
     */
    public function unsuspendInMoodle($student_id) {
        try {
            require_once __DIR__ . '/moodle_suspension_handler.php';
            $handler = new MoodleSuspensionHandler();
            return $handler->unsuspendUser($student_id);
        } catch (Exception $e) {
            error_log("Moodle unsuspension failed: " . $e->getMessage());
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Log audit trail
     */
    private function logAudit($data) {
        try {
            $stmt = $this->pdo->prepare("
                INSERT INTO payment_system_audit_log (
                    user_id, user_type, academy_reference, action, 
                    entity_type, entity_id, description, ip_address, user_agent
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");
            
            $stmt->execute([
                $_SESSION['admin_id'] ?? null,
                isset($_SESSION['admin_id']) ? 'admin' : 'system',
                $data['academy_reference'],
                $data['action'],
                $data['entity_type'],
                $data['entity_id'] ?? null,
                $data['description'],
                $_SERVER['REMOTE_ADDR'] ?? null,
                $_SERVER['HTTP_USER_AGENT'] ?? null
            ]);
        } catch (Exception $e) {
            error_log("Failed to log audit: " . $e->getMessage());
        }
    }
}

