<?php
/**
 * Moodle Enrollment Handler
 * Handles automatic student enrollment to Moodle courses
 * Uses subject-to-course mapping for conflict-free enrollment
 */

require_once __DIR__ . '/moodle_api.php';

class MoodleEnrollmentHandler {
    
    private $pdo;
    private $moodle;
    private $academy_reference;
    
    public function __construct($academy_reference, $pdo) {
        $this->pdo = $pdo;
        $this->academy_reference = $academy_reference;
        $this->moodle = new MoodleAPI($academy_reference, $pdo);
    }
    
    /**
     * Enroll student in a subject's Moodle course
     * Uses mapping table to find correct course
     */
    public function enrollStudentInSubject($student_id, $subject_id) {
        try {
            // Get student details
            $stmt = $this->pdo->prepare("
                SELECT id, username, email, full_name, first_name, last_name, moodle_user_id 
                FROM students 
                WHERE id = ?
            ");
            $stmt->execute([$student_id]);
            $student = $stmt->fetch();
            
            if (!$student) {
                throw new Exception("Student not found");
            }
            
            // Get Moodle course mapping
            $stmt = $this->pdo->prepare("
                SELECT * FROM subject_moodle_mapping 
                WHERE academy_reference = ? 
                AND subject_id = ?
                AND auto_enroll = 1
                AND sync_enabled = 1
            ");
            $stmt->execute([$this->academy_reference, $subject_id]);
            $mapping = $stmt->fetch();
            
            if (!$mapping) {
                $this->logEnrollment($student_id, $subject_id, null, null, 'enroll', 'failed', 
                    'No Moodle mapping configured for this subject');
                return [
                    'success' => false,
                    'error' => 'No Moodle mapping configured'
                ];
            }
            
            // Check if Moodle is enabled
            if (!$this->moodle->isEnabled()) {
                $this->logEnrollment($student_id, $subject_id, null, null, 'enroll', 'failed', 
                    'Moodle integration not enabled');
                return [
                    'success' => false,
                    'error' => 'Moodle not enabled'
                ];
            }
            
            // Ensure student exists in Moodle
            $moodle_user_id = $this->ensureUserInMoodle($student);
            
            // Enroll in course
            $result = $this->moodle->enrollUser(
                $moodle_user_id,
                $mapping['moodle_course_id'],
                $mapping['enrollment_role'] ?: 'student'
            );
            
            if ($result['success']) {
                // Update student's moodle_user_id if needed
                if ($student['moodle_user_id'] != $moodle_user_id) {
                    $this->pdo->prepare("UPDATE students SET moodle_user_id = ? WHERE id = ?")
                              ->execute([$moodle_user_id, $student_id]);
                }
                
                // Log success
                $this->logEnrollment(
                    $student_id, 
                    $subject_id, 
                    $mapping['moodle_course_id'], 
                    $moodle_user_id, 
                    'enroll', 
                    'success',
                    'Successfully enrolled'
                );
                
                // Update mapping sync status
                $this->pdo->prepare("
                    UPDATE subject_moodle_mapping 
                    SET last_sync_at = NOW(), sync_status = 'success', sync_error = NULL
                    WHERE id = ?
                ")->execute([$mapping['id']]);
            } else {
                // Log failure
                $this->logEnrollment(
                    $student_id, 
                    $subject_id, 
                    $mapping['moodle_course_id'], 
                    $moodle_user_id, 
                    'enroll', 
                    'failed',
                    $result['error'] ?? 'Unknown error'
                );
            }
            
            return $result;
            
        } catch (Exception $e) {
            $this->logEnrollment($student_id, $subject_id, null, null, 'enroll', 'failed', $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Enroll student in multiple subjects at once
     * Used when subjects are assigned in bulk
     */
    public function enrollStudentInMultipleSubjects($student_id, $subject_ids) {
        $results = [];
        $success_count = 0;
        $fail_count = 0;
        
        foreach ($subject_ids as $subject_id) {
            $result = $this->enrollStudentInSubject($student_id, $subject_id);
            $results[$subject_id] = $result;
            
            if ($result['success']) {
                $success_count++;
            } else {
                $fail_count++;
            }
        }
        
        return [
            'success' => $success_count > 0,
            'total' => count($subject_ids),
            'enrolled' => $success_count,
            'failed' => $fail_count,
            'results' => $results
        ];
    }
    
    /**
     * Unenroll student from a subject's Moodle course
     */
    public function unenrollStudentFromSubject($student_id, $subject_id) {
        try {
            $stmt = $this->pdo->prepare("SELECT moodle_user_id FROM students WHERE id = ?");
            $stmt->execute([$student_id]);
            $student = $stmt->fetch();
            
            if (!$student || !$student['moodle_user_id']) {
                return ['success' => false, 'error' => 'Student not found in Moodle'];
            }
            
            $stmt = $this->pdo->prepare("
                SELECT * FROM subject_moodle_mapping 
                WHERE academy_reference = ? AND subject_id = ?
            ");
            $stmt->execute([$this->academy_reference, $subject_id]);
            $mapping = $stmt->fetch();
            
            if (!$mapping) {
                return ['success' => false, 'error' => 'No mapping found'];
            }
            
            $result = $this->moodle->unenrollUser(
                $student['moodle_user_id'],
                $mapping['moodle_course_id']
            );
            
            $this->logEnrollment(
                $student_id, 
                $subject_id, 
                $mapping['moodle_course_id'], 
                $student['moodle_user_id'], 
                'unenroll', 
                $result['success'] ? 'success' : 'failed',
                $result['error'] ?? 'Unenrolled successfully'
            );
            
            return $result;
            
        } catch (Exception $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Ensure user exists in Moodle, create if not
     */
    private function ensureUserInMoodle($student) {
        // Check if user already exists in Moodle
        if ($student['moodle_user_id']) {
            return $student['moodle_user_id'];
        }
        
        // Create user in Moodle
        $student_payload = $student;
        $student_payload['id'] = $student['id'];
        if (empty($student_payload['first_name'])) {
            $student_payload['first_name'] = explode(' ', $student['full_name'])[0] ?? $student['username'];
        }
        if (empty($student_payload['last_name'])) {
            $nameParts = explode(' ', $student['full_name']);
            $student_payload['last_name'] = count($nameParts) > 1 ? end($nameParts) : 'Student';
        }
        
        $result = $this->moodle->createUser($student_payload);
        
        if ($result['success'] && isset($result['moodle_user_id'])) {
            return $result['moodle_user_id'];
        }
        
        throw new Exception('Could not create user in Moodle: ' . ($result['error'] ?? 'Unknown error'));
    }
    
    /**
     * Log enrollment attempt to database
     */
    private function logEnrollment($student_id, $subject_id, $moodle_course_id, $moodle_user_id, 
                                   $action, $status, $error_message = null) {
        try {
            $stmt = $this->pdo->prepare("
                INSERT INTO moodle_enrollment_log 
                (student_id, subject_id, academy_reference, moodle_course_id, moodle_user_id, 
                 action, status, error_message, attempted_at, completed_at)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())
            ");
            $stmt->execute([
                $student_id,
                $subject_id,
                $this->academy_reference,
                $moodle_course_id,
                $moodle_user_id,
                $action,
                $status,
                $error_message
            ]);
        } catch (Exception $e) {
            // Log error but don't fail the main operation
            error_log("Failed to log Moodle enrollment: " . $e->getMessage());
        }
    }
    
    /**
     * Sync all enrolled students for a subject to Moodle
     */
    public function syncSubjectEnrollments($subject_id) {
        try {
            // Get all students enrolled in this subject
            $stmt = $this->pdo->prepare("
                SELECT DISTINCT s.id, s.academy_reference
                FROM students s
                INNER JOIN student_subject ss ON s.id = ss.student_id
                WHERE ss.subject_id = ? AND s.academy_reference = ?
            ");
            $stmt->execute([$subject_id, $this->academy_reference]);
            $students = $stmt->fetchAll();
            
            $results = [];
            foreach ($students as $student) {
                $results[] = $this->enrollStudentInSubject($student['id'], $subject_id);
            }
            
            return [
                'success' => true,
                'synced' => count($students),
                'results' => $results
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Sync all subjects for a student to Moodle
     */
    public function syncStudentEnrollments($student_id) {
        try {
            // Get all subjects this student is enrolled in
            $stmt = $this->pdo->prepare("
                SELECT subject_id FROM student_subject WHERE student_id = ?
            ");
            $stmt->execute([$student_id]);
            $subject_ids = array_column($stmt->fetchAll(), 'subject_id');
            
            return $this->enrollStudentInMultipleSubjects($student_id, $subject_ids);
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Queue enrollment for background processing
     * Useful for bulk operations
     */
    public function queueEnrollment($student_id, $subject_id, $priority = 5) {
        try {
            $stmt = $this->pdo->prepare("
                INSERT INTO moodle_sync_queue 
                (operation_type, priority, student_id, subject_id, academy_reference, payload, status)
                VALUES ('enroll', ?, ?, ?, ?, ?, 'pending')
            ");
            $stmt->execute([
                $priority,
                $student_id,
                $subject_id,
                $this->academy_reference,
                json_encode(['timestamp' => time()])
            ]);
            
            return ['success' => true, 'queue_id' => $this->pdo->lastInsertId()];
            
        } catch (Exception $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Process pending queue items
     * Should be called by cron job
     */
    public function processQueue($limit = 10) {
        try {
            // Get pending items
            $stmt = $this->pdo->prepare("
                SELECT * FROM moodle_sync_queue 
                WHERE status = 'pending' 
                AND academy_reference = ?
                AND attempts < max_attempts
                AND scheduled_at <= NOW()
                ORDER BY priority DESC, created_at ASC
                LIMIT ?
            ");
            $stmt->execute([$this->academy_reference, $limit]);
            $items = $stmt->fetchAll();
            
            $processed = 0;
            foreach ($items as $item) {
                // Mark as processing
                $this->pdo->prepare("UPDATE moodle_sync_queue SET status = 'processing', started_at = NOW() WHERE id = ?")
                          ->execute([$item['id']]);
                
                // Process based on operation type
                if ($item['operation_type'] === 'enroll') {
                    $result = $this->enrollStudentInSubject($item['student_id'], $item['subject_id']);
                } elseif ($item['operation_type'] === 'unenroll') {
                    $result = $this->unenrollStudentFromSubject($item['student_id'], $item['subject_id']);
                }
                
                // Update queue item
                if ($result['success']) {
                    $this->pdo->prepare("
                        UPDATE moodle_sync_queue 
                        SET status = 'completed', completed_at = NOW() 
                        WHERE id = ?
                    ")->execute([$item['id']]);
                    $processed++;
                } else {
                    $attempts = $item['attempts'] + 1;
                    $next_retry = date('Y-m-d H:i:s', time() + (60 * pow(2, $attempts))); // Exponential backoff
                    
                    $this->pdo->prepare("
                        UPDATE moodle_sync_queue 
                        SET status = 'failed', 
                            attempts = ?,
                            error_message = ?,
                            next_retry_at = ?
                        WHERE id = ?
                    ")->execute([$attempts, $result['error'], $next_retry, $item['id']]);
                }
            }
            
            return [
                'success' => true,
                'processed' => $processed,
                'total' => count($items)
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
}

