<?php
/**
 * Moodle API Integration Class
 * Handles communication with Moodle LMS via Web Services
 */

class MoodleAPI {
    private $moodle_url;
    private $token;
    private $service_name;
    private $academy_reference;
    private $pdo;
    private $enabled = false;
    
    /**
     * Initialize Moodle API with school-specific configuration
     */
    public function __construct($academy_reference, $pdo = null) {
        $this->academy_reference = $academy_reference;
        $this->pdo = $pdo ?? getDB();
        
        // Load Moodle configuration from tenant_settings
        $this->loadConfiguration();
    }
    
    /**
     * Load Moodle configuration from database
     */
    private function loadConfiguration() {
        try {
            // tenant_settings is a key-value table, so we need to fetch multiple rows
            $stmt = $this->pdo->prepare("
                SELECT setting_key, setting_value
                FROM tenant_settings
                WHERE academy_reference = ? 
                AND setting_key IN ('moodle_url', 'moodle_token', 'moodle_enabled', 'moodle_service_name')
            ");
            $stmt->execute([$this->academy_reference]);
            $settings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
            
            if ($settings) {
                $this->moodle_url = rtrim($settings['moodle_url'] ?? '', '/');
                $this->token = $settings['moodle_token'] ?? '';
                $this->service_name = $settings['moodle_service_name'] ?? 'moodle_mobile_app';
                $this->enabled = (bool)($settings['moodle_enabled'] ?? false);
                
                // Log for debugging
                error_log("Moodle config loaded for {$this->academy_reference}: URL=" . 
                         (!empty($this->moodle_url) ? 'SET' : 'EMPTY') . 
                         ", Token=" . (!empty($this->token) ? 'SET' : 'EMPTY') . 
                         ", Enabled=" . ($this->enabled ? 'YES' : 'NO'));
            } else {
                error_log("No Moodle settings found for academy: {$this->academy_reference}");
            }
        } catch (PDOException $e) {
            error_log("Moodle config load error: " . $e->getMessage());
        }
    }
    
    /**
     * Check if Moodle integration is enabled and configured
     */
    public function isEnabled() {
        return $this->enabled && !empty($this->moodle_url) && !empty($this->token);
    }
    
    /**
     * Make API call to Moodle
     */
    private function callMoodleAPI($function, $params = []) {
        if (!$this->isEnabled()) {
            return ['success' => false, 'error' => 'Moodle integration is not enabled or configured'];
        }
        
        $start_time = microtime(true);
        
        // Build API endpoint URL
        $url = $this->moodle_url . '/webservice/rest/server.php';
        
        // Prepare POST data
        $post_data = [
            'wstoken' => $this->token,
            'wsfunction' => $function,
            'moodlewsrestformat' => 'json'
        ];
        
        // Add function parameters
        foreach ($params as $key => $value) {
            $post_data[$key] = $value;
        }
        
        // Make cURL request
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($post_data));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false); // For development - enable in production!
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curl_error = curl_error($ch);
        curl_close($ch);
        
        $execution_time = microtime(true) - $start_time;
        
        // Parse response
        if ($curl_error) {
            $this->logSync('api_call', null, null, null, 'failed', 
                json_encode($post_data), null, "cURL Error: $curl_error", $execution_time);
            return ['success' => false, 'error' => "Connection error: $curl_error"];
        }
        
        $result = json_decode($response, true);
        
        // Check for Moodle errors
        if (isset($result['exception']) || isset($result['errorcode'])) {
            $error = $result['message'] ?? $result['debuginfo'] ?? 'Unknown Moodle error';
            $this->logSync('api_call', null, null, null, 'failed', 
                json_encode($post_data), $response, $error, $execution_time);
            return ['success' => false, 'error' => $error, 'response' => $result];
        }
        
        $this->logSync('api_call', null, null, null, 'success', 
            json_encode($post_data), $response, null, $execution_time);
        
        return ['success' => true, 'data' => $result];
    }
    
    /**
     * Get all courses from Moodle
     */
    public function getCourses() {
        $result = $this->callMoodleAPI('core_course_get_courses');
        
        if ($result['success']) {
            // Cache courses in database
            $this->cacheCourses($result['data']);
        }
        
        return $result;
    }
    
    /**
     * Cache Moodle courses in local database
     */
    private function cacheCourses($courses) {
        try {
            $stmt = $this->pdo->prepare("
                INSERT INTO moodle_course_cache 
                (moodle_course_id, course_name, shortname, category_id, visible, startdate, enddate, academy_reference)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE
                course_name = VALUES(course_name),
                shortname = VALUES(shortname),
                category_id = VALUES(category_id),
                visible = VALUES(visible),
                startdate = VALUES(startdate),
                enddate = VALUES(enddate),
                last_synced = CURRENT_TIMESTAMP
            ");
            
            foreach ($courses as $course) {
                if ($course['id'] == 1) continue; // Skip site home course
                
                $stmt->execute([
                    $course['id'],
                    $course['fullname'],
                    $course['shortname'],
                    $course['categoryid'] ?? null,
                    $course['visible'] ?? true,
                    $course['startdate'] ?? null,
                    $course['enddate'] ?? null,
                    $this->academy_reference
                ]);
            }
        } catch (PDOException $e) {
            error_log("Moodle course cache error: " . $e->getMessage());
        }
    }
    
    /**
     * Create user in Moodle
     */
    public function createUser($student) {
        // Prepare user data
        $user_data = [
            'users[0][username]' => $student['username'],
            'users[0][password]' => $this->generateMoodlePassword($student),
            'users[0][firstname]' => $student['first_name'] ?? explode(' ', $student['full_name'])[0],
            'users[0][lastname]' => $student['last_name'] ?? explode(' ', $student['full_name'])[1] ?? 'Student',
            'users[0][email]' => $student['email'],
            'users[0][auth]' => 'manual',
            'users[0][lang]' => 'en',
            'users[0][timezone]' => 'Africa/Mbabane'
        ];
        
        $result = $this->callMoodleAPI('core_user_create_users', $user_data);
        
        if ($result['success'] && !empty($result['data'])) {
            $moodle_user_id = $result['data'][0]['id'] ?? null;
            
            if ($moodle_user_id) {
                // Update student record with Moodle user ID
                $this->updateStudentMoodleId($student['id'], $moodle_user_id, $student['username']);
                
                $this->logSync('user_create', $student['id'], null, null, 'success', 
                    json_encode($user_data), json_encode($result['data']), null, 0);
                
                return ['success' => true, 'moodle_user_id' => $moodle_user_id];
            }
        }
        
        // If creation failed, check if user already exists in Moodle
        $existing_user = $this->findExistingMoodleUser($student);
        if ($existing_user && !empty($existing_user['id'])) {
            $moodle_user_id = $existing_user['id'];
            $this->updateStudentMoodleId($student['id'], $moodle_user_id, $student['username']);
            
            $this->logSync('user_lookup', $student['id'], null, null, 'success',
                json_encode(['username' => $student['username'], 'email' => $student['email']]),
                json_encode($existing_user), 'Matched existing Moodle user', 0);
            
            return [
                'success' => true,
                'moodle_user_id' => $moodle_user_id,
                'existing' => true,
                'message' => 'Existing Moodle user linked'
            ];
        }
        
        $error_message = $result['error'] ?? 'Failed to create Moodle user';
        $response_payload = $result['response'] ?? ($result['data'] ?? $result);
        $this->logSync('user_create', $student['id'], null, null, 'failed',
            json_encode($user_data), json_encode($response_payload), $error_message, 0);
        
        return ['success' => false, 'error' => $error_message, 'response' => $result];
    }

    /**
     * Enroll an existing Moodle user in a course
     */
    public function enrollUser($moodle_user_id, $moodle_course_id, $role = 'student') {
        if (!$this->isEnabled()) {
            return ['success' => false, 'error' => 'Moodle integration is not enabled or configured'];
        }

        $role_id = $this->mapRoleToId($role);

        $enrol_data = [
            'enrolments[0][roleid]' => $role_id,
            'enrolments[0][userid]' => $moodle_user_id,
            'enrolments[0][courseid]' => $moodle_course_id
        ];

        return $this->callMoodleAPI('enrol_manual_enrol_users', $enrol_data);
    }

    /**
     * Unenroll an existing Moodle user from a course
     */
    public function unenrollUser($moodle_user_id, $moodle_course_id) {
        if (!$this->isEnabled()) {
            return ['success' => false, 'error' => 'Moodle integration is not enabled or configured'];
        }

        $unenrol_data = [
            'enrolments[0][userid]' => $moodle_user_id,
            'enrolments[0][courseid]' => $moodle_course_id
        ];

        return $this->callMoodleAPI('enrol_manual_unenrol_users', $unenrol_data);
    }

    /**
     * Map role string to Moodle role ID
     */
    private function mapRoleToId($role) {
        $roles = [
            'manager' => 1,
            'coursecreator' => 2,
            'teacher' => 3,
            'editingteacher' => 4,
            'student' => 5,
            'guest' => 6,
            'user' => 7,
            'frontpage' => 8
        ];

        if (is_numeric($role)) {
            return (int)$role;
        }

        $role_key = strtolower($role ?? '');
        return $roles[$role_key] ?? 5;
    }

    /**
     * Attempt to locate existing Moodle user by username or email
     */
    private function findExistingMoodleUser($student) {
        // Check by username first
        if (!empty($student['username'])) {
            $user = $this->getUserByField('username', $student['username']);
            if ($user) {
                return $user;
            }
        }

        // Fall back to searching by email
        if (!empty($student['email'])) {
            $user = $this->getUserByField('email', $student['email']);
            if ($user) {
                return $user;
            }
        }

        return null;
    }

    /**
     * Fetch a Moodle user using a specific field
     */
    private function getUserByField($field, $value) {
        if (empty($value)) {
            return null;
        }

        $result = $this->callMoodleAPI('core_user_get_users_by_field', [
            'field' => $field,
            'values[0]' => $value
        ]);

        if ($result['success'] && !empty($result['data'])) {
            return $result['data'][0];
        }

        return null;
    }
    
    /**
     * Enroll student in a Moodle course
     */
    public function enrollStudent($student_id, $subject_id, $moodle_course_id) {
        // Get student info
        $stmt = $this->pdo->prepare("SELECT * FROM students WHERE id = ?");
        $stmt->execute([$student_id]);
        $student = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$student) {
            return ['success' => false, 'error' => 'Student not found'];
        }
        
        // Check if student exists in Moodle, create if not
        if (empty($student['moodle_user_id'])) {
            $create_result = $this->createUser($student);
            if (!$create_result['success']) {
                return $create_result;
            }
            $moodle_user_id = $create_result['moodle_user_id'];
        } else {
            $moodle_user_id = $student['moodle_user_id'];
        }
        
        // Enroll user in course (roleid 5 = student role)
        $enroll_data = [
            'enrolments[0][roleid]' => 5,
            'enrolments[0][userid]' => $moodle_user_id,
            'enrolments[0][courseid]' => $moodle_course_id
        ];
        
        $result = $this->callMoodleAPI('enrol_manual_enrol_users', $enroll_data);
        
        // Log enrollment
        $status = $result['success'] ? 'enrolled' : 'failed';
        $error = $result['success'] ? null : ($result['error'] ?? 'Unknown error');
        
        $this->logEnrollment($student_id, $subject_id, $moodle_course_id, 
            $moodle_user_id, $status, 'auto_payment', $error);
        
        $this->logSync('enrollment', $student_id, $subject_id, $moodle_course_id, 
            $status, json_encode($enroll_data), json_encode($result), $error, 0);
        
        return $result;
    }
    
    /**
     * Unenroll student from a Moodle course
     */
    public function unenrollStudent($student_id, $moodle_course_id) {
        $stmt = $this->pdo->prepare("SELECT moodle_user_id FROM students WHERE id = ?");
        $stmt->execute([$student_id]);
        $student = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$student || empty($student['moodle_user_id'])) {
            return ['success' => false, 'error' => 'Student not found in Moodle'];
        }
        
        $unenroll_data = [
            'enrolments[0][userid]' => $student['moodle_user_id'],
            'enrolments[0][courseid]' => $moodle_course_id
        ];
        
        $result = $this->callMoodleAPI('enrol_manual_unenrol_users', $unenroll_data);
        
        // Update enrollment log
        if ($result['success']) {
            $stmt = $this->pdo->prepare("
                UPDATE moodle_enrollments 
                SET enrollment_status = 'unenrolled', updated_at = CURRENT_TIMESTAMP
                WHERE student_id = ? AND moodle_course_id = ?
            ");
            $stmt->execute([$student_id, $moodle_course_id]);
        }
        
        return $result;
    }
    
    /**
     * Generate password for Moodle (can be customized)
     */
    private function generateMoodlePassword($student) {
        // Use student username + default suffix, or generate random
        return $student['username'] . '@School2024'; // Change as needed
    }
    
    /**
     * Update student with Moodle user ID
     */
    private function updateStudentMoodleId($student_id, $moodle_user_id, $moodle_username) {
        try {
            $stmt = $this->pdo->prepare("
                UPDATE students 
                SET moodle_user_id = ?, moodle_username = ?, 
                    moodle_synced = TRUE
                WHERE id = ?
            ");
            $stmt->execute([$moodle_user_id, $moodle_username, $student_id]);
        } catch (PDOException $e) {
            error_log("Moodle user ID update error: " . $e->getMessage());
        }
    }
    
    /**
     * Log enrollment attempt
     */
    private function logEnrollment($student_id, $subject_id, $moodle_course_id, 
        $moodle_user_id, $status, $method, $error = null) {
        try {
            $stmt = $this->pdo->prepare("
                INSERT INTO moodle_enrollments 
                (student_id, subject_id, moodle_course_id, moodle_user_id, 
                 enrollment_status, enrollment_method, enrolled_at, error_message, academy_reference)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE
                enrollment_status = VALUES(enrollment_status),
                enrolled_at = VALUES(enrolled_at),
                error_message = VALUES(error_message),
                updated_at = CURRENT_TIMESTAMP
            ");
            
            $enrolled_at = ($status === 'enrolled') ? date('Y-m-d H:i:s') : null;
            
            $stmt->execute([
                $student_id, $subject_id, $moodle_course_id, $moodle_user_id,
                $status, $method, $enrolled_at, $error, $this->academy_reference
            ]);
        } catch (PDOException $e) {
            error_log("Moodle enrollment log error: " . $e->getMessage());
        }
    }
    
    /**
     * Log sync activity
     */
    private function logSync($sync_type, $student_id, $subject_id, $moodle_course_id, 
        $status, $request_data, $response_data, $error_message, $execution_time) {
        try {
            $stmt = $this->pdo->prepare("
                INSERT INTO moodle_sync_log 
                (sync_type, student_id, subject_id, moodle_course_id, status, 
                 request_data, response_data, error_message, execution_time, academy_reference)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");
            
            $stmt->execute([
                $sync_type, $student_id, $subject_id, $moodle_course_id, $status,
                $request_data, $response_data, $error_message, $execution_time, 
                $this->academy_reference
            ]);
        } catch (PDOException $e) {
            error_log("Moodle sync log error: " . $e->getMessage());
        }
    }
    
    /**
     * Test Moodle connection
     */
    public function testConnection() {
        if (!$this->isEnabled()) {
            return [
                'success' => false, 
                'message' => 'Moodle integration is not enabled or configured'
            ];
        }
        
        // Try to get site info
        $result = $this->callMoodleAPI('core_webservice_get_site_info');
        
        if ($result['success']) {
            return [
                'success' => true,
                'message' => 'Connected successfully to Moodle',
                'site_name' => $result['data']['sitename'] ?? 'Unknown',
                'moodle_version' => $result['data']['release'] ?? 'Unknown'
            ];
        }
        
        return [
            'success' => false,
            'message' => $result['error'] ?? 'Connection failed'
        ];
    }
}

