<?php
// Load Composer autoload if available (guards for servers without vendor folder)
$composerAutoload = __DIR__ . '/../vendor/autoload.php';
if (file_exists($composerAutoload)) {
    require_once $composerAutoload;
}
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/security_headers.php';
require_once __DIR__ . '/email_functions.php';

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception as MailException;
// Session management
function startSession() {
    if (session_status() === PHP_SESSION_NONE && !headers_sent()) {
        session_start();
        }
}

// CSRF protection
function generateCSRFToken() {
    startSession();
    if (empty($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

function validateCSRFToken($token) {
    startSession();
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

// Student management
function getStudentCount() {
    $pdo = getDB();
    $stmt = $pdo->query("SELECT COUNT(*) as count FROM students");
    return $stmt->fetch()['count'];
}

function isStudentLimitReached() {
    return getStudentCount() >= MAX_STUDENTS;
}

function getGrades() {
    $pdo = getDB();
    $stmt = $pdo->query("SELECT * FROM grades ORDER BY name");
    return $stmt->fetchAll();
}

function getSubjectsForGrade($gradeId) {
    $pdo = getDB();
    $stmt = $pdo->prepare("
        SELECT s.* FROM subjects s 
        JOIN grade_subject gs ON s.id = gs.subject_id 
        WHERE gs.grade_id = ?
        ORDER BY s.name
    ");
    $stmt->execute([$gradeId]);
    return $stmt->fetchAll();
}

function getAllSubjects() {
    $pdo = getDB();
    $stmt = $pdo->query("SELECT * FROM subjects ORDER BY name");
    return $stmt->fetchAll();
}

function getStudents($limit = null, $offset = 0, $includeSuspended = false) {
    $pdo = getDB();
    $sql = "SELECT s.*, g.name as grade_name FROM students s JOIN grades g ON s.grade_id = g.id";
    
    // Exclude suspended students by default
    if (!$includeSuspended) {
        $sql .= " WHERE (s.status IS NULL OR s.status != 'suspended')";
    }
    
    $sql .= " ORDER BY s.created_at DESC";
    
    if ($limit) {
        $sql .= " LIMIT $limit OFFSET $offset";
    }
    $stmt = $pdo->query($sql);
    return $stmt->fetchAll();
}

function getStudentById($id) {
    $pdo = getDB();
    $stmt = $pdo->prepare("SELECT s.*, g.name as grade_name FROM students s JOIN grades g ON s.grade_id = g.id WHERE s.id = ?");
    $stmt->execute([$id]);
    return $stmt->fetch();
}

function assignSubjectsToStudent($studentId, array $subjectIds) {
    $pdo = getDB();
    if (empty($subjectIds)) return;
    $stmt = $pdo->prepare("INSERT IGNORE INTO student_subject (student_id, subject_id) VALUES (?, ?)");
    foreach ($subjectIds as $sid) {
        $stmt->execute([$studentId, (int)$sid]);
    }
}

function getSubjectsForStudent($studentId) {
    $pdo = getDB();
    $stmt = $pdo->prepare("SELECT s.* FROM student_subject ss JOIN subjects s ON ss.subject_id = s.id WHERE ss.student_id = ? ORDER BY s.name");
    $stmt->execute([$studentId]);
    return $stmt->fetchAll();
}

function createStudent($data) {
    $pdo = getDB();
    
    // Check student limit
    if (isStudentLimitReached()) {
        throw new Exception('Maximum number of students reached (' . MAX_STUDENTS . ')');
    }
    
    // Auto-generate username: YYYYMMDDNN (daily sequence starting at 01)
    $prefix = date('Ymd');
    $stmtLast = $pdo->prepare("SELECT username FROM students WHERE username LIKE ? ORDER BY username DESC LIMIT 1");
    $stmtLast->execute([$prefix . '%']);
    $lastUsername = $stmtLast->fetchColumn();
    $seq = 1;
    if ($lastUsername && strpos($lastUsername, $prefix) === 0) {
        $tail = substr($lastUsername, 8);
        $seq = max(1, (int)$tail + 1);
    }
    $username = $prefix . str_pad((string)$seq, 2, '0', STR_PAD_LEFT);
    // Collision safety loop
    $checkStmt = $pdo->prepare('SELECT 1 FROM students WHERE username = ? LIMIT 1');
    while (true) {
        $checkStmt->execute([$username]);
        if (!$checkStmt->fetch()) break;
        $seq++;
        $username = $prefix . str_pad((string)$seq, 2, '0', STR_PAD_LEFT);
    }
    
    // Use provided password if present, else generate
    $password = isset($data['password_plain']) && is_string($data['password_plain']) && strlen($data['password_plain']) >= 8
        ? $data['password_plain']
        : bin2hex(random_bytes(8));
    $passwordHash = password_hash($password, PASSWORD_DEFAULT);
    
    $stmt = $pdo->prepare("
        INSERT INTO students (first_name, last_name, email, phone, grade_id, username, password_hash, email_verified, academy_reference) 
        VALUES (?, ?, ?, ?, ?, ?, ?, FALSE, ?)
    ");
    
    $stmt->execute([
        $data['first_name'],
        $data['last_name'],
        $data['email'],
        $data['phone'] ?? null,
        $data['grade_id'],
        $username,
        $passwordHash,
        $data['academy_reference'] ?? 'KINE' // Use provided or default to KINE
    ]);
    
    $studentId = $pdo->lastInsertId();

    // If provided, validate subject choices belong to the student's grade and save
    if (!empty($data['subject_ids']) && is_array($data['subject_ids'])) {
        $placeholders = implode(',', array_fill(0, count($data['subject_ids']), '?'));
        $validSubjectsStmt = $pdo->prepare('SELECT s.id FROM subjects s JOIN grade_subject gs ON gs.subject_id = s.id WHERE gs.grade_id = ? AND s.id IN (' . $placeholders . ')');
        $params = array_merge([$data['grade_id']], array_map('intval', $data['subject_ids']));
        $validSubjectsStmt->execute($params);
        $validIds = array_column($validSubjectsStmt->fetchAll(), 'id');
        assignSubjectsToStudent($studentId, $validIds);
    }
    
    // Generate email verification token and send verification email
    $verificationToken = createEmailVerificationToken($studentId);
    sendEmailVerification($data['email'], $data['first_name'], $verificationToken);
    
    // Note: Login credentials will be sent after email verification
    // This is handled in the markEmailAsVerified function
    
    // Update Excel file
    updateStudentsExcel();
    
    // Trigger bot to sync new student to Moodle
    triggerMoodleSyncBot();
    
    return $studentId;
}

/**
 * Format username to match what the bot will use in Moodle
 * This ensures consistency between email credentials and Moodle usernames
 */
function formatUsernameForMoodle($username) {
    $formattedUsername = trim($username);

    // Fix usernames that start with 0 or are purely numeric (Moodle doesn't like these)
    if (preg_match('/^0/', $formattedUsername) || preg_match('/^\d+$/', $formattedUsername)) {
        // Convert to a valid format: add 'user_' prefix
        $formattedUsername = 'user_' . $formattedUsername;
    }

    // Convert to lowercase for consistency
    $formattedUsername = strtolower($formattedUsername);

    return $formattedUsername;
}

function generateSecurePasswordForMoodle($lastname) {
    /**
     * Generate a secure password that meets Moodle's strict requirements
     * Uses the user's lastname as base and adds required characters
     * This function MUST match the Python bot's password generation logic
     */
    $lastname = trim((string)$lastname);
    
    // Ensure lastname is at least 3 characters
    if (strlen($lastname) < 3) {
        $lastname = $lastname . "123";
    }
    
    // Create a secure password that meets ALL Moodle requirements:
    // - Minimum 8 characters (preferably 12+)
    // - At least one uppercase letter
    // - At least one lowercase letter  
    // - At least one number
    // - At least one special character
    
    // Start with the full lastname (this makes it more personal and memorable)
    $password = strtolower($lastname);
    
    // Add required uppercase letter (use first letter of lastname capitalized)
    $password .= strtoupper($lastname[0]);
    
    // Add required numbers (use a consistent pattern based on lastname length)
    $numberSuffix = strlen($lastname) . (strlen($lastname) + 1);
    $password .= $numberSuffix;
    
    // Add required special character (use a consistent one)
    $password .= "!";
    
    // If password is still too short, add more characters
    if (strlen($password) < 12) {
        $additionalChars = 12 - strlen($password);
        // Add more numbers to reach 12 characters
        for ($i = 0; $i < $additionalChars; $i++) {
            $password .= ($i + 2); // Add sequential numbers
        }
    }
    
    // Ensure it meets all requirements
    if (!validatePasswordRequirements($password)) {
        // If it doesn't meet requirements, create a guaranteed valid password
        $password = createGuaranteedValidPassword($lastname);
    }
    
    return $password;
}

function validatePasswordRequirements($password) {
    /**
     * Validate that password meets Moodle requirements
     */
    if (strlen($password) < 8) {
        return false;
    }
    
    $hasUpper = preg_match('/[A-Z]/', $password);
    $hasLower = preg_match('/[a-z]/', $password);
    $hasDigit = preg_match('/[0-9]/', $password);
    $hasSpecial = preg_match('/[!@#$%^&*()_+\-=\[\]{}|;:,.<>?]/', $password);
    
    return $hasUpper && $hasLower && $hasDigit && $hasSpecial;
}

function createGuaranteedValidPassword($lastname) {
    /**
     * Create a password that is guaranteed to meet all requirements
     */
    // Template: [lastname][UPPER][lower][digit][special][random]
    $password = strtolower(substr($lastname, 0, 3)); // First 3 chars of lastname
    $password .= chr(rand(65, 90)); // Uppercase
    $password .= chr(rand(97, 122)); // Lowercase
    $password .= rand(0, 9); // Digit
    $password .= "!@#$%^&*"[rand(0, 7)]; // Special char
    
    // Add random characters to reach 12+ length
    for ($i = 0; $i < 7; $i++) { // Total length will be 12
        $password .= chr(rand(97, 122)) . rand(0, 9); // Mix of letters and numbers
    }
    
    return $password;
}

function triggerMoodleSyncBot() {
    /**
     * Trigger the PHP Moodle sync bot to run immediately
     * This ensures new students are created in Moodle and enrolled in courses
     */
    try {
        // Check if bot is already running
        $lockFile = 'bot/temp/sync.lock';
        if (file_exists($lockFile)) {
            error_log('Moodle sync bot is already running, skipping trigger');
            return false;
        }
        
        // Trigger PHP bot execution directly
        $botScript = 'bot/moodle_sync_bot.php';
        
        if (!file_exists($botScript)) {
            error_log('Bot script not found: ' . $botScript);
            return false;
        }
        
        // Execute bot in background
        if (PHP_OS_FAMILY === 'Windows') {
            // Windows
            $command = "start /B php \"$botScript\" > nul 2>&1";
            pclose(popen($command, 'r'));
        } else {
            // Linux/Unix
            $command = "php \"$botScript\" > /dev/null 2>&1 &";
            exec($command);
        }
        
        error_log('Moodle sync bot triggered successfully');
        return true;
        
    } catch (Exception $e) {
        error_log('Exception triggering Moodle sync bot: ' . $e->getMessage());
        return false;
    }
}

// Moodle user management functions
function unenrollStudentFromMoodle($student) {
    /**
     * Unenroll student from all courses in Moodle
     */
    try {
        $moodleUsername = formatUsernameForMoodle($student['username']);
        
        // Get Moodle user ID
        $moodleUserId = getMoodleUserId($moodleUsername);
        if (!$moodleUserId) {
            error_log("Cannot unenroll student {$moodleUsername}: User not found in Moodle");
            return false;
        }
        
        // Get all courses the user is enrolled in
        $enrollments = getMoodleUserEnrollments($moodleUserId);
        if (empty($enrollments)) {
            error_log("Student {$moodleUsername} is not enrolled in any courses");
            return true; // Not an error, just nothing to unenroll
        }
        
        // Unenroll from all courses
        $unenrollData = [];
        foreach ($enrollments as $enrollment) {
            $unenrollData[] = [
                'userid' => $moodleUserId,
                'courseid' => $enrollment['courseid']
            ];
        }
        
        $result = callMoodleAPI('enrol_manual_unenrol_users', [
            'enrolments' => $unenrollData
        ]);
        
        if ($result && !isset($result['exception'])) {
            error_log("Successfully unenrolled student {$moodleUsername} from " . count($unenrollData) . " courses");
            return true;
        } else {
            error_log("Failed to unenroll student {$moodleUsername}: " . json_encode($result));
            return false;
        }
        
    } catch (Exception $e) {
        error_log("Exception unenrolling student from Moodle: " . $e->getMessage());
        return false;
    }
}

function suspendStudentInMoodle($student) {
    /**
     * Suspend student account in Moodle
     */
    try {
        $moodleUsername = formatUsernameForMoodle($student['username']);
        
        // Get Moodle user ID
        $moodleUserId = getMoodleUserId($moodleUsername);
        if (!$moodleUserId) {
            error_log("Cannot suspend student {$moodleUsername}: User not found in Moodle");
            return false;
        }
        
        // Suspend the user account
        $result = callMoodleAPI('core_user_update_users', [
            'users' => [[
                'id' => $moodleUserId,
                'suspended' => 1, // 1 = suspended, 0 = active
                'suspendedreason' => 'Account deleted from admin portal'
            ]]
        ]);
        
        if ($result && !isset($result['exception'])) {
            error_log("Successfully suspended student {$moodleUsername} in Moodle");
            return true;
        } else {
            error_log("Failed to suspend student {$moodleUsername}: " . json_encode($result));
            return false;
        }
        
    } catch (Exception $e) {
        error_log("Exception suspending student in Moodle: " . $e->getMessage());
        return false;
    }
}

function getMoodleUserId($username) {
    /**
     * Get Moodle user ID by username
     */
    try {
        $result = callMoodleAPI('core_user_get_users_by_field', [
            'field' => 'username',
            'values' => [$username]
        ]);
        
        if ($result && !isset($result['exception']) && !empty($result['users'])) {
            return $result['users'][0]['id'];
        }
        
        return null;
    } catch (Exception $e) {
        error_log("Exception getting Moodle user ID: " . $e->getMessage());
        return null;
    }
}

function getMoodleUserEnrollments($userId) {
    /**
     * Get all course enrollments for a Moodle user
     */
    try {
        $result = callMoodleAPI('core_enrol_get_users_courses', [
            'userid' => $userId
        ]);
        
        if ($result && !isset($result['exception'])) {
            return $result;
        }
        
        return [];
    } catch (Exception $e) {
        error_log("Exception getting Moodle user enrollments: " . $e->getMessage());
        return [];
    }
}

function callMoodleAPI($function, $params = []) {
    /**
     * Make API call to Moodle
     */
    try {
        $url = MOODLE_API_URL . '?wstoken=' . MOODLE_API_TOKEN . 
               '&wsfunction=' . $function . '&moodlewsrestformat=json';
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($params));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode !== 200) {
            error_log("Moodle API call failed with HTTP code: {$httpCode}");
            return false;
        }
        
        $data = json_decode($response, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            error_log("Failed to decode Moodle API response: " . json_last_error_msg());
            return false;
        }
        
        return $data;
        
    } catch (Exception $e) {
        error_log("Exception calling Moodle API: " . $e->getMessage());
        return false;
    }
}

function restoreStudentInMoodle($student) {
    /**
     * Restore (unsuspend) student account in Moodle
     * Useful for reactivating students who were previously deleted
     */
    try {
        $moodleUsername = formatUsernameForMoodle($student['username']);
        
        // Get Moodle user ID
        $moodleUserId = getMoodleUserId($moodleUsername);
        if (!$moodleUserId) {
            error_log("Cannot restore student {$moodleUsername}: User not found in Moodle");
            return false;
        }
        
        // Restore the user account (unsuspend)
        $result = callMoodleAPI('core_user_update_users', [
            'users' => [[
                'id' => $moodleUserId,
                'suspended' => 0, // 0 = active, 1 = suspended
                'suspendedreason' => ''
            ]]
        ]);
        
        if ($result && !isset($result['exception'])) {
            error_log("Successfully restored student {$moodleUsername} in Moodle");
            return true;
        } else {
            error_log("Failed to restore student {$moodleUsername}: " . json_encode($result));
            return false;
        }
        
    } catch (Exception $e) {
        error_log("Exception restoring student in Moodle: " . $e->getMessage());
        return false;
    }
}

function updateStudent($id, $data) {
    $pdo = getDB();
    $stmt = $pdo->prepare("
        UPDATE students 
        SET first_name = ?, last_name = ?, email = ?, grade_id = ? 
        WHERE id = ?
    ");
    
    $stmt->execute([
        $data['first_name'],
        $data['last_name'],
        $data['email'],
        $data['grade_id'],
        $id
    ]);

    $rowsAffected = $stmt->rowCount();

    // Optionally update subjects if provided
    if (isset($data['subject_ids']) && is_array($data['subject_ids'])) {
        $pdo->prepare('DELETE FROM student_subject WHERE student_id = ?')->execute([$id]);
        if (!empty($data['subject_ids'])) {
            $placeholders = implode(',', array_fill(0, count($data['subject_ids']), '?'));
            $validSubjectsStmt = $pdo->prepare('SELECT s.id FROM subjects s JOIN grade_subject gs ON gs.subject_id = s.id WHERE gs.grade_id = ? AND s.id IN (' . $placeholders . ')');
            $params = array_merge([$data['grade_id']], array_map('intval', $data['subject_ids']));
            $validSubjectsStmt->execute($params);
            $validIds = array_column($validSubjectsStmt->fetchAll(), 'id');
            assignSubjectsToStudent($id, $validIds);
        }
    }
    
    // Update Excel file
    updateStudentsExcel();
    
    // Trigger bot to sync updated student to Moodle
    triggerMoodleSyncBot();

    if ($rowsAffected === 0) {
        $existsStmt = $pdo->prepare('SELECT id FROM students WHERE id = ?');
        $existsStmt->execute([$id]);
        return (bool)$existsStmt->fetchColumn();
    }
    
    return true;
}

function deleteStudent($id) {
    $pdo = getDB();
    
    try {
        // Get student details before soft deletion
        $stmt = $pdo->prepare("SELECT * FROM students WHERE id = ?");
        $stmt->execute([$id]);
        $student = $stmt->fetch();
        
        if (!$student) {
            error_log("Delete student failed: Student ID {$id} not found");
            return false;
        }
        
        // Step 1: Try to unenroll student from all courses in Moodle (non-blocking)
        try {
            $unenrollResult = unenrollStudentFromMoodle($student);
        } catch (Exception $e) {
            error_log("Moodle unenroll failed (non-critical): " . $e->getMessage());
            $unenrollResult = false;
        }
        
        // Step 2: Try to suspend student account in Moodle (non-blocking)
        try {
            $suspendResult = suspendStudentInMoodle($student);
        } catch (Exception $e) {
            error_log("Moodle suspend failed (non-critical): " . $e->getMessage());
            $suspendResult = false;
        }
        
        // Step 3: Soft delete - mark student as suspended (CRITICAL - must succeed)
        $stmt = $pdo->prepare("
            UPDATE students 
            SET status = 'suspended',
                suspended = 1,
                updated_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute([$id]);
        $softDeleteResult = $stmt->rowCount() > 0;
        
        if (!$softDeleteResult) {
            error_log("Delete student failed: Could not update student status for ID {$id}");
            return false;
        }
        
        // Step 4: Update Excel file (exclude suspended students) - non-blocking
        try {
            if (function_exists('updateStudentsExcel')) {
                updateStudentsExcel();
            }
        } catch (Exception $e) {
            error_log("Excel update failed (non-critical): " . $e->getMessage());
        }
        
        // Log the operation
        error_log("Student soft deletion completed - ID: {$id}, Unenroll: " . ($unenrollResult ? 'SUCCESS' : 'FAILED') . 
                  ", Suspend: " . ($suspendResult ? 'SUCCESS' : 'FAILED') . 
                  ", Status Update: SUCCESS");
        
        return true;
        
    } catch (Exception $e) {
        error_log("Delete student exception: " . $e->getMessage());
        return false;
    }
}

function permanentlyDeleteStudent($id) {
    /**
     * Permanently delete student from system (admin override)
     * This completely removes the student from both local database and Moodle
     */
    $pdo = getDB();
    
    // Get student details before permanent deletion
    $stmt = $pdo->prepare("SELECT * FROM students WHERE id = ?");
    $stmt->execute([$id]);
    $student = $stmt->fetch();
    
    if (!$student) {
        error_log("Permanent delete failed: Student with ID {$id} not found");
        return false;
    }
    
    // Initialize results
    $unenrollResult = false;
    $suspendResult = false;
    
    // Step 1: Try to unenroll student from all courses in Moodle (non-blocking)
    try {
        $unenrollResult = unenrollStudentFromMoodle($student);
        error_log("Moodle unenroll attempt for student {$id}: " . ($unenrollResult ? 'SUCCESS' : 'FAILED'));
    } catch (Exception $e) {
        error_log("Moodle unenroll error for student {$id}: " . $e->getMessage());
    }
    
    // Step 2: Try to suspend student account in Moodle (non-blocking)
    try {
        $suspendResult = suspendStudentInMoodle($student);
        error_log("Moodle suspend attempt for student {$id}: " . ($suspendResult ? 'SUCCESS' : 'FAILED'));
    } catch (Exception $e) {
        error_log("Moodle suspend error for student {$id}: " . $e->getMessage());
    }
    
    // Step 3: Permanently delete from local database (this is the main operation)
    try {
        $stmt = $pdo->prepare("DELETE FROM students WHERE id = ?");
        $stmt->execute([$id]);
        $deleteResult = $stmt->rowCount() > 0;
        
        if ($deleteResult) {
            error_log("Student {$id} successfully deleted from local database");
        } else {
            error_log("Failed to delete student {$id} from local database - no rows affected");
        }
    } catch (Exception $e) {
        error_log("Database delete error for student {$id}: " . $e->getMessage());
        return false;
    }
    
    // Step 4: Update Excel file (non-blocking)
    if ($deleteResult) {
        try {
            updateStudentsExcel();
            error_log("Excel file updated after deleting student {$id}");
        } catch (Exception $e) {
            error_log("Excel update error after deleting student {$id}: " . $e->getMessage());
        }
    }
    
    // Log the operation
    error_log("Student PERMANENT deletion completed - ID: {$id}, Unenroll: " . ($unenrollResult ? 'SUCCESS' : 'FAILED') . 
              ", Suspend: " . ($suspendResult ? 'SUCCESS' : 'FAILED') . 
              ", Delete: " . ($deleteResult ? 'SUCCESS' : 'FAILED'));
    
    // Return true if local deletion succeeded, regardless of Moodle status
    return $deleteResult;
}

function restoreStudent($id, $subjectIds = null) {
    /**
     * Restore a suspended student (within grace period)
     * @param int $id Student ID
     * @param array $subjectIds Optional array of subject IDs to enroll in. If null, uses original subjects
     * @return array Result with success status and details
     */
    $pdo = getDB();
    
    // Get student details
    $stmt = $pdo->prepare("SELECT * FROM students WHERE id = ? AND status = 'suspended'");
    $stmt->execute([$id]);
    $student = $stmt->fetch();
    
    if (!$student) {
        return [
            'success' => false,
            'message' => 'Student not found or not suspended',
            'details' => []
        ];
    }
    
    $result = [
        'success' => true,
        'message' => 'Student restored successfully',
        'details' => [
            'moodle_restore' => false,
            'local_restore' => false,
            'subjects_enrolled' => 0,
            'subjects_failed' => 0,
            'enrollment_details' => []
        ]
    ];
    
    // Step 1: Restore student account in Moodle
    $moodleRestoreResult = restoreStudentInMoodle($student);
    $result['details']['moodle_restore'] = $moodleRestoreResult;
    
    // Step 2: Restore in local database
    $stmt = $pdo->prepare("
        UPDATE students 
        SET status = 'active'
        WHERE id = ?
    ");
    $stmt->execute([$id]);
    $restoreLocalResult = $stmt->rowCount() > 0;
    $result['details']['local_restore'] = $restoreLocalResult;
    
    if (!$restoreLocalResult) {
        $result['success'] = false;
        $result['message'] = 'Failed to restore student in local database';
        return $result;
    }
    
    // Step 3: Handle subject enrollment
    if ($subjectIds === null) {
        // Use original subjects if no specific subjects provided
        $originalSubjects = getSubjectsForStudent($id);
        $subjectIds = array_column($originalSubjects, 'id');
    }
    
    if (!empty($subjectIds)) {
        // Enroll student in selected subjects
        $enrollmentResult = enrollStudentInSubjects($id, $subjectIds);
        $result['details']['subjects_enrolled'] = $enrollmentResult['enrolled'];
        $result['details']['subjects_failed'] = $enrollmentResult['failed'];
        $result['details']['enrollment_details'] = $enrollmentResult['details'];
    }
    
    // Step 4: Update Excel file
    updateStudentsExcel();
    
    // Log the operation
    error_log("Student restoration completed - ID: {$id}, Moodle Restore: " . ($moodleRestoreResult ? 'SUCCESS' : 'FAILED') . 
              ", Local Restore: " . ($restoreLocalResult ? 'SUCCESS' : 'FAILED') . 
              ", Subjects Enrolled: " . $result['details']['subjects_enrolled'] . 
              ", Subjects Failed: " . $result['details']['subjects_failed']);
    
    return $result;
}

function enrollStudentInSubjects($studentId, $subjectIds) {
    /**
     * Enroll student in specific subjects in Moodle
     * @param int $studentId Student ID
     * @param array $subjectIds Array of subject IDs
     * @return array Result with enrollment details
     */
    $result = [
        'enrolled' => 0,
        'failed' => 0,
        'details' => []
    ];
    
    try {
        // Get student details
        $student = getStudentById($studentId);
        if (!$student) {
            $result['details'][] = "Student not found";
            return $result;
        }
        
        $moodleUsername = formatUsernameForMoodle($student['username']);
        $moodleUserId = getMoodleUserId($moodleUsername);
        
        if (!$moodleUserId) {
            $result['details'][] = "Student not found in Moodle";
            return $result;
        }
        
        // Get subject to course mapping
        $subjectToCourseMapping = getSubjectToCourseMapping();
        
        foreach ($subjectIds as $subjectId) {
            if (!isset($subjectToCourseMapping[$subjectId])) {
                $result['failed']++;
                $result['details'][] = "No course mapping found for subject ID: {$subjectId}";
                continue;
            }
            
            $courseShortname = $subjectToCourseMapping[$subjectId];
            $courseId = getMoodleCourseId($courseShortname);
            
            if (!$courseId) {
                $result['failed']++;
                $result['details'][] = "Course not found in Moodle: {$courseShortname}";
                continue;
            }
            
            // Enroll student in course
            $enrollmentResult = enrollUserInMoodleCourse($moodleUserId, $courseId);
            
            if ($enrollmentResult) {
                $result['enrolled']++;
                $result['details'][] = "Successfully enrolled in course: {$courseShortname}";
            } else {
                $result['failed']++;
                $result['details'][] = "Failed to enroll in course: {$courseShortname}";
            }
        }
        
    } catch (Exception $e) {
        error_log("Exception enrolling student in subjects: " . $e->getMessage());
        $result['details'][] = "Enrollment error: " . $e->getMessage();
    }
    
    return $result;
}

function getSubjectToCourseMapping() {
    /**
     * Get the mapping of subject IDs to Moodle course shortnames
     */
    return [
        1 => 'COS100',   // Consumer Science
        2 => 'MTH100',   // Mathematics
        3 => 'MTH100',   // Mathematics (mapped to same as subject 2)
        4 => 'ENG100',   // English (mapped to same as subject 5)
        5 => 'ENG100',   // English
        6 => 'SCI100',   // Science (mapped to same as subject 18)
        7 => 'SIS100',   // Siswati
        8 => 'COS100',   // Consumer Science (mapped to same as subject 1)
        9 => 'REL100',   // Religious Education (mapped to same as subject 21)
        10 => 'SIS100',  // Siswati (mapped to same as subject 7)
        11 => 'ICT100',  // ICT (mapped to same as subject 24)
        18 => 'SCI100',  // Science
        21 => 'REL100',  // Religious Education
        24 => 'ICT100',  // ICT
        27 => 'EXA100',  // Expressive Arts
        31 => 'SOS100',  // Social Studies
        32 => 'AGR100',  // Agriculture
        33 => 'HPE100',  // HPE
        99 => 'COS100',  // Map to existing Consumer Science course
        100 => 'MTH100', // Map to existing Mathematics course
        101 => 'SCI100', // Map to existing Science course
    ];
}

function getMoodleCourseId($shortname) {
    /**
     * Get Moodle course ID by shortname
     */
    try {
        $result = callMoodleAPI('core_course_get_courses', [
            'options' => [
                'ids' => []
            ]
        ]);
        
        if ($result && !isset($result['exception'])) {
            foreach ($result['courses'] as $course) {
                if ($course['shortname'] === $shortname) {
                    return $course['id'];
                }
            }
        }
        
        return null;
    } catch (Exception $e) {
        error_log("Exception getting Moodle course ID: " . $e->getMessage());
        return null;
    }
}

function enrollUserInMoodleCourse($userId, $courseId) {
    /**
     * Enroll user in a specific Moodle course
     */
    try {
        $result = callMoodleAPI('enrol_manual_enrol_users', [
            'enrolments' => [[
                'userid' => $userId,
                'courseid' => $courseId,
                'roleid' => 5 // Student role
            ]]
        ]);
        
        return $result && !isset($result['exception']);
    } catch (Exception $e) {
        error_log("Exception enrolling user in course: " . $e->getMessage());
        return false;
    }
}

function processExpiredSuspendedStudents() {
    /**
     * Process suspended students
     * Note: Automatic deletion disabled - admin must manually delete students
     */
    $pdo = getDB();
    
    // Get suspended students (no automatic deletion)
    $stmt = $pdo->prepare("
        SELECT * FROM students 
        WHERE status = 'suspended'
    ");
    $stmt->execute();
    $suspendedStudents = $stmt->fetchAll();
    
    // Just log the count, no automatic deletion
    $count = count($suspendedStudents);
    error_log("Suspended students check: {$count} students currently suspended (manual deletion required)");
    
    return [
        'processed' => 0,
        'errors' => 0,
        'total' => $count,
        'message' => 'Automatic deletion disabled. Admin must manually delete students.'
    ];
}

function getSuspendedStudents() {
    /**
     * Get all suspended students
     */
    $pdo = getDB();
    $stmt = $pdo->prepare("
        SELECT s.*, 
               g.name as grade_name
        FROM students s
        LEFT JOIN grades g ON s.grade_id = g.id
        WHERE s.status = 'suspended'
        ORDER BY s.updated_at DESC
    ");
    $stmt->execute();
    return $stmt->fetchAll();
}

/**
 * Get suspended students with filters and pagination
 */
function getSuspendedStudentsFiltered($filters = [], $page = 1, $pageSize = 20, &$total = 0) {
    $pdo = getDB();
    $where = ["(s.status = 'suspended' OR s.payment_status = 'suspended')"];
    $params = [];
    
    if (!empty($filters['q'])) {
        $where[] = '(s.first_name LIKE ? OR s.last_name LIKE ? OR s.full_name LIKE ? OR s.email LIKE ? OR s.username LIKE ?)';
        $q = '%' . $filters['q'] . '%';
        array_push($params, $q, $q, $q, $q, $q);
    }
    if (!empty($filters['grade_id'])) {
        $where[] = 's.grade_id = ?';
        $params[] = (int)$filters['grade_id'];
    }
    if (!empty($filters['from'])) {
        $where[] = 's.created_at >= ?';
        $params[] = $filters['from'];
    }
    if (!empty($filters['to'])) {
        $where[] = 's.created_at <= ?';
        $params[] = $filters['to'];
    }
    
    $whereSql = implode(' AND ', $where);
    
    // Total count
    $countStmt = $pdo->prepare("SELECT COUNT(*) AS cnt FROM students s WHERE $whereSql");
    $countStmt->execute($params);
    $total = (int)$countStmt->fetch()['cnt'];
    
    $offset = max(0, ($page - 1) * $pageSize);
    
    $sql = "
        SELECT s.*, g.name as grade_name, 
               0 as days_remaining
        FROM students s
        LEFT JOIN grades g ON s.grade_id = g.id
        WHERE $whereSql
        ORDER BY s.created_at DESC
        LIMIT $pageSize OFFSET $offset
    ";
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    return $stmt->fetchAll();
}

/**
 * Get a student's original subject IDs
 */
function getOriginalSubjectIds($studentId) {
    $pdo = getDB();
    $stmt = $pdo->prepare('SELECT subject_id FROM student_subject WHERE student_id = ?');
    $stmt->execute([(int)$studentId]);
    return array_map('intval', array_column($stmt->fetchAll(), 'subject_id'));
}

/**
 * Get Moodle status info for a student (found and enrollment count)
 */
function getMoodleStatusForStudent($student) {
    try {
        $username = formatUsernameForMoodle($student['username'] ?? '');
        if (!$username) {
            return ['found' => false, 'enrollments' => 0];
        }
        $userId = getMoodleUserId($username);
        if (!$userId) {
            return ['found' => false, 'enrollments' => 0];
        }
        $enrollments = getMoodleUserEnrollments($userId);
        return [
            'found' => true,
            'enrollments' => is_array($enrollments) ? count($enrollments) : 0
        ];
    } catch (Exception $e) {
        error_log('Moodle status check failed: ' . $e->getMessage());
        return ['found' => false, 'enrollments' => 0];
    }
}

function getStudentsForCleanup() {
    /**
     * Get suspended students (automatic cleanup disabled)
     */
    $pdo = getDB();
    
    $stmt = $pdo->prepare("
        SELECT * 
        FROM students 
        WHERE status = 'suspended'
        ORDER BY updated_at DESC
    ");
    $stmt->execute();
    return $stmt->fetchAll();
}

// Email verification functions
function generateEmailVerificationToken() {
    return bin2hex(random_bytes(32));
}

function createEmailVerificationToken($studentId) {
    $pdo = getDB();
    $token = generateEmailVerificationToken();
    $expires = date('Y-m-d H:i:s', strtotime('+24 hours')); // Token expires in 24 hours
    
    $stmt = $pdo->prepare("
        UPDATE students 
        SET email_verification_token = ?, email_verification_expires = ? 
        WHERE id = ?
    ");
    $stmt->execute([$token, $expires, $studentId]);
    
    return $token;
}

function verifyEmailToken($token) {
    $pdo = getDB();
    $stmt = $pdo->prepare("
        SELECT id, email, first_name 
        FROM students 
        WHERE email_verification_token = ? 
        AND email_verification_expires > NOW() 
        AND email_verified = FALSE
    ");
    $stmt->execute([$token]);
    return $stmt->fetch();
}

function markEmailAsVerified($studentId) {
    $pdo = getDB();
    
    // Get student details before updating
    $stmt = $pdo->prepare("SELECT * FROM students WHERE id = ?");
    $stmt->execute([$studentId]);
    $student = $stmt->fetch();
    
    if (!$student) {
        return false;
    }
    
    // Mark email as verified
    $stmt = $pdo->prepare("
        UPDATE students 
        SET email_verified = TRUE, 
            email_verification_token = NULL, 
            email_verification_expires = NULL 
        WHERE id = ?
    ");
    $stmt->execute([$studentId]);
    
    if ($stmt->rowCount() > 0) {
        // Send login credentials after successful verification
        // Generate secure password that matches what the bot will use in Moodle
        $securePassword = generateSecurePasswordForMoodle($student['last_name']);
        $passwordHash = password_hash($securePassword, PASSWORD_DEFAULT);
        
        // Update the password and store the plain password temporarily for display
        $stmt = $pdo->prepare("UPDATE students SET password_hash = ?, temp_password = ? WHERE id = ?");
        $stmt->execute([$passwordHash, $securePassword, $studentId]);
        
        // Format username to match what the bot will use in Moodle
        $moodleUsername = formatUsernameForMoodle($student['username']);
        
        // Don't send credentials yet - wait for payment completion
        // sendStudentCredentials($student['email'], $student['first_name'], $moodleUsername, $securePassword);
        
        // Don't trigger bot yet - wait for payment completion
        // triggerMoodleSyncBot();
        
        return true;
    }
    
    return false;
}

function clearTempPassword($studentId) {
    /**
     * Clear the temporary password after it's been displayed to the user
     */
    $pdo = getDB();
    $stmt = $pdo->prepare("UPDATE students SET temp_password = NULL WHERE id = ?");
    $stmt->execute([$studentId]);
    return $stmt->rowCount() > 0;
}

// ===== Payments (MTN MoMo) helpers =====
function uuidv4() {
    $data = random_bytes(16);
    $data[6] = chr((ord($data[6]) & 0x0f) | 0x40);
    $data[8] = chr((ord($data[8]) & 0x3f) | 0x80);
    return vsprintf('%s%s-%s-%s-%s-%s%s%s', str_split(bin2hex($data), 4));
}

// Alias for compatibility
function generateUuidV4() {
    return uuidv4();
}

// Get all subjects
function getSubjects() {
    try {
        $pdo = getDB();
        $stmt = $pdo->query("SELECT * FROM subjects ORDER BY name ASC");
        return $stmt->fetchAll();
    } catch (Exception $e) {
        error_log("Error getting subjects: " . $e->getMessage());
        return [];
    }
}

// Get grade name by ID
function getGradeName($grade_id) {
    try {
        $pdo = getDB();
        $stmt = $pdo->prepare("SELECT name FROM grades WHERE id = ?");
        $stmt->execute([$grade_id]);
        $grade = $stmt->fetch();
        return $grade ? $grade['name'] : 'Unknown Grade';
    } catch (Exception $e) {
        error_log("Error getting grade name: " . $e->getMessage());
        return 'Unknown Grade';
    }
}

function ensurePaymentsTables() {
    $pdo = getDB();
    $pdo->exec("CREATE TABLE IF NOT EXISTS payments (
        id INT AUTO_INCREMENT PRIMARY KEY,
        student_id INT NOT NULL,
        reference_id VARCHAR(64) NOT NULL UNIQUE,
        external_id VARCHAR(64) NULL,
        amount DECIMAL(10,2) NOT NULL,
        currency VARCHAR(8) NOT NULL,
        msisdn VARCHAR(32) NOT NULL,
        status VARCHAR(32) NOT NULL DEFAULT 'PENDING',
        raw_response TEXT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;");
    $pdo->exec("CREATE TABLE IF NOT EXISTS payment_items (
        id INT AUTO_INCREMENT PRIMARY KEY,
        payment_id INT NOT NULL,
        subject_id INT NOT NULL,
        price DECIMAL(10,2) NOT NULL,
        FOREIGN KEY (payment_id) REFERENCES payments(id) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;");
}

function momo_build_callback_url($referenceId) {
    $base = rtrim(PUBLIC_BASE_URL, '/');
    return $base . MOMO_CALLBACK_PATH . '?ref=' . urlencode($referenceId);
}

function momo_get_access_token(&$error = null) {
    $url = MOMO_COLLECTION_BASE_URL . '/token/';
    $ch = curl_init($url);
    $auth = base64_encode(MOMO_API_USER_ID . ':' . MOMO_API_KEY);
    curl_setopt_array($ch, [
        CURLOPT_POST => true,
        CURLOPT_HTTPHEADER => [
            'Authorization: Basic ' . $auth,
            'Ocp-Apim-Subscription-Key: ' . MOMO_COLLECTION_SUBSCRIPTION_KEY,
        ],
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_TIMEOUT => 30,
    ]);
    $res = curl_exec($ch);
    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    if ($code !== 200) {
        $error = 'Token HTTP ' . $code . ' ' . $res;
        return null;
    }
    $json = json_decode($res, true);
    return $json['access_token'] ?? null;
}

function momo_request_to_pay($token, $referenceId, $amount, $currency, $msisdn, $externalId, $payerMessage = '', $payeeNote = '') {
    $url = MOMO_COLLECTION_BASE_URL . '/v1_0/requesttopay';
    $payload = [
        'amount' => number_format((float)$amount, 2, '.', ''),
        'currency' => $currency,
        'externalId' => (string)$externalId,
        'payer' => [
            'partyIdType' => 'MSISDN',
            'partyId' => $msisdn
        ],
        'payerMessage' => $payerMessage,
        'payeeNote' => $payeeNote
    ];
    $headers = [
        'Authorization: Bearer ' . $token,
        'X-Reference-Id: ' . $referenceId,
        'X-Target-Environment: ' . MOMO_ENV,
        'Ocp-Apim-Subscription-Key: ' . MOMO_COLLECTION_SUBSCRIPTION_KEY,
        'Content-Type: application/json',
        'X-Callback-Url: ' . momo_build_callback_url($referenceId)
    ];
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_POST => true,
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_POSTFIELDS => json_encode($payload),
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_TIMEOUT => 30,
    ]);
    $res = curl_exec($ch);
    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    return [$code, $res];
}

function momo_get_request_status($token, $referenceId) {
    $url = MOMO_COLLECTION_BASE_URL . '/v1_0/requesttopay/' . urlencode($referenceId);
    $headers = [
        'Authorization: Bearer ' . $token,
        'X-Target-Environment: ' . MOMO_ENV,
        'Ocp-Apim-Subscription-Key: ' . MOMO_COLLECTION_SUBSCRIPTION_KEY,
        'Accept: application/json'
    ];
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_HTTPGET => true,
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_TIMEOUT => 30,
    ]);
    $res = curl_exec($ch);
    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    return [$code, $res];
}

function sendPaymentEmail($email, $firstName, $status, $amount, $currency, $subjects, $referenceId) {
    $subject = ($status === 'SUCCESSFUL' ? 'Payment Successful' : 'Payment Update') . ' - ' . APP_NAME;
    $subjectList = implode(', ', array_map(function($s){ return is_array($s)?($s['name']??$s['id']):$s; }, $subjects));
    $body = "Hello {$firstName},<br><br>" .
            ($status === 'SUCCESSFUL'
                ? "Your payment was successful."
                : "Your payment status: {$status}.") .
            "<br>Amount: {$currency} " . number_format((float)$amount,2) .
            "<br>Subjects: {$subjectList}" .
            "<br>Reference: {$referenceId}<br><br>Regards,<br>" . APP_NAME;
    try {
        $mail = new PHPMailer(true);
        $mail->isSMTP();
        $mail->Host = SMTP_HOST;
        $mail->SMTPAuth = true;
        $mail->Username = SMTP_USERNAME;
        $mail->Password = SMTP_PASSWORD;
        $mail->SMTPSecure = SMTP_ENCRYPTION;
        $mail->Port = SMTP_PORT;
        $mail->setFrom(SMTP_FROM_EMAIL, SMTP_FROM_NAME);
        $mail->addAddress($email, $firstName);
        $mail->isHTML(true);
        $mail->Subject = $subject;
        $mail->Body = $body;
        $mail->send();
    } catch (Exception $e) {
        error_log('Payment email error: ' . $e->getMessage());
    }
}

function sendPaymentSuccessEmail($email, $firstName, $amount, $currency, $subjects, $referenceId) {
    $subject = 'Payment Successful - ' . APP_NAME;
    $subjectList = implode(', ', array_map(function($s){ return is_array($s)?($s['name']??$s['id']):$s; }, $subjects));
    $body = "Hello {$firstName},<br><br>" .
            "Your payment of {$currency} {$amount} (Reference: {$referenceId}) was successful!<br><br>" .
            "You have been enrolled in the following subjects: {$subjectList}<br><br>" .
            "You will receive your login credentials shortly.<br><br>" .
            "Best regards,<br>" . APP_NAME;
    
    try {
        $mail = new PHPMailer(true);
        $mail->isSMTP();
        $mail->Host = SMTP_HOST;
        $mail->SMTPAuth = true;
        $mail->Username = SMTP_USERNAME;
        $mail->Password = SMTP_PASSWORD;
        $mail->SMTPSecure = SMTP_ENCRYPTION;
        $mail->Port = SMTP_PORT;
        $mail->setFrom(SMTP_FROM_EMAIL, SMTP_FROM_NAME);
        $mail->addAddress($email, $firstName);
        $mail->isHTML(true);
        $mail->Subject = $subject;
        $mail->Body = $body;
        $mail->send();
        return true;
    } catch (Exception $e) {
        error_log('Payment success email error: ' . $e->getMessage());
        return false;
    }
}

function sendPaymentFailureEmail($email, $firstName, $amount, $currency, $referenceId) {
    $subject = 'Payment Failed - ' . APP_NAME;
    $body = "Hello {$firstName},<br><br>" .
            "Your payment of {$currency} {$amount} (Reference: {$referenceId}) was not successful.<br><br>" .
            "Please try again or contact support if you continue to experience issues.<br><br>" .
            "Best regards,<br>" . APP_NAME;
    
    try {
        $mail = new PHPMailer(true);
        $mail->isSMTP();
        $mail->Host = SMTP_HOST;
        $mail->SMTPAuth = true;
        $mail->Username = SMTP_USERNAME;
        $mail->Password = SMTP_PASSWORD;
        $mail->SMTPSecure = SMTP_ENCRYPTION;
        $mail->Port = SMTP_PORT;
        $mail->setFrom(SMTP_FROM_EMAIL, SMTP_FROM_NAME);
        $mail->addAddress($email, $firstName);
        $mail->isHTML(true);
        $mail->Subject = $subject;
        $mail->Body = $body;
        $mail->send();
        return true;
    } catch (Exception $e) {
        error_log('Payment failure email error: ' . $e->getMessage());
        return false;
    }
}

// Admin payments reporting helpers
function getPaymentsReport($filters = [], $page = 1, $pageSize = 25, &$total = 0) {
    $pdo = getDB();
    $where = [];
    $params = [];
    if (!empty($filters['status'])) { $where[] = 'p.status = ?'; $params[] = $filters['status']; }
    if (!empty($filters['from'])) { $where[] = 'p.created_at >= ?'; $params[] = $filters['from']; }
    if (!empty($filters['to'])) { $where[] = 'p.created_at <= ?'; $params[] = $filters['to']; }
    if (!empty($filters['grade_id'])) { $where[] = 's.grade_id = ?'; $params[] = (int)$filters['grade_id']; }
    if (!empty($filters['subject_id'])) { $where[] = 'EXISTS (SELECT 1 FROM payment_items pi2 WHERE pi2.payment_id=p.id AND pi2.subject_id=?)'; $params[] = (int)$filters['subject_id']; }
    $whereSql = $where ? ('WHERE ' . implode(' AND ', $where)) : '';
    $count = $pdo->prepare("SELECT COUNT(*) AS c FROM payments p JOIN students s ON s.id=p.student_id $whereSql");
    $count->execute($params); $total = (int)$count->fetch()['c'];
    $offset = max(0, ($page-1)*$pageSize);
    $sql = "SELECT p.*, 
            COALESCE(s.first_name, SUBSTRING_INDEX(s.full_name, ' ', 1)) as first_name,
            COALESCE(s.last_name, SUBSTRING_INDEX(s.full_name, ' ', -1)) as last_name,
            COALESCE(s.full_name, CONCAT(s.first_name, ' ', s.last_name)) as full_name,
            s.email, s.grade_id, g.name AS grade_name
            FROM payments p
            JOIN students s ON s.id=p.student_id
            LEFT JOIN grades g ON g.id=s.grade_id
            $whereSql
            ORDER BY COALESCE(p.created_at, p.payment_date, p.id) DESC
            LIMIT $pageSize OFFSET $offset";
    $stmt = $pdo->prepare($sql); $stmt->execute($params);
    return $stmt->fetchAll();
}

// ===== MTN MoMo API user provisioning (sandbox) =====
function momo_create_api_user($referenceId, $callbackHost) {
    $url = rtrim(MOMO_COLLECTION_BASE_URL, '/') . '/v1_0/apiuser';
    $headers = [
        'X-Reference-Id: ' . $referenceId,
        'Ocp-Apim-Subscription-Key: ' . MOMO_COLLECTION_SUBSCRIPTION_KEY,
        'Content-Type: application/json'
    ];
    $payload = [ 'providerCallbackHost' => $callbackHost ];
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_POST => true,
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_POSTFIELDS => json_encode($payload),
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_HEADER => true,
        CURLOPT_NOBODY => true,
        CURLOPT_TIMEOUT => 30,
    ]);
    $res = curl_exec($ch);
    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    return [$code, $res];
}

function momo_create_api_key($apiUserId) {
    $url = rtrim(MOMO_COLLECTION_BASE_URL, '/') . '/v1_0/apiuser/' . urlencode($apiUserId) . '/apikey';
    $headers = [
        'Ocp-Apim-Subscription-Key: ' . MOMO_COLLECTION_SUBSCRIPTION_KEY,
        'Content-Type: application/json'
    ];
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_POST => true,
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_TIMEOUT => 30,
    ]);
    $res = curl_exec($ch);
    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    $json = json_decode($res, true);
    return [$code, $json['apiKey'] ?? null, $res];
}

function getPaymentSubjects($paymentId) {
    $pdo = getDB();
    $stmt = $pdo->prepare('SELECT pi.subject_id, sub.name FROM payment_items pi JOIN subjects sub ON sub.id=pi.subject_id WHERE pi.payment_id=?');
    $stmt->execute([(int)$paymentId]);
    return $stmt->fetchAll();
}

// Create payment record
function createPaymentRecord($student_id, $amount, $reference_id, $subject_ids) {
    $pdo = getDB();
    try {
        // Only start transaction if not already in one
        $inTransaction = $pdo->inTransaction();
        if (!$inTransaction) {
            $pdo->beginTransaction();
        }
        
        $stmt = $pdo->prepare('INSERT INTO payments (student_id, amount, reference_id, status, created_at) VALUES (?, ?, ?, "PENDING", NOW())');
        $stmt->execute([$student_id, $amount, $reference_id]);
        $payment_id = $pdo->lastInsertId();
        
        // Add payment items for each subject
        $stmt = $pdo->prepare('INSERT INTO payment_items (payment_id, subject_id) VALUES (?, ?)');
        foreach ($subject_ids as $subject_id) {
            $stmt->execute([$payment_id, $subject_id]);
        }
        
        // Only commit if we started the transaction
        if (!$inTransaction && $pdo->inTransaction()) {
            $pdo->commit();
        }
        return $payment_id;
    } catch (Exception $e) {
        // Only rollback if we started the transaction
        if (!$inTransaction && $pdo->inTransaction()) {
            $pdo->rollBack();
        }
        return false;
    }
}

// Get payment by ID
function getPaymentById($payment_id) {
    $pdo = getDB();
    $stmt = $pdo->prepare('SELECT * FROM payments WHERE id = ? LIMIT 1');
    $stmt->execute([$payment_id]);
    return $stmt->fetch();
}

// Update payment status
function updatePaymentStatus($payment_id, $status, $reference_id = null) {
    $pdo = getDB();
    $sql = 'UPDATE payments SET status = ?';
    $params = [$status];
    
    if ($reference_id) {
        $sql .= ', reference_id = ?';
        $params[] = $reference_id;
    }
    
    $sql .= ' WHERE id = ?';
    $params[] = $payment_id;
    
    $stmt = $pdo->prepare($sql);
    return $stmt->execute($params);
}

// Get subjects by IDs
function getSubjectsByIds($subject_ids) {
    if (empty($subject_ids)) return [];
    
    $pdo = getDB();
    $placeholders = str_repeat('?,', count($subject_ids) - 1) . '?';
    $stmt = $pdo->prepare("SELECT * FROM subjects WHERE id IN ($placeholders)");
    $stmt->execute($subject_ids);
    return $stmt->fetchAll();
}

function adminMarkPaymentPaid($paymentId) {
    $pdo = getDB();
    $stmt = $pdo->prepare('SELECT * FROM payments WHERE id=? LIMIT 1');
    $stmt->execute([(int)$paymentId]);
    $payment = $stmt->fetch();
    if (!$payment) return false;
    if (strtoupper($payment['status']) !== 'SUCCESSFUL') {
        $upd = $pdo->prepare('UPDATE payments SET status="SUCCESSFUL" WHERE id=?');
        $upd->execute([(int)$paymentId]);
    }
    // Enroll subjects
    $items = $pdo->prepare('SELECT subject_id FROM payment_items WHERE payment_id=?');
    $items->execute([(int)$paymentId]);
    $subjectIds = array_map('intval', array_column($items->fetchAll(), 'subject_id'));
    if (!empty($subjectIds)) {
        enrollStudentInSubjects((int)$payment['student_id'], $subjectIds);
    }
    return true;
}

function getStudentPaymentStatus($studentId) {
    $pdo = getDB();
    
    // Check if student has any successful payments
    $stmt = $pdo->prepare('SELECT COUNT(*) as count FROM payments WHERE student_id = ? AND status = "SUCCESSFUL"');
    $stmt->execute([(int)$studentId]);
    $hasPaid = $stmt->fetch()['count'] > 0;
    
    // Check if student has any pending payments
    $stmt = $pdo->prepare('SELECT COUNT(*) as count FROM payments WHERE student_id = ? AND status IN ("PENDING", "FAILED")');
    $stmt->execute([(int)$studentId]);
    $hasPending = $stmt->fetch()['count'] > 0;
    
    return [
        'has_paid' => $hasPaid,
        'has_pending' => $hasPending
    ];
}

function sendEmailVerification($email, $firstName, $token) {
    $mail = new PHPMailer(true);
    
    try {
        // Server settings
        $mail->isSMTP();
        $mail->Host = SMTP_HOST;
        $mail->SMTPAuth = true;
        $mail->Username = SMTP_USERNAME;
        $mail->Password = SMTP_PASSWORD;
        $mail->SMTPSecure = SMTP_ENCRYPTION;
        $mail->Port = SMTP_PORT;
        
        // Additional SMTP settings for better deliverability
        $mail->SMTPOptions = array(
            'ssl' => array(
                'verify_peer' => false,
                'verify_peer_name' => false,
                'allow_self_signed' => true
            )
        );
        
        // Timeout settings
        $mail->Timeout = 30;
        $mail->SMTPKeepAlive = true;
        
        // Recipients
        $mail->setFrom(SMTP_FROM_EMAIL, SMTP_FROM_NAME);
        $mail->addAddress($email, $firstName);
        
        // Add reply-to
        $mail->addReplyTo(SMTP_FROM_EMAIL, SMTP_FROM_NAME);
        
        // Create verification URL - always point to root level verify-email.php
        $baseUrl = (isset($_SERVER['HTTPS']) ? 'https' : 'http') . '://' . $_SERVER['HTTP_HOST'];
        $scriptPath = $_SERVER['REQUEST_URI'];
        
        // Remove admin/ from path if present to get root path
        $rootPath = str_replace('/admin', '', dirname($scriptPath));
        $rootPath = rtrim($rootPath, '/');
        
        $verificationUrl = $baseUrl . $rootPath . '/verify-email.php?token=' . urlencode($token);
        
        // Content
        $mail->isHTML(true);
        $mail->Subject = 'Verify Your Email - ' . APP_NAME;
        
        // Improved email template with better anti-spam measures
        $mail->Body = "
            <!DOCTYPE html>
            <html>
            <head>
                <meta charset='UTF-8'>
                <meta name='viewport' content='width=device-width, initial-scale=1.0'>
                <title>Email Verification</title>
            </head>
            <body style='font-family: Arial, sans-serif; line-height: 1.6; color: #333; max-width: 600px; margin: 0 auto; padding: 20px;'>
                <div style='background: linear-gradient(135deg, #6f42c1, #20c997); padding: 30px; text-align: center; border-radius: 10px 10px 0 0;'>
                    <h1 style='color: white; margin: 0; font-size: 28px;'>" . APP_NAME . "</h1>
                    <p style='color: white; margin: 10px 0 0 0; font-size: 16px;'>Email Verification Required</p>
                </div>
                
                <div style='background: #f8f9fa; padding: 30px; border-radius: 0 0 10px 10px;'>
                    <h2 style='color: #6f42c1; margin-top: 0;'>Hello $firstName,</h2>
                    
                    <p>Thank you for registering with " . APP_NAME . "! To complete your registration and access your learning platform, please verify your email address.</p>
                    
                    <div style='text-align: center; margin: 30px 0;'>
                        <a href='$verificationUrl' style='background: linear-gradient(135deg, #6f42c1, #20c997); color: white; padding: 15px 30px; text-decoration: none; border-radius: 25px; display: inline-block; font-weight: bold; font-size: 16px;'>Verify Email Address</a>
                    </div>
                    
                    <p><strong>Alternative:</strong> Copy and paste this link into your browser:</p>
                    <p style='background: #e9ecef; padding: 10px; border-radius: 5px; word-break: break-all;'><a href='$verificationUrl'>$verificationUrl</a></p>
                    
                    <div style='background: #fff3cd; border: 1px solid #ffeaa7; padding: 15px; border-radius: 5px; margin: 20px 0;'>
                        <p style='margin: 0; color: #856404;'><strong>⏰ Important:</strong> This verification link will expire in 24 hours for security reasons.</p>
                    </div>
                    
                    <p>If you did not create an account with us, please ignore this email.</p>
                    
                    <hr style='border: none; border-top: 1px solid #dee2e6; margin: 30px 0;'>
                    
                    <p style='color: #6c757d; font-size: 14px; margin: 0;'>
                        Best regards,<br>
                        <strong>" . APP_NAME . " Team</strong><br>
                        <a href='" . MOODLE_URL . "' style='color: #6f42c1;'>" . MOODLE_URL . "</a>
                    </p>
                </div>
            </body>
            </html>
        ";
        
        // Plain text version for better deliverability
        $mail->AltBody = "
" . APP_NAME . " - Email Verification Required

Hello $firstName,

Thank you for registering with " . APP_NAME . "! To complete your registration, please verify your email address by visiting this link:

$verificationUrl

This link will expire in 24 hours for security reasons.

If you did not create an account with us, please ignore this email.

Best regards,
" . APP_NAME . " Team
" . MOODLE_URL . "
        ";
        
        // Send the email
        $result = $mail->send();
        
        if ($result) {
            error_log("Email verification sent successfully to: $email");
            return true;
        } else {
            error_log("Email verification failed to send to: $email");
            return false;
        }
        
    } catch (MailException $e) {
        error_log("Email verification sending failed for $email: " . $e->getMessage());
        error_log("SMTP Error: " . $mail->ErrorInfo);
        return false;
    }
}

// Email functions
function sendStudentCredentials($email, $firstName, $username, $password) {
    $mail = new PHPMailer(true);
    
    try {
        // Server settings
        $mail->isSMTP();
        $mail->Host = SMTP_HOST;
        $mail->SMTPAuth = true;
        $mail->Username = SMTP_USERNAME;
        $mail->Password = SMTP_PASSWORD;
        $mail->SMTPSecure = SMTP_ENCRYPTION;
        $mail->Port = SMTP_PORT;
        
        // Additional SMTP settings for better deliverability
        $mail->SMTPOptions = array(
            'ssl' => array(
                'verify_peer' => false,
                'verify_peer_name' => false,
                'allow_self_signed' => true
            )
        );
        
        // Timeout settings
        $mail->Timeout = 30;
        $mail->SMTPKeepAlive = true;
        
        // Recipients
        $mail->setFrom(SMTP_FROM_EMAIL, SMTP_FROM_NAME);
        $mail->addAddress($email, $firstName);
        
        // Add reply-to
        $mail->addReplyTo(SMTP_FROM_EMAIL, SMTP_FROM_NAME);
        
        // Content
        $mail->isHTML(true);
        $mail->Subject = 'Your ' . APP_NAME . ' Login Credentials - Welcome!';
        
        // Improved email template
        $mail->Body = "
            <!DOCTYPE html>
            <html>
            <head>
                <meta charset='UTF-8'>
                <meta name='viewport' content='width=device-width, initial-scale=1.0'>
                <title>Login Credentials</title>
            </head>
            <body style='font-family: Arial, sans-serif; line-height: 1.6; color: #333; max-width: 600px; margin: 0 auto; padding: 20px;'>
                <div style='background: linear-gradient(135deg, #6f42c1, #20c997); padding: 30px; text-align: center; border-radius: 10px 10px 0 0;'>
                    <h1 style='color: white; margin: 0; font-size: 28px;'>🎓 Welcome to " . APP_NAME . "!</h1>
                    <p style='color: white; margin: 10px 0 0 0; font-size: 16px;'>Your account is ready</p>
                </div>
                
                <div style='background: #f8f9fa; padding: 30px; border-radius: 0 0 10px 10px;'>
                    <h2 style='color: #6f42c1; margin-top: 0;'>Hello $firstName,</h2>
                    
                    <p>Congratulations! Your email has been verified and your account is now active. Here are your login credentials:</p>
                    
                    <div style='background: #ffffff; border: 2px solid #6f42c1; padding: 25px; border-radius: 10px; margin: 25px 0; text-align: center;'>
                        <h3 style='color: #6f42c1; margin-top: 0;'>🔑 Your Login Credentials</h3>
                        <div style='background: #f8f9fa; padding: 15px; border-radius: 8px; margin: 15px 0;'>
                            <p style='margin: 10px 0; font-size: 18px;'><strong>Username:</strong> <code style='background: #e9ecef; padding: 5px 10px; border-radius: 4px; font-size: 16px;'>$username</code></p>
                            <p style='margin: 10px 0; font-size: 18px;'><strong>Password:</strong> <code style='background: #e9ecef; padding: 5px 10px; border-radius: 4px; font-size: 16px;'>$password</code></p>
                        </div>
                        <p style='color: #dc3545; font-weight: bold; margin: 15px 0 0 0;'>⚠️ Keep these credentials safe and do not share them!</p>
                    </div>
                    
                    <div style='text-align: center; margin: 30px 0;'>
                        <a href='" . MOODLE_URL . "' style='background: linear-gradient(135deg, #6f42c1, #20c997); color: white; padding: 15px 30px; text-decoration: none; border-radius: 25px; display: inline-block; font-weight: bold; font-size: 16px; margin: 10px;'>🎓 Access Learning Platform</a>
                    </div>
                    
                    <div style='background: #25D366; color: white; padding: 20px; border-radius: 10px; margin: 25px 0; text-align: center;'>
                        <h3 style='color: white; margin-top: 0;'>📱 Join Our WhatsApp Group!</h3>
                        <p style='color: white; margin-bottom: 15px;'>Connect with other students, get course updates, and participate in discussions.</p>
                        <a href='" . WHATSAPP_GROUP_LINK . "' style='background-color: white; color: #25D366; padding: 12px 24px; text-decoration: none; border-radius: 25px; font-weight: bold; display: inline-block;'>Join WhatsApp Group</a>
                    </div>
                    
                    <div style='background: #d1ecf1; border: 1px solid #bee5eb; padding: 15px; border-radius: 5px; margin: 20px 0;'>
                        <p style='margin: 0; color: #0c5460;'><strong>💡 Need Help?</strong> If you have any questions or need assistance, please contact our support team.</p>
                    </div>
                    
                    <hr style='border: none; border-top: 1px solid #dee2e6; margin: 30px 0;'>
                    
                    <p style='color: #6c757d; font-size: 14px; margin: 0;'>
                        Best regards,<br>
                        <strong>" . APP_NAME . " Team</strong><br>
                        <a href='" . MOODLE_URL . "' style='color: #6f42c1;'>" . MOODLE_URL . "</a>
                    </p>
                </div>
            </body>
            </html>
        ";
        
        // Plain text version
        $mail->AltBody = "
" . APP_NAME . " - Your Login Credentials

Hello $firstName,

Congratulations! Your email has been verified and your account is now active.

Your Login Credentials:
Username: $username
Password: $password

⚠️ Keep these credentials safe and do not share them!

Access your learning platform: " . MOODLE_URL . "

Join our WhatsApp group: " . WHATSAPP_GROUP_LINK . "

Need Help? If you have any questions or need assistance, please contact our support team.

Best regards,
" . APP_NAME . " Team
" . MOODLE_URL . "
        ";
        
        // Send the email
        $result = $mail->send();
        
        if ($result) {
            error_log("Credentials email sent successfully to: $email");
            return true;
        } else {
            error_log("Credentials email failed to send to: $email");
            return false;
        }
        
    } catch (MailException $e) {
        error_log("Credentials email sending failed for $email: " . $e->getMessage());
        error_log("SMTP Error: " . $mail->ErrorInfo);
        return false;
    }
}

// Excel functions
function updateStudentsExcel() {
    $spreadsheetClass = 'PhpOffice\\PhpSpreadsheet\\Spreadsheet';
    $writerClass = 'PhpOffice\\PhpSpreadsheet\\Writer\\Xlsx';

    if (!class_exists($spreadsheetClass) || !class_exists($writerClass)) {
        error_log('updateStudentsExcel skipped: PhpSpreadsheet library not available');
        return false;
    }

    $students = getStudents();
    $filePath = __DIR__ . '/../exports/students.xlsx';
    
    // Ensure exports directory exists
    $exportDir = dirname($filePath);
    if (!is_dir($exportDir)) {
        mkdir($exportDir, 0755, true);
    }
    
    $spreadsheet = new $spreadsheetClass();
    $sheet = $spreadsheet->getActiveSheet();
    
    // Headers for Moodle import format: username, firstname, lastname, password, email, grade, subjects
    $headers = ['username', 'firstname', 'lastname', 'password', 'email', 'grade', 'subjects'];
    $sheet->fromArray($headers, null, 'A1');
    
    // Data
    $data = [];
    foreach ($students as $student) {
        // Get subjects for this student
        $subjects = getSubjectsForStudent($student['id']);
        $subjectNames = array_map(function($s) { return $s['name']; }, $subjects);
        $subjectsString = implode(', ', $subjectNames);
        
        // Use secure password that matches what users receive and bot uses
        $securePassword = generateSecurePasswordForMoodle($student['last_name']);
        
        $data[] = [
            formatUsernameForMoodle($student['username']), // Use formatted username for Moodle compatibility
            $student['first_name'],
            $student['last_name'],
            $securePassword,
            $student['email'],
            $student['grade_name'],
            $subjectsString
        ];
    }
    $sheet->fromArray($data, null, 'A2');
    
    // Save file
    $writer = new $writerClass($spreadsheet);
    $writer->save($filePath);
}

// Generate Moodle-compatible CSV export
function generateMoodleCSV() {
    $students = getStudents();
    $filePath = __DIR__ . '/../exports/moodle_import.csv';
    
    // Ensure exports directory exists
    $exportDir = dirname($filePath);
    if (!is_dir($exportDir)) {
        mkdir($exportDir, 0755, true);
    }
    
    $csv = fopen($filePath, 'w');
    
    // Headers for Moodle import
    fputcsv($csv, ['username', 'firstname', 'lastname', 'password', 'email', 'grade', 'subjects']);
    
    foreach ($students as $student) {
        // Get subjects for this student
        $subjects = getSubjectsForStudent($student['id']);
        $subjectNames = array_map(function($s) { return $s['name']; }, $subjects);
        $subjectsString = implode(', ', $subjectNames);
        
        // Use secure password that matches what users receive and bot uses
        $securePassword = generateSecurePasswordForMoodle($student['last_name']);
        
        fputcsv($csv, [
            formatUsernameForMoodle($student['username']), // Use formatted username for Moodle compatibility
            $student['first_name'],
            $student['last_name'],
            $securePassword,
            $student['email'],
            $student['grade_name'],
            $subjectsString
        ]);
    }
    
    fclose($csv);
    return $filePath;
}

// Admin authentication
function isAdminLoggedIn() {
    startSession();
    return isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in'] === true;
}

function requireAdminLogin() {
    if (!isAdminLoggedIn()) {
        // Use clean URL for redirect
        header('Location: /Multi-Tanent/admin/login');
        exit;
    }
}

function adminLogin($username, $password) {
    // First try database authentication
    $pdo = getDB();
    
    try {
        $stmt = $pdo->prepare("
            SELECT id, username, password_hash, email, academy_reference, is_active 
            FROM admins 
            WHERE username = ? AND is_active = 1
        ");
        $stmt->execute([$username]);
        $admin = $stmt->fetch();
        
        if ($admin && password_verify($password, $admin['password_hash'])) {
            // Database authentication successful
            startSession();
            $_SESSION['admin_logged_in'] = true;
            $_SESSION['admin_id'] = $admin['id'];
            $_SESSION['admin_username'] = $admin['username'];
            $_SESSION['admin_email'] = $admin['email'];
            
            // Check if super admin (academy_reference is NULL)
            if ($admin['academy_reference'] === null) {
                $_SESSION['is_super_admin'] = true;
            } else {
                $_SESSION['is_super_admin'] = false;
                $_SESSION['admin_academy_reference'] = $admin['academy_reference'];
            }
            
            return true;
        }
    } catch (PDOException $e) {
        error_log("Login error: " . $e->getMessage());
    }
    
    // Fallback to config-based authentication for backward compatibility
    if (defined('ADMIN_USERNAME') && defined('ADMIN_PASSWORD')) {
        if ($username === ADMIN_USERNAME && $password === ADMIN_PASSWORD) {
            startSession();
            $_SESSION['admin_logged_in'] = true;
            $_SESSION['is_super_admin'] = true; // Config admin is super admin
            return true;
        }
    }
    
    return false;
}

function adminLogout() {
    startSession();
    unset($_SESSION['admin_logged_in']);
}

// =====================================================
// STUDENT AUTHENTICATION FUNCTIONS
// =====================================================

function isStudentLoggedIn() {
    startSession();
    return isset($_SESSION['student_logged_in']) && $_SESSION['student_logged_in'] === true;
}

function requireStudentLogin() {
    if (!isStudentLoggedIn()) {
        header('Location: ../student_login.php');
        exit;
    }
}

function studentLogin($username, $password, $academy_reference = null) {
    $pdo = getDB();
    
    try {
        $sql = "
            SELECT id, username, password_hash, email, first_name, last_name, 
                   academy_reference, grade_id, is_active 
            FROM students 
            WHERE username = ? AND is_active = 1
        ";
        
        // If academy reference provided, add to query
        if ($academy_reference) {
            $sql .= " AND academy_reference = ?";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([$username, $academy_reference]);
        } else {
            $stmt = $pdo->prepare($sql);
            $stmt->execute([$username]);
        }
        
        $student = $stmt->fetch();
        
        if ($student && password_verify($password, $student['password_hash'])) {
            // Authentication successful
            startSession();
            $_SESSION['student_logged_in'] = true;
            $_SESSION['student_id'] = $student['id'];
            $_SESSION['student_username'] = $student['username'];
            $_SESSION['student_email'] = $student['email'];
            $_SESSION['student_name'] = $student['first_name'] . ' ' . $student['last_name'];
            $_SESSION['student_academy'] = $student['academy_reference'];
            $_SESSION['student_grade_id'] = $student['grade_id'];
            
            // Update last login
            $updateStmt = $pdo->prepare("
                UPDATE students 
                SET last_login = NOW() 
                WHERE id = ?
            ");
            $updateStmt->execute([$student['id']]);
            
            return true;
        }
    } catch (PDOException $e) {
        error_log("Student login error: " . $e->getMessage());
    }
    
    return false;
}

function studentLogout() {
    startSession();
    unset($_SESSION['student_logged_in']);
    unset($_SESSION['student_id']);
    unset($_SESSION['student_username']);
    unset($_SESSION['student_email']);
    unset($_SESSION['student_name']);
    unset($_SESSION['student_academy']);
    unset($_SESSION['student_grade_id']);
}

function getCurrentStudent() {
    startSession();
    if (isStudentLoggedIn()) {
        return [
            'id' => $_SESSION['student_id'] ?? null,
            'username' => $_SESSION['student_username'] ?? null,
            'email' => $_SESSION['student_email'] ?? null,
            'name' => $_SESSION['student_name'] ?? null,
            'academy_reference' => $_SESSION['student_academy'] ?? null,
            'grade_id' => $_SESSION['student_grade_id'] ?? null,
        ];
    }
    return null;
}

// Email validation functions
function validateEmailExists($email) {
    // First, check basic email format
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        return ['valid' => false, 'message' => 'Invalid email format'];
    }
    
    // Extract domain from email
    $domain = substr(strrchr($email, "@"), 1);
    
    // Check if domain has valid MX records
    if (!checkdnsrr($domain, "MX") && !checkdnsrr($domain, "A")) {
        return ['valid' => false, 'message' => 'Email domain does not exist'];
    }
    
    // Additional validation: Check for common disposable email domains
    $disposableDomains = [
        '10minutemail.com', 'tempmail.org', 'guerrillamail.com', 'mailinator.com',
        'throwaway.email', 'temp-mail.org', 'sharklasers.com', 'grr.la',
        'guerrillamailblock.com', 'pokemail.net', 'spam4.me', 'bccto.me',
        'chacuo.net', 'dispostable.com', 'mailnesia.com', 'maildrop.cc',
        'getairmail.com', 'mohmal.com', 'mytrashmail.com', 'trashmail.com',
        'yopmail.com', 'mailcatch.com', 'inboxalias.com', 'mailmetrash.com',
        'trashymail.com', 'trashmail.net', 'spamgourmet.com', 'spam.la',
        'binkmail.com', 'bobmail.info', 'chammy.info', 'devnullmail.com',
        'letthemeatspam.com', 'mailin8r.com', 'mailinator2.com', 'notmailinator.com',
        'reallymymail.com', 'reconmail.com', 'safetymail.info', 'sogetthis.com',
        'spamhereplease.com', 'superrito.com', 'thisisnotmyrealemail.com',
        'tradermail.info', 'veryrealemail.com', 'wegwerfadresse.de'
    ];
    
    if (in_array(strtolower($domain), $disposableDomains)) {
        return ['valid' => false, 'message' => 'Disposable email addresses are not allowed'];
    }
    
    // Check for common typos in popular domains
    $commonDomains = [
        'gmail.com', 'yahoo.com', 'hotmail.com', 'outlook.com', 'live.com',
        'icloud.com', 'aol.com', 'msn.com', 'comcast.net', 'verizon.net'
    ];
    
    $suggestions = [];
    foreach ($commonDomains as $commonDomain) {
        if (levenshtein($domain, $commonDomain) <= 2 && $domain !== $commonDomain) {
            $suggestions[] = str_replace($domain, $commonDomain, $email);
        }
    }
    
    if (!empty($suggestions)) {
        return [
            'valid' => false, 
            'message' => 'Did you mean: ' . implode(', ', array_slice($suggestions, 0, 3)) . '?'
        ];
    }
    
    return ['valid' => true, 'message' => 'Email appears to be valid'];
}

function checkEmailAvailability($email) {
    $pdo = getDB();
    $stmt = $pdo->prepare("SELECT id FROM students WHERE email = ?");
    $stmt->execute([$email]);
    return $stmt->fetch() === false; // Returns true if email is available
}

// Dashboard data
function getDashboardStats() {
    $pdo = getDB();
    $currentSchool = getCurrentSchool();
    
    $stats = [];
    
    // Total students - filtered by current school
    $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM students WHERE academy_reference = ?");
    $stmt->execute([$currentSchool]);
    $stats['total_students'] = $stmt->fetch()['count'];
    
    // Students by grade - filtered by current school
    $stmt = $pdo->prepare("
        SELECT g.name as grade, COUNT(DISTINCT s.id) as count 
        FROM grades g 
        LEFT JOIN students s ON g.id = s.grade_id AND s.academy_reference = ?
        GROUP BY g.id, g.name 
        ORDER BY g.name
    ");
    $stmt->execute([$currentSchool]);
    $stats['students_by_grade'] = $stmt->fetchAll();
    
    // Recent registrations (last 7 days) - filtered by current school
    $stmt = $pdo->prepare("
        SELECT COUNT(*) as count 
        FROM students 
        WHERE created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)
        AND academy_reference = ?
    ");
    $stmt->execute([$currentSchool]);
    $stats['recent_registrations'] = $stmt->fetch()['count'];
    
    // Email verification statistics - filtered by current school
    $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM students WHERE email_verified = TRUE AND academy_reference = ?");
    $stmt->execute([$currentSchool]);
    $stats['verified_emails'] = $stmt->fetch()['count'];
    
    $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM students WHERE email_verified = FALSE AND academy_reference = ?");
    $stmt->execute([$currentSchool]);
    $stats['pending_verifications'] = $stmt->fetch()['count'];
    
    return $stats;
}

// Subject management
function generateSubjectCode($subjectName, $gradeNumber) {
    // Get first 3 letters of subject name, uppercase
    $prefix = strtoupper(substr(preg_replace('/[^a-zA-Z]/', '', $subjectName), 0, 3));
    
    // Handle special cases for subjects with short names
    $specialCases = [
        'HPE' => 'HPE',
        'ICT' => 'ICT',
        'RE' => 'REL', // Religious Education
        'CS' => 'CON', // Consumer Science
        'SS' => 'SOC', // Social Studies
        'EA' => 'EXP'  // Expressive Arts
    ];
    
    if (isset($specialCases[$prefix])) {
        $prefix = $specialCases[$prefix];
    }
    
    // Format: First 3 letters + Grade number + 100
    return $prefix . $gradeNumber . '100';
}

function createSubject($name, $code) {
    $pdo = getDB();
    $stmt = $pdo->prepare("INSERT INTO subjects (name, code) VALUES (?, ?)");
    $stmt->execute([$name, $code]);
    return $pdo->lastInsertId();
}

function createSubjectForGrade($subjectName, $gradeNumber) {
    $code = generateSubjectCode($subjectName, $gradeNumber);
    return createSubject($subjectName, $code);
}

function updateSubject($id, $name, $code) {
    $pdo = getDB();
    $stmt = $pdo->prepare("UPDATE subjects SET name = ?, code = ? WHERE id = ?");
    $stmt->execute([$name, $code, $id]);
    return $stmt->rowCount() > 0;
}

function deleteSubject($id) {
    $pdo = getDB();
    $stmt = $pdo->prepare("DELETE FROM subjects WHERE id = ?");
    $stmt->execute([$id]);
    return $stmt->rowCount() > 0;
}

function assignSubjectToGrade($subjectId, $gradeId) {
    $pdo = getDB();
    $stmt = $pdo->prepare("INSERT IGNORE INTO grade_subject (grade_id, subject_id) VALUES (?, ?)");
    $stmt->execute([$gradeId, $subjectId]);
    return $stmt->rowCount() > 0;
}

function removeSubjectFromGrade($subjectId, $gradeId) {
    $pdo = getDB();
    $stmt = $pdo->prepare("DELETE FROM grade_subject WHERE grade_id = ? AND subject_id = ?");
    $stmt->execute([$gradeId, $subjectId]);
    return $stmt->rowCount() > 0;
}

function getGradeSubjectAssignments() {
    $pdo = getDB();
    $stmt = $pdo->query("
        SELECT gs.*, g.name as grade_name, s.name as subject_name, s.code as subject_code
        FROM grade_subject gs
        JOIN grades g ON gs.grade_id = g.id
        JOIN subjects s ON gs.subject_id = s.id
        ORDER BY g.name, s.name
    ");
    return $stmt->fetchAll();
}

/**
 * Clean up unverified users older than 24 hours
 * This allows users to re-register if they didn't verify their email
 */
function cleanupUnverifiedUsers() {
    $pdo = getDB();
    
    try {
        // Delete unverified users older than 24 hours
        $stmt = $pdo->prepare("
            DELETE FROM students 
            WHERE email_verified = FALSE 
            AND created_at < DATE_SUB(NOW(), INTERVAL 24 HOUR)
        ");
        
        $result = $stmt->execute();
        $deletedCount = $stmt->rowCount();
        
        // Log the cleanup operation
        error_log("Cleanup: Deleted $deletedCount unverified users older than 24 hours");
        
        return [
            'success' => true,
            'deleted_count' => $deletedCount,
            'message' => "Successfully deleted $deletedCount unverified users"
        ];
        
    } catch (Exception $e) {
        error_log("Cleanup error: " . $e->getMessage());
        return [
            'success' => false,
            'deleted_count' => 0,
            'message' => "Error during cleanup: " . $e->getMessage()
        ];
    }
}

/**
 * Get statistics about unverified users
 */
function getUnverifiedUserStats() {
    $pdo = getDB();
    
    // Count unverified users
    $stmt = $pdo->query("
        SELECT COUNT(*) as total_unverified 
        FROM students 
        WHERE email_verified = FALSE
    ");
    $totalUnverified = $stmt->fetch()['total_unverified'];
    
    // Count unverified users older than 24 hours
    $stmt = $pdo->query("
        SELECT COUNT(*) as expired_unverified 
        FROM students 
        WHERE email_verified = FALSE 
        AND created_at < DATE_SUB(NOW(), INTERVAL 24 HOUR)
    ");
    $expiredUnverified = $stmt->fetch()['expired_unverified'];
    
    // Count unverified users by age
    $stmt = $pdo->query("
        SELECT 
            COUNT(CASE WHEN created_at >= DATE_SUB(NOW(), INTERVAL 1 HOUR) THEN 1 END) as last_hour,
            COUNT(CASE WHEN created_at >= DATE_SUB(NOW(), INTERVAL 6 HOUR) THEN 1 END) as last_6_hours,
            COUNT(CASE WHEN created_at >= DATE_SUB(NOW(), INTERVAL 12 HOUR) THEN 1 END) as last_12_hours,
            COUNT(CASE WHEN created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR) THEN 1 END) as last_24_hours
        FROM students 
        WHERE email_verified = FALSE
    ");
    $ageStats = $stmt->fetch();
    
    return [
        'total_unverified' => $totalUnverified,
        'expired_unverified' => $expiredUnverified,
        'age_breakdown' => $ageStats
    ];
}

/**
 * Fix password consistency for all students
 * Ensures all students use their last name as password
 */
function fixPasswordConsistency() {
    $pdo = getDB();
    
    try {
        $pdo->beginTransaction();
        
        // Get all students
        $stmt = $pdo->query("SELECT id, last_name FROM students");
        $students = $stmt->fetchAll();
        
        $fixedCount = 0;
        foreach ($students as $student) {
            // Update password to use last name
            $passwordHash = password_hash($student['last_name'], PASSWORD_DEFAULT);
            $updateStmt = $pdo->prepare("UPDATE students SET password_hash = ? WHERE id = ?");
            $updateStmt->execute([$passwordHash, $student['id']]);
            $fixedCount++;
        }
        
        $pdo->commit();
        
        return [
            'success' => true,
            'fixed_count' => $fixedCount,
            'message' => "Successfully fixed passwords for $fixedCount students"
        ];
        
    } catch (Exception $e) {
        $pdo->rollBack();
        error_log("Password consistency fix error: " . $e->getMessage());
        return [
            'success' => false,
            'fixed_count' => 0,
            'message' => "Error fixing passwords: " . $e->getMessage()
        ];
    }
}

/**
 * Check password consistency for all students
 */
function checkPasswordConsistency() {
    $pdo = getDB();
    
    $stmt = $pdo->query("SELECT id, username, last_name, password_hash FROM students WHERE email_verified = TRUE");
    $students = $stmt->fetchAll();
    
    $inconsistent = [];
    $consistent = [];
    
    foreach ($students as $student) {
        // Check if the stored hash matches the last name
        if (password_verify($student['last_name'], $student['password_hash'])) {
            $consistent[] = $student;
        } else {
            $inconsistent[] = $student;
        }
    }
    
    return [
        'total_verified' => count($students),
        'consistent' => count($consistent),
        'inconsistent' => count($inconsistent),
        'inconsistent_students' => $inconsistent
    ];
}

/**
 * Verify student password
 */
function verifyStudentPassword($username, $password) {
    $pdo = getDB();
    $stmt = $pdo->prepare("SELECT id, username, password_hash, last_name FROM students WHERE username = ? AND email_verified = TRUE");
    $stmt->execute([$username]);
    $student = $stmt->fetch();
    
    if (!$student) {
        return false;
    }
    
    // Check if password matches the stored hash
    if (password_verify($password, $student['password_hash'])) {
        return $student;
    }
    
    // Also check if password matches last name (for compatibility)
    if ($password === $student['last_name']) {
        return $student;
    }
    
    return false;
}

/**
 * Update student subjects
 */
function updateStudentSubjects($studentId, $subjectIds) {
    $pdo = getDB();
    
    try {
        $pdo->beginTransaction();
        
        // Remove existing subject associations
        $stmt = $pdo->prepare("DELETE FROM student_subject WHERE student_id = ?");
        $stmt->execute([$studentId]);
        
        // Add new subject associations
        if (!empty($subjectIds)) {
            $stmt = $pdo->prepare("INSERT INTO student_subject (student_id, subject_id) VALUES (?, ?)");
            foreach ($subjectIds as $subjectId) {
                $stmt->execute([$studentId, (int)$subjectId]);
            }
        }
        
        $pdo->commit();
        
        // Trigger bot to sync updated student subjects to Moodle
        triggerMoodleSyncBot();
        
        return true;
    } catch (Exception $e) {
        $pdo->rollBack();
        error_log("Error updating student subjects: " . $e->getMessage());
        return false;
    }
}