# Complete Guide: Multi-Tenant Skolo-Kine with Cursor AI

## 📋 Table of Contents
1. [Pre-Implementation Planning](#pre-implementation-planning)
2. [Phase 1: Database Schema (Week 1)](#phase-1-database-schema)
3. [Phase 2: Authentication & Access Control (Week 2)](#phase-2-authentication-access-control)
4. [Phase 3: UI/UX & Tenant Management (Week 3)](#phase-3-uiux-tenant-management)
5. [Phase 4: Testing & Deployment (Week 4)](#phase-4-testing-deployment)
6. [Cursor AI Prompts Library](#cursor-ai-prompts-library)

---

## Pre-Implementation Planning

### Step 1: Answer Business Questions

**Create file:** `docs/multi-tenant-requirements.md`

```markdown
# Multi-Tenant Requirements

## Business Model
- [ ] Initial tenant count: ___
- [ ] Pricing tiers: Basic / Premium / Enterprise
- [ ] Custom domain support: Yes / No
- [ ] Billing model: Monthly / Annual / Per-student

## Technical Architecture
- [ ] Tenant identification: Subdomain / Path / Domain
- [ ] Database strategy: Shared DB / Separate DB per tenant
- [ ] Email system: Shared SMTP / Tenant-specific
- [ ] Payment integration: Shared / Tenant-specific MTN MoMo

## Migration Plan
- [ ] Existing data treatment: Make first tenant / Keep separate
- [ ] Acceptable downtime: ___
- [ ] Rollback plan: Yes / No
```

### Step 2: Choose Implementation Approach

**Give this to Cursor:**

```
I want to implement multi-tenancy in my Skolo-Kine school management system.

REQUIREMENTS:
- Current system: Single school PHP/MySQL application
- Target: Multiple schools on one platform
- Architecture preference: [Choose one]
  * Option A: Subdomain-based (school1.skolokine.com)
  * Option B: Path-based (skolokine.com/school1)
  * Option C: Custom domain (school1.com)

APPROACH: [Choose one]
- Full Multi-Tenancy: Complete isolation, custom branding, all features
- Simplified Multi-Tenancy: Basic isolation, shared branding, core features only

Please analyze my current database schema and create a detailed implementation plan.
```

---

## Phase 1: Database Schema (Week 1)

### Day 1-2: Create Tenant Management Tables

**Prompt for Cursor:**

```
TASK: Create multi-tenant database schema

CONTEXT: I have an existing school management system with these tables:
- students, grades, subjects, student_subject, admins
- payments, payment_status
- email_queue, email_logs, security_logs

REQUIREMENTS:
1. Create a `tenants` table with:
   - tenant_id (primary key, UUID)
   - tenant_name, subdomain, custom_domain
   - plan_type (basic/premium/enterprise)
   - status (active/suspended/trial)
   - created_at, expires_at
   - settings (JSON for tenant-specific config)

2. Create `tenant_users` table to replace/extend admins:
   - id, tenant_id, user_id
   - role (super_admin/tenant_admin/staff)
   - permissions (JSON)

3. Create `tenant_settings` table:
   - tenant_id, setting_key, setting_value
   - For: branding, email config, payment config

4. Create migration script to add tenant_id to all existing tables

Please generate:
- Complete SQL schema
- Migration script (UP and DOWN)
- Rollback script
```

**Expected Output:** `database/migrations/001_add_multi_tenancy.sql`

---

### Day 3-4: Add tenant_id to Existing Tables

**Prompt for Cursor:**

```
TASK: Add tenant_id column to all existing tables

CURRENT TABLES:
- students, grades, subjects, student_subject
- admins, payments, payment_status
- email_queue, email_logs, security_logs
- system_metrics, app_performance, etc.

REQUIREMENTS:
1. Add tenant_id CHAR(36) column to each table
2. Add foreign key constraint to tenants(tenant_id)
3. Add index on tenant_id for performance
4. Update existing data to default tenant (UUID: generate one)
5. Make tenant_id NOT NULL after migration

Generate:
- ALTER TABLE statements for each table
- Index creation statements
- Data migration script
- Verification queries

IMPORTANT: Create backup commands first!
```

**Expected Output:** `database/migrations/002_add_tenant_columns.sql`

---

### Day 5: Create Database Helper Functions

**Prompt for Cursor:**

```
TASK: Create tenant-aware database helper functions

CREATE FILE: includes/tenant_db.php

REQUIREMENTS:
1. Function: getTenantConnection($tenant_id)
   - Returns database connection scoped to tenant
   - Sets session variable for current tenant

2. Function: getTenantData($table, $conditions, $tenant_id)
   - Automatically adds tenant_id to WHERE clause
   - Prevents cross-tenant data access

3. Function: insertTenantData($table, $data, $tenant_id)
   - Automatically adds tenant_id to INSERT
   - Validates tenant exists

4. Function: updateTenantData($table, $data, $conditions, $tenant_id)
   - Automatically adds tenant_id to WHERE clause
   - Prevents cross-tenant updates

5. Function: deleteTenantData($table, $conditions, $tenant_id)
   - Automatically adds tenant_id to WHERE clause
   - Prevents cross-tenant deletes

6. Function: validateTenantAccess($user_id, $tenant_id)
   - Checks if user has access to tenant
   - Returns user role in tenant

Include comprehensive error handling and logging.
```

---

## Phase 2: Authentication & Access Control (Week 2)

### Day 1-2: Tenant Detection System

**Prompt for Cursor:**

```
TASK: Create tenant detection and resolution system

CREATE FILE: includes/tenant_resolver.php

REQUIREMENTS:
1. Detect tenant from:
   - Subdomain (school1.skolokine.com → school1)
   - Custom domain (myschool.com → lookup in tenants table)
   - Path parameter (?tenant=school1) [fallback for testing]

2. Function: getCurrentTenant()
   - Returns tenant object with all details
   - Caches result in session
   - Returns null if invalid/suspended tenant

3. Function: setCurrentTenant($tenant_id)
   - Sets tenant in session
   - Validates tenant exists and is active

4. Function: requireTenant()
   - Middleware to ensure tenant is set
   - Redirects to tenant selection if not set

5. Handle edge cases:
   - Suspended tenants (show suspension message)
   - Expired trials (show upgrade prompt)
   - Invalid subdomains (404 page)

Also update .htaccess for subdomain routing.
```

---

### Day 3-4: Multi-Tenant Authentication

**Prompt for Cursor:**

```
TASK: Update authentication system for multi-tenancy

UPDATE FILES:
- admin/login.php
- admin/logout.php
- includes/auth.php

REQUIREMENTS:
1. Login process should:
   - Verify user belongs to current tenant
   - Check user role in tenant (super_admin/tenant_admin/staff)
   - Store tenant_id in session
   - Deny access if user not in tenant

2. Create Super Admin login (separate):
   - Access all tenants
   - Manage tenant accounts
   - View cross-tenant analytics

3. Update all authentication checks:
   - requireLogin() → requireTenantLogin($required_role)
   - Check both user authentication AND tenant access

4. Create middleware:
   - requireSuperAdmin() - only super admins
   - requireTenantAdmin() - tenant admins or higher
   - requireStaff() - any authenticated user in tenant

5. Session structure:
   $_SESSION['user_id']
   $_SESSION['tenant_id']
   $_SESSION['tenant_role']
   $_SESSION['is_super_admin']
```

---

### Day 5: Access Control Lists (ACL)

**Prompt for Cursor:**

```
TASK: Create role-based access control system

CREATE FILE: includes/tenant_acl.php

REQUIREMENTS:
1. Define permissions matrix:
   SUPER_ADMIN: Everything across all tenants
   TENANT_ADMIN: Everything within their tenant
   STAFF: Limited access (view students, grades)
   TEACHER: View/edit grades only
   ACCOUNTANT: Payments only

2. Function: hasPermission($user_id, $tenant_id, $permission)
   - Checks if user has specific permission
   - Returns boolean

3. Function: getPermissions($user_id, $tenant_id)
   - Returns array of all permissions for user

4. Create permission constants:
   PERM_VIEW_STUDENTS
   PERM_EDIT_STUDENTS
   PERM_DELETE_STUDENTS
   PERM_VIEW_PAYMENTS
   PERM_MANAGE_PAYMENTS
   etc.

5. Update all admin pages to check permissions before allowing actions
```

---

## Phase 3: UI/UX & Tenant Management (Week 3)

### Day 1-2: Super Admin Panel

**Prompt for Cursor:**

```
TASK: Create Super Admin dashboard and tenant management

CREATE FILES:
- super_admin/dashboard.php
- super_admin/tenants/list.php
- super_admin/tenants/create.php
- super_admin/tenants/edit.php
- super_admin/tenants/suspend.php

REQUIREMENTS:
1. Dashboard should show:
   - Total tenants (active/suspended/trial)
   - Revenue metrics
   - System health across all tenants
   - Recent activity log

2. Tenant list should show:
   - Tenant name, subdomain, plan, status
   - Number of students, admins
   - Last login, created date
   - Quick actions: Edit, Suspend, Delete

3. Create tenant form:
   - Tenant name, subdomain
   - Admin email (creates first admin account)
   - Plan selection
   - Trial days or activation date

4. Edit tenant:
   - Update plan
   - Change subdomain
   - Add custom domain
   - Modify settings

5. Tenant statistics:
   - Storage usage
   - API calls
   - Active users
   - Payment history

Use modern UI with Tailwind CSS.
```

---

### Day 3-4: Tenant-Specific Settings

**Prompt for Cursor:**

```
TASK: Create tenant settings management interface

CREATE FILES:
- admin/settings/general.php
- admin/settings/branding.php
- admin/settings/email.php
- admin/settings/payment.php
- admin/settings/api.php

REQUIREMENTS:
1. General Settings:
   - School name, address, phone
   - Academic year settings
   - Timezone, language

2. Branding Settings:
   - Upload logo (stored per tenant)
   - Primary/secondary colors
   - Custom CSS (for premium plans)
   - Email header/footer

3. Email Settings (per tenant):
   - SMTP server, port, username, password
   - From email, from name
   - Test email function

4. Payment Settings (per tenant):
   - MTN MoMo credentials (separate per tenant)
   - Currency (default SZL, but allow override)
   - Payment terms, late fees

5. API Settings:
   - Moodle URL, token (per tenant)
   - Webhook URLs
   - API key for integrations

All settings should be stored in tenant_settings table and cached.
```

---

### Day 5: Tenant Isolation Verification

**Prompt for Cursor:**

```
TASK: Create tenant isolation test suite

CREATE FILE: tools/test_tenant_isolation.php

REQUIREMENTS:
Test that:
1. User from Tenant A cannot access Tenant B data
2. All queries include tenant_id filter
3. No cross-tenant data leakage in:
   - Student records
   - Payment data
   - Grades
   - Admin access

4. Test scenarios:
   - Direct URL manipulation
   - SQL injection attempts
   - Session hijacking
   - API endpoint access

5. Generate report showing:
   - ✅ Tests passed
   - ❌ Security vulnerabilities found
   - Recommendations

Also create automated test that runs daily via cron.
```

---

## Phase 4: Testing & Deployment (Week 4)

### Day 1-2: Update All Existing Code

**Prompt for Cursor:**

```
TASK: Update all existing code to be tenant-aware

SCAN AND UPDATE:
- admin/*.php
- includes/*.php
- All CRUD operations

REQUIREMENTS:
1. Find all database queries
2. Add tenant_id to WHERE clauses
3. Add tenant_id to INSERT statements
4. Update session checks to include tenant verification

5. Files to prioritize:
   - admin/students/*.php
   - admin/grades/*.php
   - admin/payments/*.php
   - admin/reports/*.php

6. Update all forms to include hidden tenant_id field (from session)

7. Update all reports to filter by current tenant

Generate a checklist of all modified files.
```

---

### Day 3: Data Migration Script

**Prompt for Cursor:**

```
TASK: Create data migration script for existing data

CREATE FILE: tools/migrate_to_multitenant.php

REQUIREMENTS:
1. Create default tenant:
   - Name: "Default School" (or use existing school name)
   - Subdomain: "main" or "default"
   - Status: active
   - Plan: enterprise

2. Assign all existing data to default tenant:
   - Update all tables with default tenant_id
   - Create default admin user in tenant_users

3. Verification:
   - Count records before migration
   - Count records after migration
   - Verify no data loss

4. Rollback capability:
   - Save original state
   - Restore if migration fails

5. Progress reporting:
   - Show progress per table
   - Estimated time remaining
   - Success/failure status

6. Backup:
   - Create full database backup before migration
   - Store backup with timestamp
```

---

### Day 4: Performance Testing

**Prompt for Cursor:**

```
TASK: Create performance testing and optimization

CREATE FILE: tools/test_multitenant_performance.php

TEST SCENARIOS:
1. Concurrent tenant access (simulate 10 tenants)
2. Query performance with tenant_id filters
3. Database index effectiveness
4. Session management overhead

OPTIMIZATION:
1. Add missing indexes
2. Implement query caching per tenant
3. Optimize slow queries
4. Add tenant-specific connection pooling

MONITORING:
1. Log query execution times
2. Track tenant-specific load
3. Alert on performance degradation
4. Generate optimization recommendations

Create performance report comparing before/after multi-tenancy.
```

---

### Day 5: Deployment Checklist

**Prompt for Cursor:**

```
TASK: Create deployment checklist and automation

CREATE FILES:
- deploy/pre_deployment_checklist.md
- deploy/deployment_script.sh
- deploy/post_deployment_tests.php
- deploy/rollback_script.sh

CHECKLIST:
1. Pre-Deployment:
   - [ ] Full database backup
   - [ ] Code backup
   - [ ] Test on staging environment
   - [ ] Performance tests passed
   - [ ] Security audit completed
   - [ ] Documentation updated

2. Deployment:
   - [ ] Put site in maintenance mode
   - [ ] Run database migrations
   - [ ] Deploy new code
   - [ ] Clear caches
   - [ ] Run post-deployment tests

3. Post-Deployment:
   - [ ] Verify default tenant works
   - [ ] Test creating new tenant
   - [ ] Test tenant isolation
   - [ ] Check error logs
   - [ ] Monitor performance

4. Rollback Plan:
   - [ ] Restore database backup
   - [ ] Restore code backup
   - [ ] Test system functionality

Create automated script that runs through these steps.
```

---

## Cursor AI Prompts Library

### Quick Prompts for Common Tasks

#### 1. Generate Tenant CRUD Operations
```
Create complete CRUD operations for tenant management:
- List all tenants (with pagination, search, filters)
- View tenant details
- Create new tenant (with validation)
- Update tenant information
- Soft delete/suspend tenant
- Restore suspended tenant

Include proper error handling, security checks, and audit logging.
```

#### 2. Add Tenant Awareness to Existing Feature
```
Update [FEATURE_NAME] to be tenant-aware:

FILE: [path/to/file.php]

CURRENT BEHAVIOR: [describe current behavior]
REQUIRED CHANGES:
1. Add tenant_id filter to all queries
2. Verify user has access to current tenant
3. Update forms to include tenant context
4. Add tenant-specific settings support

Maintain backward compatibility where possible.
```

#### 3. Create Tenant API Endpoints
```
Create REST API endpoints for tenant operations:

ENDPOINTS:
POST /api/tenants - Create tenant
GET /api/tenants - List tenants
GET /api/tenants/{id} - Get tenant details
PUT /api/tenants/{id} - Update tenant
DELETE /api/tenants/{id} - Delete tenant
GET /api/tenants/{id}/stats - Get tenant statistics

Include:
- Authentication (JWT tokens)
- Authorization (super admin only)
- Input validation
- Rate limiting
- API documentation
```

#### 4. Generate Tenant Reports
```
Create tenant analytics and reporting:

REPORTS NEEDED:
1. Tenant usage dashboard
2. Revenue by tenant
3. Active users per tenant
4. Storage usage per tenant
5. API calls per tenant
6. Tenant growth over time

Output formats: PDF, Excel, JSON
Include filters: date range, tenant, status
Visualizations: Charts, graphs, tables
```

#### 5. Implement Tenant Branding
```
Create tenant-specific branding system:

FEATURES:
1. Upload custom logo per tenant
2. Custom color scheme (primary, secondary, accent)
3. Custom email templates with tenant branding
4. Tenant-specific header/footer
5. Custom CSS for premium tenants

Storage: Store tenant assets in separate directories
Caching: Cache tenant branding settings
Fallback: Default branding if tenant branding not set
```

---

## Implementation Timeline

### Week 1: Foundation
```
Day 1-2: Database schema design and creation
Day 3-4: Migrate existing tables, add tenant_id
Day 5: Create tenant database helpers

DELIVERABLE: Tenant-aware database structure
```

### Week 2: Security
```
Day 1-2: Tenant detection and resolution
Day 3-4: Multi-tenant authentication system
Day 5: Role-based access control

DELIVERABLE: Secure tenant isolation
```

### Week 3: Interface
```
Day 1-2: Super admin panel
Day 3-4: Tenant-specific settings
Day 5: Tenant isolation testing

DELIVERABLE: Complete tenant management UI
```

### Week 4: Launch
```
Day 1-2: Update all existing code
Day 3: Data migration
Day 4: Performance testing
Day 5: Deployment and verification

DELIVERABLE: Production-ready multi-tenant system
```

---

## Cost Breakdown

### Development Costs
```
Database Schema (Week 1):        $500-800
Authentication System (Week 2):   $800-1,200
UI/UX Implementation (Week 3):    $800-1,200
Testing & Deployment (Week 4):    $400-600
Buffer for issues (10%):          $250-380

TOTAL: $2,750-4,180
```

### Infrastructure Costs (Ongoing)
```
Database (larger storage):        $50-100/month
Subdomain SSL certificates:       $0 (Let's Encrypt)
CDN for tenant assets:            $20-50/month
Monitoring tools:                 $30/month

MONTHLY: $100-180
```

---

## Testing Checklist

### Security Tests
```
- [ ] Cross-tenant data access attempts fail
- [ ] SQL injection with tenant_id fails
- [ ] Session hijacking prevented
- [ ] Direct URL manipulation blocked
- [ ] API endpoints enforce tenant isolation
```

### Functionality Tests
```
- [ ] Create new tenant works
- [ ] Tenant admin can access only their data
- [ ] Super admin can access all tenants
- [ ] Tenant settings persist correctly
- [ ] Payment integration per tenant works
- [ ] Email system uses tenant settings
```

### Performance Tests
```
- [ ] Query response time < 100ms with tenant_id
- [ ] Page load time < 2s
- [ ] Support 100 concurrent tenants
- [ ] Database queries use indexes
- [ ] No N+1 query problems
```

---

## Maintenance & Support

### Daily Tasks
```
- Monitor tenant creation/suspension
- Check error logs for tenant-related issues
- Verify tenant isolation
```

### Weekly Tasks
```
- Review tenant usage statistics
- Check database performance
- Update tenant billing
```

### Monthly Tasks
```
- Performance optimization review
- Security audit
- Backup verification
- Tenant satisfaction survey
```

---

## Success Metrics

### Technical Metrics
```
- 100% tenant data isolation (0 cross-tenant leaks)
- Query performance < 100ms
- 99.9% uptime per tenant
- Zero data loss during migration
```

### Business Metrics
```
- Number of active tenants
- Revenue per tenant
- Tenant churn rate
- Average tenant lifetime value
```

---

## Next Steps

1. **Review this guide completely**
2. **Answer the business questions** in the requirements document
3. **Choose your implementation approach** (Full vs Simplified)
4. **Start with Week 1, Day 1** - Give Cursor the first prompt
5. **Test thoroughly** at each phase
6. **Deploy incrementally** - don't rush

🎯 **Goal:** Production-ready multi-tenant system in 4 weeks!