# Multi-Tenant Implementation Plan
## Skolo-Kine School Management System

### ✅ Your Requirements Summary
- **Target:** 1-20 schools in Year 1
- **Pricing:** Per-student (E0.30 per subject)
- **Access Method:** Folder-based (`/school1`, `/school2`) or query parameter
- **Custom Domains:** No
- **Current Status:** 60% complete (academy_reference system exists)

---

## 📊 Current Database Analysis

### ✅ Already Multi-Tenant Ready
```
✓ students.academy_reference
✓ payments.academy_reference  
✓ admins.academy_reference
✓ academy_references table (full tenant management)
✓ platform_fees table (tracks per-academy earnings)
✓ admin_wallet system (per-admin wallet tracking)
```

### ❌ Missing Multi-Tenancy
```
✗ grades table (no academy_reference)
✗ subjects table (no academy_reference)
✗ grade_subject table (no academy_reference)
✗ email templates (no per-academy customization)
✗ tenant-specific settings (branding, contact info)
✗ school selection UI
✗ query filtering by academy
```

---

## 🎯 Implementation Strategy

### **Option A: Shared Grades/Subjects (RECOMMENDED)**
**Best for:** 1-20 schools with similar curricula

**Approach:**
- Grades and subjects are SHARED across all schools
- Only student enrollment is school-specific
- Much simpler to maintain
- Schools can't customize grade names or subjects

**Pros:**
- ✅ Simpler implementation (2-3 days)
- ✅ Less database complexity
- ✅ Standard curriculum across schools
- ✅ Easier reporting and analytics

**Cons:**
- ❌ All schools must use same grade structure
- ❌ Can't have school-specific subjects
- ❌ Less flexibility

**Use Case:** Perfect if all schools follow the same curriculum (e.g., national curriculum)

---

### **Option B: School-Specific Grades/Subjects**
**Best for:** Schools with different curricula

**Approach:**
- Each school has their own grades and subjects
- Add `academy_reference` to grades, subjects, grade_subject
- More flexibility but more complexity

**Pros:**
- ✅ Each school can customize everything
- ✅ Support different curricula
- ✅ Ultimate flexibility

**Cons:**
- ❌ More complex queries
- ❌ Duplicate data (grades, subjects)
- ❌ Takes longer to implement (5-7 days)
- ❌ More maintenance

**Use Case:** If schools have completely different grade systems or curricula

---

## 🚀 Recommended Implementation: Option A

### Phase 1: Database Updates (Day 1) - 4 hours

#### 1.1 Create Tenant Settings Table
```sql
CREATE TABLE IF NOT EXISTS `tenant_settings` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `academy_reference` VARCHAR(50) NOT NULL,
    `setting_key` VARCHAR(100) NOT NULL,
    `setting_value` TEXT,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    UNIQUE KEY `uk_academy_setting` (`academy_reference`, `setting_key`),
    KEY `idx_academy_ref` (`academy_reference`),
    CONSTRAINT `fk_tenant_settings_academy` FOREIGN KEY (`academy_reference`) 
        REFERENCES `academy_references`(`reference_code`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
```

**Settings to Store:**
- `school_name`: Full school name
- `school_address`: Physical address
- `school_phone`: Contact number
- `school_email`: Contact email
- `logo_url`: School logo path
- `primary_color`: Brand color #hexcode
- `secondary_color`: Brand color #hexcode
- `email_from_name`: Email sender name
- `email_signature`: School email signature
- `moodle_url`: School-specific Moodle (if different)
- `academic_year`: e.g., "2024/2025"

#### 1.2 Add Missing Indexes
```sql
-- Ensure all academy_reference columns are indexed
ALTER TABLE `students` ADD INDEX IF NOT EXISTS `idx_students_academy_ref` (`academy_reference`);
ALTER TABLE `payments` ADD INDEX IF NOT EXISTS `idx_payments_academy_ref` (`academy_reference`);
ALTER TABLE `platform_fees` ADD INDEX IF NOT EXISTS `idx_platform_fees_academy_ref` (`academy_reference`);
```

---

### Phase 2: School Selection System (Day 2) - 6 hours

#### 2.1 Create School Selector Page
**File:** `select_school.php`

Features:
- List all active schools from `academy_references`
- Show school logo (if available)
- Display school name
- Store selected school in session
- Redirect to dashboard

#### 2.2 Create School Context Manager
**File:** `includes/school_context.php`

```php
<?php
function getCurrentSchool() {
    return $_SESSION['academy_reference'] ?? null;
}

function setCurrentSchool($reference_code) {
    // Validate school exists and is active
    $pdo = getDB();
    $stmt = $pdo->prepare("
        SELECT * FROM academy_references 
        WHERE reference_code = ? AND is_active = 1
    ");
    $stmt->execute([$reference_code]);
    $school = $stmt->fetch();
    
    if ($school) {
        $_SESSION['academy_reference'] = $reference_code;
        $_SESSION['academy_name'] = $school['academy_name'];
        $_SESSION['academy_id'] = $school['id'];
        return true;
    }
    return false;
}

function requireSchool() {
    if (!getCurrentSchool()) {
        header('Location: /select_school.php');
        exit;
    }
}

function getSchoolSetting($key, $default = null) {
    $pdo = getDB();
    $stmt = $pdo->prepare("
        SELECT setting_value FROM tenant_settings 
        WHERE academy_reference = ? AND setting_key = ?
    ");
    $stmt->execute([getCurrentSchool(), $key]);
    $result = $stmt->fetchColumn();
    return $result ?: $default;
}
```

---

### Phase 3: Update Authentication (Day 2-3) - 4 hours

#### 3.1 Update Login System
**File:** `admin/login.php`

Add after successful login:
```php
// After admin login succeeds
$academy_ref = $admin['academy_reference'];
setCurrentSchool($academy_ref);

// Redirect to dashboard
header('Location: dashboard.php');
```

#### 3.2 Add School Context to All Admin Pages
Add to top of every admin page:
```php
require_once '../includes/school_context.php';
requireSchool(); // Ensure school is selected
```

---

### Phase 4: Update Queries (Day 3-4) - 8 hours

#### 4.1 Update Student Queries
**Pattern to follow:**

**BEFORE:**
```php
$stmt = $pdo->query("SELECT * FROM students WHERE status = 'active'");
```

**AFTER:**
```php
$academy_ref = getCurrentSchool();
$stmt = $pdo->prepare("
    SELECT * FROM students 
    WHERE status = 'active' AND academy_reference = ?
");
$stmt->execute([$academy_ref]);
```

#### 4.2 Files to Update (Priority Order)
1. `admin/dashboard.php` - Add school filter to all queries
2. `admin/create.php` - Add academy_reference to INSERT
3. `admin/edit.php` - Add academy_reference to WHERE clause
4. `admin/delete.php` - Add academy_reference to WHERE clause
5. `admin/payments.php` - Already has academy_reference
6. All report pages - Add school filter

#### 4.3 Update Helper Functions
**File:** `includes/functions.php`

Add school context to all data retrieval functions:
```php
function getStudents($filters = []) {
    $pdo = getDB();
    $academy_ref = getCurrentSchool();
    
    $sql = "SELECT s.*, g.name as grade_name 
            FROM students s 
            JOIN grades g ON s.grade_id = g.id 
            WHERE s.academy_reference = ?";
    
    $params = [$academy_ref];
    
    // Add other filters...
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    return $stmt->fetchAll();
}
```

---

### Phase 5: School Management UI (Day 5) - 6 hours

#### 5.1 Create Super Admin Section
**Files:**
- `super_admin/schools.php` - List all schools
- `super_admin/create_school.php` - Add new school
- `super_admin/edit_school.php` - Edit school details
- `super_admin/suspend_school.php` - Suspend/activate

#### 5.2 Create School Settings Page
**File:** `admin/settings/school_settings.php`

Allow each school admin to customize:
- School name, address, phone, email
- Logo upload
- Brand colors
- Email settings
- Academic year

---

### Phase 6: Testing & Verification (Day 6-7) - 8 hours

#### 6.1 Test Scenarios
```
✓ Create new school
✓ Admin from School A cannot see School B students
✓ Payments are tracked per school
✓ Platform fees calculated correctly per school
✓ Wallet earnings separated per admin
✓ Reports filtered by school
✓ Email sends with school branding
✓ Moodle sync per school (if applicable)
```

#### 6.2 Create Test Script
**File:** `tools/test_multi_tenant_isolation.php`

Tests cross-school data access and ensures isolation.

---

## 💰 Pricing & Revenue Model

### Current System
- Platform fee: **E0.50 per student**
- Subject price: **E0.30 per subject**

### Multi-Tenant Pricing Options

#### Option 1: Keep Current Model (RECOMMENDED)
**Per-Student Fee:**
- You keep: E0.50 per student per month
- School keeps: Student payments minus your fee
- Simple, transparent, scales well

**Example:**
- School has 100 students
- Each student pays E3.00 (10 subjects × E0.30)
- School collects: E300
- Your platform fee: E50 (100 students × E0.50)
- School nets: E250

#### Option 2: Percentage Model
- You take: 20% of all payments
- School keeps: 80%

#### Option 3: Hybrid Model
- Base fee: E50/month per school
- Plus: E0.30 per student
- Good for predictable revenue

### Year 1 Revenue Projection (Conservative)

Assuming:
- 10 schools onboarded
- Average 50 students per school
- E0.50 platform fee per student/month

**Monthly Revenue:**
```
10 schools × 50 students × E0.50 = E250/month
```

**Annual Revenue:**
```
E250/month × 12 months = E3,000/year
```

**If you reach 20 schools:**
```
20 schools × 100 students × E0.50 × 12 = E12,000/year
```

---

## 📁 File Structure Recommendations

### Folder-Based Multi-Tenancy

#### Option A: Query Parameter (SIMPLEST)
```
https://skolokine.com/admin/dashboard.php?school=KINE
https://skolokine.com/admin/dashboard.php?school=STMARY
```

**Implementation:**
```php
// In includes/school_context.php
if (isset($_GET['school'])) {
    setCurrentSchool($_GET['school']);
}
```

**Pros:**
- ✅ Zero configuration
- ✅ Works immediately
- ✅ Easy to test

**Cons:**
- ❌ Not as "clean" URLs
- ❌ School code visible in URL

---

#### Option B: Path-Based (BETTER UX)
```
https://skolokine.com/school/KINE/admin/dashboard.php
https://skolokine.com/school/STMARY/admin/dashboard.php
```

**Implementation:** Update `.htaccess`
```apache
RewriteEngine On
RewriteRule ^school/([^/]+)/(.*)$ /$2?school=$1 [L,QSA]
```

**Pros:**
- ✅ Clean URLs
- ✅ Professional appearance
- ✅ School context in path

**Cons:**
- ❌ Requires .htaccess configuration
- ❌ Slightly more complex

---

## 🔒 Security Considerations

### Must-Have Security Measures

1. **Query Validation**
```php
// Always validate school access
function validateSchoolAccess($academy_ref) {
    $admin_school = $_SESSION['admin']['academy_reference'];
    
    // Super admin can access all schools
    if ($_SESSION['is_super_admin']) {
        return true;
    }
    
    // Regular admin can only access their school
    return $academy_ref === $admin_school;
}
```

2. **Data Isolation**
```php
// NEVER trust user input for school selection
$academy_ref = getCurrentSchool(); // From session, not $_GET
```

3. **Admin Permissions**
```php
// Create role hierarchy
SUPER_ADMIN: Access all schools, create new schools
SCHOOL_ADMIN: Access only their school, full permissions
TEACHER: Access only their school, limited permissions
```

---

## ✅ Implementation Checklist

### Week 1: Foundation
- [ ] Create `tenant_settings` table
- [ ] Create school selection page (`select_school.php`)
- [ ] Create school context manager (`includes/school_context.php`)
- [ ] Add school requirement to all admin pages
- [ ] Test school switching

### Week 2: Query Updates
- [ ] Update dashboard queries
- [ ] Update student CRUD operations
- [ ] Update payment queries
- [ ] Update report generation
- [ ] Test data isolation

### Week 3: Management UI
- [ ] Create super admin panel
- [ ] Create school management pages
- [ ] Create school settings interface
- [ ] Add school selector to main navigation
- [ ] Test admin access controls

### Week 4: Testing & Launch
- [ ] Security testing (cross-school access)
- [ ] Performance testing (query optimization)
- [ ] Create test schools with sample data
- [ ] Documentation for adding new schools
- [ ] Launch to first pilot school

---

## 📝 Next Steps - START HERE

### Immediate Action Items

1. **Decide on approach:**
   - [ ] Shared grades/subjects (RECOMMENDED - simpler)
   - [ ] School-specific grades/subjects (more flexible)

2. **Choose URL structure:**
   - [ ] Query parameter (`?school=KINE`) - easiest
   - [ ] Path-based (`/school/KINE/`) - cleaner

3. **Run database migration:**
   ```bash
   php database/run_all_missing_columns.php  # Ensures latest schema
   ```

4. **Create first SQL migration:**
   - Create `tenant_settings` table
   - Add any missing indexes

5. **Implement school context:**
   - Create `includes/school_context.php`
   - Update authentication to set school

---

## 🎯 Success Criteria

### Week 1 Success
- ✅ Can create new schools in database
- ✅ Admin can select/switch schools
- ✅ School context persists in session

### Week 2 Success  
- ✅ Queries filtered by school
- ✅ Data completely isolated
- ✅ No cross-school data leaks

### Week 3 Success
- ✅ Super admin can manage schools
- ✅ School admins see only their data
- ✅ School settings customizable

### Week 4 Success
- ✅ 2-3 pilot schools running
- ✅ All tests passing
- ✅ Ready to onboard new schools

---

## 💡 Pro Tips

1. **Start Small:** Get 2-3 schools working perfectly before scaling

2. **Test Isolation:** Frequently test that School A cannot see School B data

3. **Use Existing System:** Your `academy_references` table is solid - build on it

4. **Keep It Simple:** Don't over-engineer for 1-20 schools

5. **Document Everything:** Document school onboarding process

6. **Backup First:** Always backup before running migrations

---

## 📞 Support & Resources

- Multi-tenant guide: `docs/multi-tenant-requirements.md`
- Current database: `SKOLO_KINE_COMPLETE_DATABASE.sql`
- Migration tool: `database/run_all_missing_columns.php`

---

**Total Implementation Time: 2-3 weeks**
**Estimated Cost: E2,000-3,000 (if outsourced)**
**DIY with AI: Free + your time**

🎉 **You're 60% there already - let's finish it!**

