<?php
/**
 * Complete Migration Script - Add All Missing Columns
 * This script adds all missing columns found during development
 * Safe to run multiple times - skips existing columns
 */

require_once __DIR__ . '/../config.php';

echo "╔════════════════════════════════════════════════════════════╗\n";
echo "║  Skolo-Kine Database Migration - Add Missing Columns      ║\n";
echo "╚════════════════════════════════════════════════════════════╝\n\n";

$pdo = getDB();
$totalSuccess = 0;
$totalSkipped = 0;
$totalErrors = 0;

// =====================================================
// SECTION 1: Students Table - Moodle Integration
// =====================================================
echo "📋 Section 1: Adding Moodle Integration Columns to Students Table\n";
echo str_repeat("-", 60) . "\n";

$studentColumns = [
    'academy_reference' => "ALTER TABLE `students` ADD COLUMN `academy_reference` VARCHAR(100) DEFAULT NULL",
    'last_payment_date' => "ALTER TABLE `students` ADD COLUMN `last_payment_date` DATETIME NULL",
    'next_payment_due' => "ALTER TABLE `students` ADD COLUMN `next_payment_due` DATETIME NULL",
    'payment_cycle_start' => "ALTER TABLE `students` ADD COLUMN `payment_cycle_start` DATETIME NULL",
    'payment_reminder_sent' => "ALTER TABLE `students` ADD COLUMN `payment_reminder_sent` TINYINT(1) DEFAULT 0",
    'warning_email_sent' => "ALTER TABLE `students` ADD COLUMN `warning_email_sent` TINYINT(1) DEFAULT 0",
    'suspension_warning_sent' => "ALTER TABLE `students` ADD COLUMN `suspension_warning_sent` TINYINT(1) DEFAULT 0",
    'moodle_status' => "ALTER TABLE `students` ADD COLUMN `moodle_status` VARCHAR(20) DEFAULT 'active'",
    'suspended_date' => "ALTER TABLE `students` ADD COLUMN `suspended_date` DATETIME NULL"
];

foreach ($studentColumns as $column => $sql) {
    try {
        $pdo->exec($sql);
        echo "  ✓ Added: $column\n";
        $totalSuccess++;
    } catch (PDOException $e) {
        if (strpos($e->getMessage(), 'Duplicate column name') !== false) {
            echo "  ⚠ Exists: $column\n";
            $totalSkipped++;
        } else {
            echo "  ✗ Error adding $column: " . $e->getMessage() . "\n";
            $totalErrors++;
        }
    }
}

// Add indexes for students table
$studentIndexes = [
    'idx_students_academy_ref' => "ALTER TABLE `students` ADD INDEX `idx_students_academy_ref` (`academy_reference`)",
    'idx_students_moodle_status' => "ALTER TABLE `students` ADD INDEX `idx_students_moodle_status` (`moodle_status`)",
    'idx_students_next_payment_due' => "ALTER TABLE `students` ADD INDEX `idx_students_next_payment_due` (`next_payment_due`)"
];

foreach ($studentIndexes as $index => $sql) {
    try {
        $pdo->exec($sql);
        echo "  ✓ Index: $index\n";
        $totalSuccess++;
    } catch (PDOException $e) {
        if (strpos($e->getMessage(), 'Duplicate key name') !== false) {
            echo "  ⚠ Index exists: $index\n";
            $totalSkipped++;
        } else {
            echo "  ✗ Error creating index $index: " . $e->getMessage() . "\n";
            $totalErrors++;
        }
    }
}

echo "\n";

// =====================================================
// SECTION 2: Payments Table - Payment Method & Admin Earnings
// =====================================================
echo "📋 Section 2: Adding Payment Columns to Payments Table\n";
echo str_repeat("-", 60) . "\n";

// Add payment_method column
try {
    $pdo->exec("ALTER TABLE `payments` ADD COLUMN `payment_method` VARCHAR(20) DEFAULT 'MOMO'");
    echo "  ✓ Added: payment_method\n";
    $totalSuccess++;
    
    // Update existing records
    $pdo->exec("UPDATE `payments` SET `payment_method` = 'MOMO' WHERE `payment_method` IS NULL");
    
} catch (PDOException $e) {
    if (strpos($e->getMessage(), 'Duplicate column name') !== false) {
        echo "  ⚠ Exists: payment_method\n";
        $totalSkipped++;
    } else {
        echo "  ✗ Error: " . $e->getMessage() . "\n";
        $totalErrors++;
    }
}

// Add admin_earnings column
try {
    $pdo->exec("ALTER TABLE `payments` ADD COLUMN `admin_earnings` DECIMAL(10,2) DEFAULT NULL");
    echo "  ✓ Added: admin_earnings\n";
    $totalSuccess++;
} catch (PDOException $e) {
    if (strpos($e->getMessage(), 'Duplicate column name') !== false) {
        echo "  ⚠ Exists: admin_earnings\n";
        $totalSkipped++;
    } else {
        echo "  ✗ Error: " . $e->getMessage() . "\n";
        $totalErrors++;
    }
}

// Add platform_fee column
try {
    $pdo->exec("ALTER TABLE `payments` ADD COLUMN `platform_fee` DECIMAL(10,2) DEFAULT 0.50");
    echo "  ✓ Added: platform_fee\n";
    $totalSuccess++;
} catch (PDOException $e) {
    if (strpos($e->getMessage(), 'Duplicate column name') !== false) {
        echo "  ⚠ Exists: platform_fee\n";
        $totalSkipped++;
    } else {
        echo "  ✗ Error: " . $e->getMessage() . "\n";
        $totalErrors++;
    }
}

// Add academy_reference column
try {
    $pdo->exec("ALTER TABLE `payments` ADD COLUMN `academy_reference` VARCHAR(100) DEFAULT NULL");
    echo "  ✓ Added: academy_reference\n";
    $totalSuccess++;
} catch (PDOException $e) {
    if (strpos($e->getMessage(), 'Duplicate column name') !== false) {
        echo "  ⚠ Exists: academy_reference\n";
        $totalSkipped++;
    } else {
        echo "  ✗ Error: " . $e->getMessage() . "\n";
        $totalErrors++;
    }
}

// Add indexes
$paymentIndexes = [
    'idx_payments_payment_method' => "ALTER TABLE `payments` ADD INDEX `idx_payments_payment_method` (`payment_method`)",
    'idx_payments_academy_ref' => "ALTER TABLE `payments` ADD INDEX `idx_payments_academy_ref` (`academy_reference`)"
];

foreach ($paymentIndexes as $index => $sql) {
    try {
        $pdo->exec($sql);
        echo "  ✓ Index: $index\n";
        $totalSuccess++;
    } catch (PDOException $e) {
        if (strpos($e->getMessage(), 'Duplicate key name') !== false) {
            echo "  ⚠ Index exists: $index\n";
            $totalSkipped++;
        } else {
            echo "  ✗ Error: " . $e->getMessage() . "\n";
            $totalErrors++;
        }
    }
}

echo "\n";

// =====================================================
// SECTION 3: Payment Cycles Table
// =====================================================
echo "📋 Section 3: Creating Payment Cycles Table\n";
echo str_repeat("-", 60) . "\n";

try {
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS `payment_cycles` (
            `id` INT NOT NULL AUTO_INCREMENT,
            `student_id` INT NOT NULL,
            `cycle_start` DATETIME NOT NULL,
            `cycle_end` DATETIME NOT NULL,
            `amount_due` DECIMAL(10,2) NOT NULL,
            `amount_paid` DECIMAL(10,2) DEFAULT 0.00,
            `status` ENUM('active','paid','overdue','suspended') DEFAULT 'active',
            `payment_date` DATETIME NULL,
            `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            KEY `idx_student_cycle` (`student_id`,`cycle_start`),
            KEY `idx_status` (`status`),
            KEY `idx_cycle_end` (`cycle_end`),
            CONSTRAINT `fk_payment_cycles_student` FOREIGN KEY (`student_id`) REFERENCES `students`(`id`) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
    ");
    echo "  ✓ Created: payment_cycles table\n";
    $totalSuccess++;
} catch (PDOException $e) {
    if (strpos($e->getMessage(), 'already exists') !== false) {
        echo "  ⚠ Exists: payment_cycles table\n";
        $totalSkipped++;
    } else {
        echo "  ✗ Error: " . $e->getMessage() . "\n";
        $totalErrors++;
    }
}

echo "\n";

// =====================================================
// SECTION 4: Platform Fees Table
// =====================================================
echo "📋 Section 4: Creating Platform Fees Table\n";
echo str_repeat("-", 60) . "\n";

try {
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS `platform_fees` (
            `id` INT NOT NULL AUTO_INCREMENT,
            `payment_id` INT NOT NULL,
            `admin_id` INT NOT NULL,
            `gross_amount` DECIMAL(10,2) NOT NULL,
            `fee_amount` DECIMAL(10,2) NOT NULL DEFAULT 0.50,
            `net_amount` DECIMAL(10,2) NOT NULL,
            `payment_method` VARCHAR(20) DEFAULT 'MOMO',
            `academy_reference` VARCHAR(100) DEFAULT NULL,
            `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            KEY `idx_payment_id` (`payment_id`),
            KEY `idx_admin_id` (`admin_id`),
            KEY `idx_payment_method` (`payment_method`),
            KEY `idx_academy_reference` (`academy_reference`),
            KEY `idx_created_at` (`created_at`),
            CONSTRAINT `fk_fee_payment` FOREIGN KEY (`payment_id`) REFERENCES `payments`(`id`) ON DELETE CASCADE,
            CONSTRAINT `fk_fee_admin` FOREIGN KEY (`admin_id`) REFERENCES `admins`(`id`) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
    ");
    echo "  ✓ Created: platform_fees table\n";
    $totalSuccess++;
} catch (PDOException $e) {
    if (strpos($e->getMessage(), 'already exists') !== false) {
        echo "  ⚠ Exists: platform_fees table\n";
        $totalSkipped++;
        
        // If table exists, check and add payment_method column
        try {
            $pdo->exec("ALTER TABLE `platform_fees` ADD COLUMN `payment_method` VARCHAR(20) DEFAULT 'MOMO'");
            echo "  ✓ Added payment_method to platform_fees\n";
            $totalSuccess++;
        } catch (PDOException $e2) {
            if (strpos($e2->getMessage(), 'Duplicate column') !== false) {
                echo "  ⚠ platform_fees.payment_method already exists\n";
                $totalSkipped++;
            }
        }
    } else {
        echo "  ✗ Error: " . $e->getMessage() . "\n";
        $totalErrors++;
    }
}

echo "\n";

// =====================================================
// SECTION 5: Academy References Table
// =====================================================
echo "📋 Section 5: Creating Academy References Table\n";
echo str_repeat("-", 60) . "\n";

try {
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS `academy_references` (
            `id` INT NOT NULL AUTO_INCREMENT,
            `admin_id` INT NOT NULL,
            `academy_name` VARCHAR(100) NOT NULL,
            `reference_code` VARCHAR(50) NOT NULL,
            `description` TEXT DEFAULT NULL,
            `is_active` TINYINT(1) DEFAULT 1,
            `total_students` INT DEFAULT 0,
            `total_payments` DECIMAL(10,2) DEFAULT 0.00,
            `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            UNIQUE KEY `reference_code` (`reference_code`),
            KEY `idx_admin_id` (`admin_id`),
            CONSTRAINT `fk_academy_admin` FOREIGN KEY (`admin_id`) REFERENCES `admins`(`id`) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
    ");
    echo "  ✓ Created: academy_references table\n";
    $totalSuccess++;
    
    // Insert default academy reference
    $pdo->exec("
        INSERT IGNORE INTO `academy_references` 
        (`admin_id`, `academy_name`, `reference_code`, `description`, `is_active`) 
        VALUES (1, 'Kine Academy', 'KINE', 'Main Kine Academy', 1)
    ");
    echo "  ✓ Added default academy: Kine Academy\n";
    
} catch (PDOException $e) {
    if (strpos($e->getMessage(), 'already exists') !== false) {
        echo "  ⚠ Exists: academy_references table\n";
        $totalSkipped++;
    } else {
        echo "  ✗ Error: " . $e->getMessage() . "\n";
        $totalErrors++;
    }
}

echo "\n";

// =====================================================
// SECTION 6: Wallet System Tables
// =====================================================
echo "📋 Section 6: Creating Wallet System Tables\n";
echo str_repeat("-", 60) . "\n";

// Admin Wallet Table
try {
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS `admin_wallet` (
            `id` INT NOT NULL AUTO_INCREMENT,
            `admin_id` INT NOT NULL,
            `balance` DECIMAL(10,2) NOT NULL DEFAULT 0.00,
            `total_earned` DECIMAL(10,2) NOT NULL DEFAULT 0.00,
            `total_withdrawn` DECIMAL(10,2) NOT NULL DEFAULT 0.00,
            `pending_withdrawal` DECIMAL(10,2) NOT NULL DEFAULT 0.00,
            `currency` VARCHAR(8) NOT NULL DEFAULT 'SZL',
            `momo_number` VARCHAR(32) DEFAULT NULL,
            `account_verified` TINYINT(1) DEFAULT 0,
            `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            UNIQUE KEY `admin_id` (`admin_id`),
            CONSTRAINT `fk_admin_wallet` FOREIGN KEY (`admin_id`) REFERENCES `admins`(`id`) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
    ");
    echo "  ✓ Created: admin_wallet table\n";
    $totalSuccess++;
    
    // Insert default wallet for admin
    $pdo->exec("INSERT IGNORE INTO `admin_wallet` (`admin_id`, `balance`, `currency`) VALUES (1, 0.00, 'SZL')");
    echo "  ✓ Created default admin wallet\n";
    
} catch (PDOException $e) {
    if (strpos($e->getMessage(), 'already exists') !== false) {
        echo "  ⚠ Exists: admin_wallet table\n";
        $totalSkipped++;
    } else {
        echo "  ✗ Error: " . $e->getMessage() . "\n";
        $totalErrors++;
    }
}

// Wallet Transactions Table
try {
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS `wallet_transactions` (
            `id` INT NOT NULL AUTO_INCREMENT,
            `admin_id` INT NOT NULL,
            `payment_id` INT DEFAULT NULL,
            `withdrawal_id` INT DEFAULT NULL,
            `type` ENUM('CREDIT','DEBIT','WITHDRAWAL','REFUND','PLATFORM_FEE') NOT NULL,
            `amount` DECIMAL(10,2) NOT NULL,
            `balance_before` DECIMAL(10,2) NOT NULL,
            `balance_after` DECIMAL(10,2) NOT NULL,
            `description` VARCHAR(255) DEFAULT NULL,
            `reference` VARCHAR(100) DEFAULT NULL,
            `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            KEY `idx_admin_id` (`admin_id`),
            KEY `idx_payment_id` (`payment_id`),
            KEY `idx_withdrawal_id` (`withdrawal_id`),
            KEY `idx_type` (`type`),
            KEY `idx_created_at` (`created_at`),
            CONSTRAINT `fk_wallet_txn_admin` FOREIGN KEY (`admin_id`) REFERENCES `admins`(`id`) ON DELETE CASCADE,
            CONSTRAINT `fk_wallet_txn_payment` FOREIGN KEY (`payment_id`) REFERENCES `payments`(`id`) ON DELETE SET NULL
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
    ");
    echo "  ✓ Created: wallet_transactions table\n";
    $totalSuccess++;
} catch (PDOException $e) {
    if (strpos($e->getMessage(), 'already exists') !== false) {
        echo "  ⚠ Exists: wallet_transactions table\n";
        $totalSkipped++;
    } else {
        echo "  ✗ Error: " . $e->getMessage() . "\n";
        $totalErrors++;
    }
}

// Withdrawal Requests Table
try {
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS `withdrawal_requests` (
            `id` INT NOT NULL AUTO_INCREMENT,
            `admin_id` INT NOT NULL,
            `amount` DECIMAL(10,2) NOT NULL,
            `momo_number` VARCHAR(32) NOT NULL,
            `status` ENUM('PENDING','PROCESSING','COMPLETED','FAILED','CANCELLED') DEFAULT 'PENDING',
            `momo_reference_id` VARCHAR(64) DEFAULT NULL,
            `momo_response` TEXT DEFAULT NULL,
            `error_message` TEXT DEFAULT NULL,
            `requested_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            `processed_at` TIMESTAMP NULL DEFAULT NULL,
            `completed_at` TIMESTAMP NULL DEFAULT NULL,
            PRIMARY KEY (`id`),
            KEY `idx_admin_id` (`admin_id`),
            KEY `idx_status` (`status`),
            KEY `idx_momo_reference_id` (`momo_reference_id`),
            KEY `idx_requested_at` (`requested_at`),
            CONSTRAINT `fk_withdrawal_admin` FOREIGN KEY (`admin_id`) REFERENCES `admins`(`id`) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
    ");
    echo "  ✓ Created: withdrawal_requests table\n";
    $totalSuccess++;
} catch (PDOException $e) {
    if (strpos($e->getMessage(), 'already exists') !== false) {
        echo "  ⚠ Exists: withdrawal_requests table\n";
        $totalSkipped++;
    } else {
        echo "  ✗ Error: " . $e->getMessage() . "\n";
        $totalErrors++;
    }
}

echo "\n";

// =====================================================
// SUMMARY
// =====================================================
echo "╔════════════════════════════════════════════════════════════╗\n";
echo "║  Migration Summary                                         ║\n";
echo "╚════════════════════════════════════════════════════════════╝\n";
echo "  ✓ Successfully Added: $totalSuccess\n";
echo "  ⚠ Already Exists:     $totalSkipped\n";
echo "  ✗ Errors:             $totalErrors\n";
echo "\n";

if ($totalErrors === 0) {
    echo "✅ All migrations completed successfully!\n";
    echo "   Your dashboard should now work without column errors.\n\n";
} else {
    echo "⚠️ Some migrations failed. Please review the errors above.\n\n";
    exit(1);
}

