-- =====================================================
-- FIX SUBJECTS TABLE - REMOVE NAME UNIQUE CONSTRAINT
-- =====================================================
-- Problem: subjects.name has UNIQUE constraint preventing
--          same subject name across different grades
-- Solution: Remove UNIQUE from 'name', ensure 'code' is UNIQUE
-- Date: 2025
-- =====================================================

-- =====================================================
-- STEP 1: BACKUP EXISTING DATA
-- =====================================================

-- Create backup of subjects table
CREATE TABLE IF NOT EXISTS subjects_backup_20250129 AS SELECT * FROM subjects;

-- Create backup of grade_subject mappings
CREATE TABLE IF NOT EXISTS grade_subject_backup_20250129 AS SELECT * FROM grade_subject;

-- Verify backups
SELECT 
    (SELECT COUNT(*) FROM subjects) as original_subjects,
    (SELECT COUNT(*) FROM subjects_backup_20250129) as backup_subjects,
    (SELECT COUNT(*) FROM grade_subject) as original_mappings,
    (SELECT COUNT(*) FROM grade_subject_backup_20250129) as backup_mappings;

-- =====================================================
-- STEP 2: CHECK CURRENT TABLE STRUCTURE
-- =====================================================

-- Show current structure
DESCRIBE subjects;

-- Show current indexes
SHOW INDEXES FROM subjects;

-- =====================================================
-- STEP 3: REMOVE UNIQUE CONSTRAINT FROM 'name'
-- =====================================================

-- Check if 'name' has UNIQUE constraint and drop it
SET @constraint_exists = (
    SELECT COUNT(*)
    FROM information_schema.TABLE_CONSTRAINTS
    WHERE TABLE_SCHEMA = DATABASE()
    AND TABLE_NAME = 'subjects'
    AND CONSTRAINT_NAME = 'name'
    AND CONSTRAINT_TYPE = 'UNIQUE'
);

-- Drop UNIQUE constraint on 'name' if it exists
SET @drop_sql = IF(
    @constraint_exists > 0,
    'ALTER TABLE subjects DROP INDEX name',
    'SELECT "UNIQUE constraint on name does not exist" as message'
);

PREPARE stmt FROM @drop_sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Alternative method (try both if needed)
-- ALTER TABLE subjects DROP INDEX name;

-- If using KEY instead of INDEX
-- ALTER TABLE subjects DROP KEY name;

-- =====================================================
-- STEP 4: ENSURE 'code' COLUMN IS UNIQUE
-- =====================================================

-- Check if 'code' already has UNIQUE constraint
SET @code_unique_exists = (
    SELECT COUNT(*)
    FROM information_schema.TABLE_CONSTRAINTS
    WHERE TABLE_SCHEMA = DATABASE()
    AND TABLE_NAME = 'subjects'
    AND CONSTRAINT_NAME IN ('unique_code', 'code')
    AND CONSTRAINT_TYPE = 'UNIQUE'
);

-- Add UNIQUE constraint on 'code' if it doesn't exist
SET @add_unique_sql = IF(
    @code_unique_exists = 0,
    'ALTER TABLE subjects ADD UNIQUE KEY unique_code (code)',
    'SELECT "UNIQUE constraint on code already exists" as message'
);

PREPARE stmt FROM @add_unique_sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- =====================================================
-- STEP 5: VERIFY CHANGES
-- =====================================================

-- Show new structure
DESCRIBE subjects;

-- Show new indexes
SHOW INDEXES FROM subjects;

-- Check for duplicate codes (should be 0)
SELECT 
    code, 
    COUNT(*) as count
FROM subjects
GROUP BY code
HAVING count > 1;

-- Check for same subject names with different codes (should show results)
SELECT 
    name, 
    COUNT(DISTINCT code) as different_codes,
    GROUP_CONCAT(code ORDER BY code) as codes
FROM subjects
GROUP BY name
HAVING different_codes > 1
ORDER BY different_codes DESC;

-- =====================================================
-- STEP 6: CLEAN UP OLD DUPLICATE SUBJECTS (IF ANY)
-- =====================================================

-- Check if there are any subjects with NULL or empty codes
SELECT COUNT(*) as subjects_with_invalid_codes
FROM subjects
WHERE code IS NULL OR code = '';

-- Display subjects with invalid codes
SELECT id, name, code
FROM subjects
WHERE code IS NULL OR code = ''
ORDER BY id;

-- If you want to delete subjects with invalid codes (CAREFUL!):
-- DELETE FROM subjects WHERE code IS NULL OR code = '';

-- =====================================================
-- VERIFICATION SUMMARY
-- =====================================================

SELECT '=== VERIFICATION SUMMARY ===' as status;

-- 1. Check constraints
SELECT 
    'Constraints Check' as test,
    CASE 
        WHEN COUNT(*) = 0 THEN '✅ PASS: No UNIQUE constraint on name'
        ELSE '❌ FAIL: UNIQUE constraint still exists on name'
    END as result
FROM information_schema.TABLE_CONSTRAINTS
WHERE TABLE_SCHEMA = DATABASE()
AND TABLE_NAME = 'subjects'
AND CONSTRAINT_NAME = 'name'
AND CONSTRAINT_TYPE = 'UNIQUE';

SELECT 
    'Code Uniqueness' as test,
    CASE 
        WHEN COUNT(*) > 0 THEN '✅ PASS: UNIQUE constraint exists on code'
        ELSE '❌ FAIL: No UNIQUE constraint on code'
    END as result
FROM information_schema.TABLE_CONSTRAINTS
WHERE TABLE_SCHEMA = DATABASE()
AND TABLE_NAME = 'subjects'
AND CONSTRAINT_NAME IN ('unique_code', 'code')
AND CONSTRAINT_TYPE = 'UNIQUE';

-- 2. Check for duplicate codes
SELECT 
    'Duplicate Codes' as test,
    CASE 
        WHEN COUNT(*) = 0 THEN '✅ PASS: No duplicate codes'
        ELSE CONCAT('❌ FAIL: ', COUNT(*), ' duplicate codes found')
    END as result
FROM (
    SELECT code
    FROM subjects
    GROUP BY code
    HAVING COUNT(*) > 1
) as duplicates;

-- 3. Data integrity check
SELECT 
    'Data Integrity' as test,
    CASE 
        WHEN backup_count = current_count THEN '✅ PASS: No data lost'
        ELSE CONCAT('⚠️ WARNING: Subjects count changed from ', backup_count, ' to ', current_count)
    END as result
FROM (
    SELECT 
        (SELECT COUNT(*) FROM subjects_backup_20250129) as backup_count,
        (SELECT COUNT(*) FROM subjects) as current_count
) as counts;

-- =====================================================
-- COMPLETE!
-- =====================================================

SELECT '=== MIGRATION COMPLETE ===' as status;
SELECT 'You can now re-import ESWATINI_CURRICULUM_SUBJECTS.sql' as next_step;
SELECT 'Same subject names (e.g., Mathematics) can now exist with different codes' as note;

-- =====================================================
-- ROLLBACK SCRIPT (IF NEEDED)
-- =====================================================
-- To rollback these changes, run:
-- 
-- -- Restore subjects from backup
-- TRUNCATE TABLE subjects;
-- INSERT INTO subjects SELECT * FROM subjects_backup_20250129;
-- 
-- -- Restore grade_subject mappings
-- TRUNCATE TABLE grade_subject;
-- INSERT INTO grade_subject SELECT * FROM grade_subject_backup_20250129;
-- 
-- -- Re-add UNIQUE constraint on 'name' (if you really want it back)
-- ALTER TABLE subjects ADD UNIQUE KEY name (name);
-- 
-- =====================================================

