-- =====================================================
-- FIX AND RECREATE TENANT FILES TABLE
-- =====================================================
-- This script drops old tables and recreates with correct structure
-- Run this to fix missing columns error
-- =====================================================

-- =====================================================
-- STEP 1: BACKUP EXISTING DATA (IF ANY)
-- =====================================================

CREATE TABLE IF NOT EXISTS tenant_files_backup_old AS 
SELECT * FROM tenant_files 
WHERE EXISTS (
    SELECT 1 FROM information_schema.TABLES 
    WHERE TABLE_SCHEMA = DATABASE() 
    AND TABLE_NAME = 'tenant_files'
);

SELECT 'Backup created (if table existed)' AS status;

-- =====================================================
-- STEP 2: DROP OLD TABLES IN CORRECT ORDER
-- =====================================================

-- Temporarily disable foreign key checks
SET FOREIGN_KEY_CHECKS = 0;

-- Drop views first
DROP VIEW IF EXISTS v_active_tenant_files;
DROP VIEW IF EXISTS v_tenant_file_stats;

-- Drop triggers
DROP TRIGGER IF EXISTS before_tenant_file_access_insert;

-- Drop all related tables (in any order with FK checks off)
DROP TABLE IF EXISTS tenant_file_versions;
DROP TABLE IF EXISTS tenant_file_access_log;
DROP TABLE IF EXISTS file_shares;
DROP TABLE IF EXISTS tenant_files;

-- Re-enable foreign key checks
SET FOREIGN_KEY_CHECKS = 1;

SELECT 'Old tables dropped' AS status;

-- =====================================================
-- STEP 3: CREATE TENANT FILES TABLE (CORRECT VERSION)
-- =====================================================

CREATE TABLE tenant_files (
    -- Primary Key
    id INT AUTO_INCREMENT PRIMARY KEY,
    
    -- Tenant Information
    tenant_id CHAR(36) NOT NULL COMMENT 'Tenant/School identifier',
    
    -- File Classification
    file_category ENUM(
        'document',
        'photo',
        'receipt',
        'report',
        'backup',
        'other'
    ) NOT NULL DEFAULT 'other' COMMENT 'File category',
    
    -- File Information
    original_filename VARCHAR(255) NOT NULL COMMENT 'Original filename',
    stored_filename VARCHAR(255) NOT NULL UNIQUE COMMENT 'Stored filename',
    file_path VARCHAR(500) NOT NULL COMMENT 'Relative path',
    file_size BIGINT NOT NULL COMMENT 'Size in bytes',
    mime_type VARCHAR(100) NULL COMMENT 'MIME type',
    file_hash VARCHAR(64) NULL COMMENT 'SHA-256 hash',
    
    -- User Information
    uploaded_by INT NULL COMMENT 'User who uploaded',
    
    -- Relationship (polymorphic)
    related_id INT NULL COMMENT 'Related entity ID',
    related_type VARCHAR(50) NULL COMMENT 'Related entity type',
    
    -- Metadata
    description TEXT NULL COMMENT 'File description',
    tags VARCHAR(255) NULL COMMENT 'Search tags',
    
    -- Access Control
    is_public TINYINT(1) DEFAULT 0 COMMENT 'Public access',
    access_count INT DEFAULT 0 COMMENT 'Access count',
    last_accessed_at DATETIME NULL COMMENT 'Last access',
    
    -- Timestamps
    upload_date DATETIME DEFAULT CURRENT_TIMESTAMP COMMENT 'Upload time',
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    -- Soft Delete
    is_deleted TINYINT(1) DEFAULT 0 COMMENT 'Deleted flag',
    deleted_at DATETIME NULL COMMENT 'Delete time',
    deleted_by INT NULL COMMENT 'User who deleted',
    
    -- Constraints
    CHECK (file_size >= 0),
    CHECK (access_count >= 0)
    
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- STEP 4: CREATE INDEXES
-- =====================================================

CREATE INDEX idx_tenant_id ON tenant_files(tenant_id);
CREATE INDEX idx_tenant_category ON tenant_files(tenant_id, file_category);
CREATE INDEX idx_stored_filename ON tenant_files(stored_filename);
CREATE INDEX idx_related ON tenant_files(related_type, related_id);
CREATE INDEX idx_uploaded_by ON tenant_files(uploaded_by);
CREATE INDEX idx_upload_date ON tenant_files(upload_date);
CREATE INDEX idx_is_deleted ON tenant_files(is_deleted);
CREATE INDEX idx_tenant_not_deleted ON tenant_files(tenant_id, is_deleted);

-- =====================================================
-- STEP 5: CREATE RELATED TABLES
-- =====================================================

CREATE TABLE tenant_file_access_log (
    id BIGINT AUTO_INCREMENT PRIMARY KEY,
    file_id INT NOT NULL,
    tenant_id CHAR(36) NOT NULL,
    user_id INT NULL,
    access_type ENUM('view', 'download', 'preview') DEFAULT 'view',
    ip_address VARCHAR(45) NULL,
    user_agent VARCHAR(255) NULL,
    accessed_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    
    INDEX idx_file_id (file_id),
    INDEX idx_tenant_id (tenant_id),
    INDEX idx_user_id (user_id),
    INDEX idx_accessed_at (accessed_at),
    
    FOREIGN KEY (file_id) REFERENCES tenant_files(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE tenant_file_versions (
    id BIGINT AUTO_INCREMENT PRIMARY KEY,
    file_id INT NOT NULL,
    tenant_id CHAR(36) NOT NULL,
    version_number INT NOT NULL,
    stored_filename VARCHAR(255) NOT NULL,
    file_path VARCHAR(500) NOT NULL,
    file_size BIGINT NOT NULL,
    uploaded_by INT NULL,
    version_notes TEXT NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    
    INDEX idx_file_id (file_id),
    INDEX idx_tenant_id (tenant_id),
    
    FOREIGN KEY (file_id) REFERENCES tenant_files(id) ON DELETE CASCADE,
    UNIQUE KEY unique_file_version (file_id, version_number)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- STEP 6: CREATE VIEWS
-- =====================================================

CREATE OR REPLACE VIEW v_active_tenant_files AS
SELECT 
    id, tenant_id, file_category, original_filename, stored_filename,
    file_path, file_size, ROUND(file_size / 1024 / 1024, 2) AS size_mb,
    mime_type, uploaded_by, related_id, related_type,
    description, tags, is_public, access_count, upload_date, last_accessed_at
FROM tenant_files
WHERE is_deleted = 0;

CREATE OR REPLACE VIEW v_tenant_file_stats AS
SELECT 
    tenant_id,
    COUNT(*) AS total_files,
    SUM(file_size) AS total_bytes,
    ROUND(SUM(file_size) / 1024 / 1024, 2) AS total_mb,
    ROUND(SUM(file_size) / 1024 / 1024 / 1024, 2) AS total_gb,
    MAX(upload_date) AS last_upload,
    SUM(CASE WHEN file_category = 'document' THEN 1 ELSE 0 END) AS documents,
    SUM(CASE WHEN file_category = 'photo' THEN 1 ELSE 0 END) AS photos,
    SUM(CASE WHEN file_category = 'receipt' THEN 1 ELSE 0 END) AS receipts,
    SUM(CASE WHEN file_category = 'report' THEN 1 ELSE 0 END) AS reports
FROM tenant_files
WHERE is_deleted = 0
GROUP BY tenant_id;

-- =====================================================
-- STEP 7: CREATE TRIGGER
-- =====================================================

DELIMITER $$

CREATE TRIGGER before_tenant_file_access_insert
AFTER INSERT ON tenant_file_access_log
FOR EACH ROW
BEGIN
    UPDATE tenant_files 
    SET 
        access_count = access_count + 1,
        last_accessed_at = NEW.accessed_at
    WHERE id = NEW.file_id;
END$$

DELIMITER ;

-- =====================================================
-- VERIFICATION
-- =====================================================

SELECT 'Tables recreated successfully!' AS status;

DESCRIBE tenant_files;

SELECT 
    'tenant_files' AS table_name,
    COUNT(*) AS column_count
FROM information_schema.COLUMNS
WHERE TABLE_SCHEMA = DATABASE()
AND TABLE_NAME = 'tenant_files';

-- Check all required columns exist
SELECT 
    COLUMN_NAME,
    DATA_TYPE,
    IS_NULLABLE,
    COLUMN_DEFAULT
FROM information_schema.COLUMNS
WHERE TABLE_SCHEMA = DATABASE()
AND TABLE_NAME = 'tenant_files'
AND COLUMN_NAME IN (
    'id', 'tenant_id', 'file_category', 'original_filename',
    'stored_filename', 'file_path', 'file_size', 'mime_type',
    'uploaded_by', 'related_id', 'related_type', 'upload_date',
    'is_deleted', 'deleted_at'
)
ORDER BY ORDINAL_POSITION;

-- =====================================================
-- COMPLETE!
-- =====================================================

SELECT '=== FIX COMPLETE ===' AS status;
SELECT 'All columns are now present' AS message;
SELECT 'Test the upload system again' AS next_step;

