-- =====================================================
-- SCHOOL GRADES CONFIGURATION
-- =====================================================
-- Allows each school to select which grades they offer
-- Multi-tenant: Each school can offer different grades
-- =====================================================

DROP TABLE IF EXISTS school_grades;

CREATE TABLE IF NOT EXISTS school_grades (
    id INT AUTO_INCREMENT PRIMARY KEY,
    
    -- Which school
    academy_reference VARCHAR(50) NOT NULL,
    
    -- Which grade
    grade_id INT NOT NULL,
    
    -- Settings
    is_active TINYINT(1) DEFAULT 1,
    display_order INT DEFAULT 0 COMMENT 'Custom sort order for this school',
    
    -- Metadata
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    created_by INT NULL COMMENT 'Admin who enabled this grade',
    
    -- Indexes
    INDEX idx_academy (academy_reference),
    INDEX idx_grade (grade_id),
    INDEX idx_active (is_active),
    INDEX idx_order (display_order),
    
    -- Unique: one entry per school per grade
    UNIQUE KEY unique_school_grade (academy_reference, grade_id),
    
    -- Foreign keys
    FOREIGN KEY (grade_id) REFERENCES grades(id) ON DELETE CASCADE
    -- FOREIGN KEY (academy_reference) REFERENCES academy_references(reference_code) ON DELETE CASCADE
    
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
COMMENT='Stores which grades each school offers';

-- =====================================================
-- INDEXES FOR PERFORMANCE
-- =====================================================

-- For quick lookups
CREATE INDEX idx_lookup ON school_grades(academy_reference, is_active);

-- =====================================================
-- DEFAULT DATA (Optional)
-- =====================================================
-- Uncomment to auto-enable all grades for existing schools

/*
INSERT INTO school_grades (academy_reference, grade_id, is_active)
SELECT ar.reference_code, g.id, 1
FROM academy_references ar
CROSS JOIN grades g
WHERE ar.is_active = 1
ON DUPLICATE KEY UPDATE is_active = 1;
*/

-- =====================================================
-- VERIFICATION QUERIES
-- =====================================================

-- Check table structure
DESCRIBE school_grades;

-- Count grades per school
-- SELECT 
--     academy_reference,
--     COUNT(*) as total_grades,
--     SUM(is_active) as active_grades
-- FROM school_grades
-- GROUP BY academy_reference;

-- List active grades for a specific school
-- SELECT 
--     sg.academy_reference,
--     g.name as grade_name,
--     sg.is_active,
--     sg.display_order
-- FROM school_grades sg
-- JOIN grades g ON sg.grade_id = g.id
-- WHERE sg.academy_reference = 'YOUR_SCHOOL'
-- AND sg.is_active = 1
-- ORDER BY sg.display_order, g.name;

-- =====================================================
-- ROLLBACK SCRIPT
-- =====================================================

-- DROP TABLE IF EXISTS school_grades;

-- =====================================================
-- COMPLETE!
-- =====================================================

