-- =====================================================
-- PAYMENT REMINDER SYSTEM - DATABASE VIEWS
-- Run this AFTER creating the main tables
-- =====================================================

-- Payment Reminder System Views
DROP VIEW IF EXISTS v_students_need_reminder;
DROP VIEW IF EXISTS v_students_need_warning;
DROP VIEW IF EXISTS v_students_to_suspend;

-- =====================================================
-- View: Students requiring payment reminders
-- =====================================================
CREATE VIEW v_students_need_reminder AS
SELECT 
    s.id AS student_id,
    s.full_name,
    s.email,
    s.phone,
    s.academy_reference,
    sps.last_payment_date,
    sps.next_due_date,
    sps.total_amount_due,
    sps.balance,
    sps.reminder_sent_at,
    sps.warning_sent_at,
    tps.reminder_days_before,
    tps.warning_days_after_reminder,
    DATEDIFF(CURDATE(), sps.last_payment_date) AS days_since_payment,
    DATEDIFF(sps.next_due_date, CURDATE()) AS days_until_due
FROM students s
INNER JOIN student_payment_schedules sps ON s.id = sps.student_id
INNER JOIN tenant_payment_settings tps ON s.academy_reference COLLATE utf8mb4_general_ci = tps.academy_reference COLLATE utf8mb4_general_ci
WHERE sps.payment_status IN ('pending', 'overdue')
  AND tps.auto_reminder_enabled = 1
  AND DATEDIFF(CURDATE(), sps.last_payment_date) >= tps.reminder_days_before
  AND (sps.reminder_sent_at IS NULL OR DATE(sps.reminder_sent_at) < CURDATE());

-- =====================================================
-- View: Students requiring warnings
-- =====================================================
CREATE VIEW v_students_need_warning AS
SELECT 
    s.id AS student_id,
    s.full_name,
    s.email,
    s.phone,
    s.academy_reference,
    sps.last_payment_date,
    sps.next_due_date,
    sps.reminder_sent_at,
    sps.warning_sent_at,
    sps.total_amount_due,
    tps.warning_days_after_reminder,
    DATEDIFF(CURDATE(), sps.reminder_sent_at) AS days_since_reminder
FROM students s
INNER JOIN student_payment_schedules sps ON s.id = sps.student_id
INNER JOIN tenant_payment_settings tps ON s.academy_reference COLLATE utf8mb4_general_ci = tps.academy_reference COLLATE utf8mb4_general_ci
WHERE sps.payment_status = 'overdue'
  AND tps.auto_warning_enabled = 1
  AND sps.reminder_sent_at IS NOT NULL
  AND DATEDIFF(CURDATE(), sps.reminder_sent_at) >= tps.warning_days_after_reminder
  AND (sps.warning_sent_at IS NULL OR DATE(sps.warning_sent_at) < CURDATE());

-- =====================================================
-- View: Students to be suspended
-- =====================================================
CREATE VIEW v_students_to_suspend AS
SELECT 
    s.id AS student_id,
    s.full_name,
    s.email,
    s.academy_reference,
    sps.last_payment_date,
    sps.suspension_date,
    sps.total_amount_due,
    sps.balance,
    DATEDIFF(CURDATE(), sps.last_payment_date) AS days_overdue
FROM students s
INNER JOIN student_payment_schedules sps ON s.id = sps.student_id
INNER JOIN tenant_payment_settings tps ON s.academy_reference COLLATE utf8mb4_general_ci = tps.academy_reference COLLATE utf8mb4_general_ci
WHERE sps.payment_status = 'overdue'
  AND tps.auto_suspend_enabled = 1
  AND sps.suspended_at IS NULL
  AND sps.suspension_date <= CURDATE();

-- =====================================================
-- COMPLETION MESSAGE
-- =====================================================
SELECT 'Payment Reminder System Views Created Successfully!' AS Status;

