-- =====================================================
-- WALLET & WITHDRAWAL SYSTEM FOR SKOLO-KINE
-- =====================================================
-- This creates the wallet system for admin earnings tracking
-- and automated withdrawals via MTN MoMo Disbursement API
-- =====================================================

-- Admin wallet to track balances
CREATE TABLE IF NOT EXISTS `admin_wallet` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `admin_id` INT NOT NULL UNIQUE,
    `balance` DECIMAL(10,2) NOT NULL DEFAULT 0.00,
    `total_earned` DECIMAL(10,2) NOT NULL DEFAULT 0.00,
    `total_withdrawn` DECIMAL(10,2) NOT NULL DEFAULT 0.00,
    `pending_withdrawal` DECIMAL(10,2) NOT NULL DEFAULT 0.00,
    `currency` VARCHAR(8) NOT NULL DEFAULT 'SZL',
    `momo_number` VARCHAR(32) NULL,
    `account_verified` BOOLEAN DEFAULT FALSE,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    CONSTRAINT `fk_admin_wallet` FOREIGN KEY (`admin_id`) REFERENCES `admins`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- Wallet transactions (deposits from payments)
CREATE TABLE IF NOT EXISTS `wallet_transactions` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `admin_id` INT NOT NULL,
    `payment_id` INT NULL,
    `withdrawal_id` INT NULL,
    `type` ENUM('CREDIT', 'DEBIT', 'WITHDRAWAL', 'REFUND', 'PLATFORM_FEE') NOT NULL,
    `amount` DECIMAL(10,2) NOT NULL,
    `balance_before` DECIMAL(10,2) NOT NULL,
    `balance_after` DECIMAL(10,2) NOT NULL,
    `description` VARCHAR(255) NULL,
    `reference` VARCHAR(100) NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    KEY `idx_admin_id` (`admin_id`),
    KEY `idx_payment_id` (`payment_id`),
    KEY `idx_withdrawal_id` (`withdrawal_id`),
    KEY `idx_type` (`type`),
    KEY `idx_created_at` (`created_at`),
    CONSTRAINT `fk_wallet_txn_admin` FOREIGN KEY (`admin_id`) REFERENCES `admins`(`id`) ON DELETE CASCADE,
    CONSTRAINT `fk_wallet_txn_payment` FOREIGN KEY (`payment_id`) REFERENCES `payments`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- Withdrawal requests
CREATE TABLE IF NOT EXISTS `withdrawal_requests` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `admin_id` INT NOT NULL,
    `amount` DECIMAL(10,2) NOT NULL,
    `momo_number` VARCHAR(32) NOT NULL,
    `status` ENUM('PENDING', 'PROCESSING', 'COMPLETED', 'FAILED', 'CANCELLED') DEFAULT 'PENDING',
    `momo_reference_id` VARCHAR(64) NULL,
    `momo_response` TEXT NULL,
    `error_message` TEXT NULL,
    `requested_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `processed_at` TIMESTAMP NULL,
    `completed_at` TIMESTAMP NULL,
    KEY `idx_admin_id` (`admin_id`),
    KEY `idx_status` (`status`),
    KEY `idx_momo_reference_id` (`momo_reference_id`),
    KEY `idx_requested_at` (`requested_at`),
    CONSTRAINT `fk_withdrawal_admin` FOREIGN KEY (`admin_id`) REFERENCES `admins`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- Platform fees tracking (your earnings)
CREATE TABLE IF NOT EXISTS `platform_fees` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `payment_id` INT NOT NULL,
    `admin_id` INT NOT NULL,
    `gross_amount` DECIMAL(10,2) NOT NULL,
    `fee_amount` DECIMAL(10,2) NOT NULL DEFAULT 0.50,
    `net_amount` DECIMAL(10,2) NOT NULL,
    `academy_reference` VARCHAR(100) NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    KEY `idx_payment_id` (`payment_id`),
    KEY `idx_admin_id` (`admin_id`),
    KEY `idx_academy_reference` (`academy_reference`),
    KEY `idx_created_at` (`created_at`),
    CONSTRAINT `fk_fee_payment` FOREIGN KEY (`payment_id`) REFERENCES `payments`(`id`) ON DELETE CASCADE,
    CONSTRAINT `fk_fee_admin` FOREIGN KEY (`admin_id`) REFERENCES `admins`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- Academy/School references for tracking payment sources
CREATE TABLE IF NOT EXISTS `academy_references` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `admin_id` INT NOT NULL,
    `academy_name` VARCHAR(100) NOT NULL,
    `reference_code` VARCHAR(50) NOT NULL UNIQUE,
    `description` TEXT NULL,
    `is_active` BOOLEAN DEFAULT TRUE,
    `total_students` INT DEFAULT 0,
    `total_payments` DECIMAL(10,2) DEFAULT 0.00,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    KEY `idx_admin_id` (`admin_id`),
    KEY `idx_reference_code` (`reference_code`),
    CONSTRAINT `fk_academy_admin` FOREIGN KEY (`admin_id`) REFERENCES `admins`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- Add columns to existing tables
ALTER TABLE `admins` 
ADD COLUMN IF NOT EXISTS `momo_number` VARCHAR(32) NULL,
ADD COLUMN IF NOT EXISTS `bank_account` VARCHAR(50) NULL,
ADD COLUMN IF NOT EXISTS `account_verified` BOOLEAN DEFAULT FALSE,
ADD COLUMN IF NOT EXISTS `academy_reference` VARCHAR(50) NULL;

ALTER TABLE `payments`
ADD COLUMN IF NOT EXISTS `academy_reference` VARCHAR(100) NULL,
ADD COLUMN IF NOT EXISTS `platform_fee` DECIMAL(10,2) DEFAULT 0.50,
ADD COLUMN IF NOT EXISTS `admin_earnings` DECIMAL(10,2) NULL;

ALTER TABLE `students`
ADD COLUMN IF NOT EXISTS `academy_reference` VARCHAR(100) NULL;

-- Create indexes for performance
CREATE INDEX IF NOT EXISTS `idx_payments_academy_ref` ON `payments`(`academy_reference`);
CREATE INDEX IF NOT EXISTS `idx_students_academy_ref` ON `students`(`academy_reference`);

-- Insert default academy reference for Kine Academy
INSERT INTO `academy_references` (`admin_id`, `academy_name`, `reference_code`, `description`, `is_active`)
VALUES (1, 'Kine Academy', 'KINE', 'Main Kine Academy', TRUE)
ON DUPLICATE KEY UPDATE `updated_at` = CURRENT_TIMESTAMP;

-- Insert default platform admin wallet
INSERT INTO `admin_wallet` (`admin_id`, `balance`, `currency`)
VALUES (1, 0.00, 'SZL')
ON DUPLICATE KEY UPDATE `updated_at` = CURRENT_TIMESTAMP;

-- =====================================================
-- VIEWS FOR REPORTING
-- =====================================================

-- Admin wallet summary view
CREATE OR REPLACE VIEW `v_admin_wallet_summary` AS
SELECT 
    aw.admin_id,
    a.username,
    aw.balance,
    aw.total_earned,
    aw.total_withdrawn,
    aw.pending_withdrawal,
    aw.momo_number,
    aw.account_verified,
    COUNT(DISTINCT p.id) as total_transactions,
    SUM(pf.fee_amount) as total_platform_fees,
    aw.updated_at
FROM admin_wallet aw
LEFT JOIN admins a ON aw.admin_id = a.id
LEFT JOIN payments p ON p.status = 'SUCCESSFUL'
LEFT JOIN platform_fees pf ON pf.admin_id = aw.admin_id
GROUP BY aw.admin_id;

-- Academy performance view
CREATE OR REPLACE VIEW `v_academy_performance` AS
SELECT 
    ar.id,
    ar.admin_id,
    ar.academy_name,
    ar.reference_code,
    COUNT(DISTINCT s.id) as total_students,
    COUNT(DISTINCT p.id) as total_payments,
    SUM(p.amount) as total_revenue,
    SUM(pf.fee_amount) as total_platform_fees,
    SUM(pf.net_amount) as total_admin_earnings,
    ar.created_at
FROM academy_references ar
LEFT JOIN students s ON s.academy_reference = ar.reference_code
LEFT JOIN payments p ON p.academy_reference = ar.reference_code AND p.status = 'SUCCESSFUL'
LEFT JOIN platform_fees pf ON pf.academy_reference = ar.reference_code
WHERE ar.is_active = TRUE
GROUP BY ar.id;

-- =====================================================
-- STORED PROCEDURES
-- =====================================================

DELIMITER //

-- Procedure to credit admin wallet when payment is successful
CREATE PROCEDURE IF NOT EXISTS `sp_credit_admin_wallet`(
    IN p_payment_id INT,
    IN p_admin_id INT,
    IN p_academy_reference VARCHAR(100)
)
BEGIN
    DECLARE v_amount DECIMAL(10,2);
    DECLARE v_platform_fee DECIMAL(10,2) DEFAULT 0.50;
    DECLARE v_net_amount DECIMAL(10,2);
    DECLARE v_balance_before DECIMAL(10,2);
    DECLARE v_balance_after DECIMAL(10,2);
    
    -- Get payment amount
    SELECT amount INTO v_amount FROM payments WHERE id = p_payment_id;
    
    -- Calculate net amount (payment - platform fee)
    SET v_net_amount = v_amount - v_platform_fee;
    
    -- Get current balance
    SELECT COALESCE(balance, 0) INTO v_balance_before 
    FROM admin_wallet WHERE admin_id = p_admin_id;
    
    SET v_balance_after = v_balance_before + v_net_amount;
    
    -- Update wallet
    INSERT INTO admin_wallet (admin_id, balance, total_earned, currency)
    VALUES (p_admin_id, v_net_amount, v_net_amount, 'SZL')
    ON DUPLICATE KEY UPDATE
        balance = balance + v_net_amount,
        total_earned = total_earned + v_net_amount,
        updated_at = NOW();
    
    -- Record platform fee
    INSERT INTO platform_fees (payment_id, admin_id, gross_amount, fee_amount, net_amount, academy_reference)
    VALUES (p_payment_id, p_admin_id, v_amount, v_platform_fee, v_net_amount, p_academy_reference);
    
    -- Record wallet transaction
    INSERT INTO wallet_transactions 
    (admin_id, payment_id, type, amount, balance_before, balance_after, description, reference)
    VALUES 
    (p_admin_id, p_payment_id, 'CREDIT', v_net_amount, v_balance_before, v_balance_after, 
     CONCAT('Payment from student (', p_academy_reference, ')'), p_academy_reference);
    
    -- Update payment record
    UPDATE payments 
    SET platform_fee = v_platform_fee,
        admin_earnings = v_net_amount,
        updated_at = NOW()
    WHERE id = p_payment_id;
    
    -- Update academy reference stats
    UPDATE academy_references 
    SET total_students = total_students + 1,
        total_payments = total_payments + v_amount,
        updated_at = NOW()
    WHERE reference_code = p_academy_reference;
END //

-- Procedure to process withdrawal refund if failed
CREATE PROCEDURE IF NOT EXISTS `sp_refund_withdrawal`(
    IN p_withdrawal_id INT
)
BEGIN
    DECLARE v_admin_id INT;
    DECLARE v_amount DECIMAL(10,2);
    DECLARE v_balance_before DECIMAL(10,2);
    DECLARE v_balance_after DECIMAL(10,2);
    
    -- Get withdrawal details
    SELECT admin_id, amount INTO v_admin_id, v_amount
    FROM withdrawal_requests WHERE id = p_withdrawal_id;
    
    -- Get current balance
    SELECT balance INTO v_balance_before 
    FROM admin_wallet WHERE admin_id = v_admin_id;
    
    SET v_balance_after = v_balance_before + v_amount;
    
    -- Refund to wallet
    UPDATE admin_wallet 
    SET balance = balance + v_amount,
        total_withdrawn = total_withdrawn - v_amount,
        pending_withdrawal = pending_withdrawal - v_amount,
        updated_at = NOW()
    WHERE admin_id = v_admin_id;
    
    -- Record refund transaction
    INSERT INTO wallet_transactions 
    (admin_id, withdrawal_id, type, amount, balance_before, balance_after, description)
    VALUES 
    (v_admin_id, p_withdrawal_id, 'REFUND', v_amount, v_balance_before, v_balance_after, 
     'Withdrawal refund - transfer failed');
END //

DELIMITER ;

-- =====================================================
-- INITIAL DATA
-- =====================================================

-- Create default admin wallet if not exists
INSERT IGNORE INTO admin_wallet (admin_id, balance, currency, momo_number)
SELECT id, 0.00, 'SZL', NULL FROM admins WHERE id = 1;

COMMIT;

-- =====================================================
-- END OF WALLET SYSTEM SCHEMA
-- =====================================================
