-- =====================================================
-- TENANT FILE SYSTEM - DATABASE TABLES (FIXED)
-- =====================================================
-- Tables for logging file operations and access
-- Fixed version: Foreign keys are optional
-- =====================================================

-- =====================================================
-- 1. FILE OPERATIONS LOG
-- =====================================================
-- Logs all file operations (upload, delete, etc.)

CREATE TABLE IF NOT EXISTS file_operations_log (
    id BIGINT AUTO_INCREMENT PRIMARY KEY,
    tenant_id VARCHAR(50) NOT NULL,
    user_id INT NULL,
    operation VARCHAR(50) NOT NULL,
    message TEXT NOT NULL,
    metadata JSON NULL,
    ip_address VARCHAR(45) NULL,
    user_agent VARCHAR(255) NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    INDEX idx_tenant (tenant_id),
    INDEX idx_user (user_id),
    INDEX idx_operation (operation),
    INDEX idx_created (created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- 2. FILE ACCESS LOG
-- =====================================================
-- Logs when users access/download files
-- Note: Foreign key to users table removed for compatibility

CREATE TABLE IF NOT EXISTS file_access_log (
    id BIGINT AUTO_INCREMENT PRIMARY KEY,
    tenant_id VARCHAR(50) NOT NULL,
    user_id INT NOT NULL,
    file_type VARCHAR(50) NOT NULL,
    filename VARCHAR(255) NOT NULL,
    ip_address VARCHAR(45) NULL,
    user_agent VARCHAR(255) NULL,
    accessed_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    INDEX idx_tenant (tenant_id),
    INDEX idx_user (user_id),
    INDEX idx_file_type (file_type),
    INDEX idx_accessed (accessed_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- 3. TENANT FILES REGISTRY (OPTIONAL)
-- =====================================================
-- Optional table to track all files in database
-- Useful for advanced features like file versioning, sharing, etc.

CREATE TABLE IF NOT EXISTS tenant_files (
    id BIGINT AUTO_INCREMENT PRIMARY KEY,
    tenant_id VARCHAR(50) NOT NULL,
    file_type VARCHAR(50) NOT NULL,
    filename VARCHAR(255) NOT NULL,
    original_name VARCHAR(255) NOT NULL,
    file_path VARCHAR(500) NOT NULL,
    relative_path VARCHAR(500) NOT NULL,
    file_size BIGINT NOT NULL,
    mime_type VARCHAR(100) NULL,
    extension VARCHAR(10) NULL,
    uploaded_by INT NULL,
    description TEXT NULL,
    is_public BOOLEAN DEFAULT FALSE,
    is_archived BOOLEAN DEFAULT FALSE,
    download_count INT DEFAULT 0,
    last_accessed_at TIMESTAMP NULL,
    uploaded_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    deleted_at TIMESTAMP NULL,
    
    INDEX idx_tenant (tenant_id),
    INDEX idx_type (file_type),
    INDEX idx_uploaded_by (uploaded_by),
    INDEX idx_public (is_public),
    INDEX idx_archived (is_archived),
    INDEX idx_uploaded (uploaded_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- 4. FILE SHARES (OPTIONAL)
-- =====================================================
-- Optional table for sharing files between users

CREATE TABLE IF NOT EXISTS file_shares (
    id BIGINT AUTO_INCREMENT PRIMARY KEY,
    file_id BIGINT NOT NULL,
    tenant_id VARCHAR(50) NOT NULL,
    shared_by INT NOT NULL,
    shared_with INT NULL,
    share_token VARCHAR(64) NULL UNIQUE,
    permissions ENUM('view', 'download', 'edit') DEFAULT 'view',
    expires_at TIMESTAMP NULL,
    access_count INT DEFAULT 0,
    max_access_count INT NULL,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    last_accessed_at TIMESTAMP NULL,
    
    INDEX idx_file (file_id),
    INDEX idx_tenant (tenant_id),
    INDEX idx_token (share_token),
    INDEX idx_shared_with (shared_with),
    INDEX idx_active (is_active),
    
    FOREIGN KEY (file_id) REFERENCES tenant_files(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- 5. TENANT STORAGE QUOTAS
-- =====================================================
-- Track storage quotas and usage per tenant

CREATE TABLE IF NOT EXISTS tenant_storage_quotas (
    id INT AUTO_INCREMENT PRIMARY KEY,
    tenant_id VARCHAR(50) NOT NULL UNIQUE,
    storage_quota BIGINT DEFAULT 1073741824, -- 1GB default
    storage_used BIGINT DEFAULT 0,
    file_count INT DEFAULT 0,
    last_calculated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    quota_warning_sent BOOLEAN DEFAULT FALSE,
    quota_exceeded BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    INDEX idx_tenant (tenant_id),
    INDEX idx_quota_exceeded (quota_exceeded)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- INITIAL DATA
-- =====================================================

-- Storage quotas will be created automatically when tenant directories
-- are created via TenantFileSystem::createTenantDirectory()

-- If you have existing tenants and want to populate quotas, run this manually:
-- INSERT IGNORE INTO tenant_storage_quotas (tenant_id, storage_quota)
-- SELECT DISTINCT academy_reference, 1073741824
-- FROM tenants
-- WHERE academy_reference IS NOT NULL AND academy_reference != '';

SELECT 'Tables created successfully. Storage quotas will be created automatically.' as status;

-- =====================================================
-- VERIFICATION
-- =====================================================

-- Check tables created
SELECT 'Tables created successfully!' as status;

SELECT 
    TABLE_NAME,
    ENGINE,
    TABLE_ROWS,
    CREATE_TIME
FROM information_schema.TABLES
WHERE TABLE_SCHEMA = DATABASE()
AND TABLE_NAME IN ('file_operations_log', 'file_access_log', 'tenant_files', 'file_shares', 'tenant_storage_quotas')
ORDER BY TABLE_NAME;

-- =====================================================
-- NOTES
-- =====================================================
-- 
-- Foreign keys removed for compatibility:
-- - file_access_log.user_id (was referencing users.id)
-- - tenant_files.uploaded_by (was referencing users.id)
-- - file_shares.shared_by (was referencing users.id)
-- - file_shares.shared_with (was referencing users.id)
-- 
-- These can be added manually if your users table exists:
-- 
-- ALTER TABLE file_access_log 
-- ADD CONSTRAINT fk_file_access_user 
-- FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE;
-- 
-- ALTER TABLE tenant_files 
-- ADD CONSTRAINT fk_tenant_files_user 
-- FOREIGN KEY (uploaded_by) REFERENCES users(id) ON DELETE SET NULL;
-- 
-- ALTER TABLE file_shares 
-- ADD CONSTRAINT fk_file_shares_shared_by 
-- FOREIGN KEY (shared_by) REFERENCES users(id) ON DELETE CASCADE;
-- 
-- ALTER TABLE file_shares 
-- ADD CONSTRAINT fk_file_shares_shared_with 
-- FOREIGN KEY (shared_with) REFERENCES users(id) ON DELETE CASCADE;
-- 
-- =====================================================

