-- =====================================================
-- MOODLE LMS INTEGRATION SCHEMA
-- =====================================================
-- Adds Moodle integration capabilities to link subjects
-- with Moodle courses for automatic student enrollment
-- =====================================================

-- =====================================================
-- STEP 1: Add Moodle fields to subjects table
-- =====================================================

ALTER TABLE subjects 
ADD COLUMN moodle_course_id INT NULL COMMENT 'Moodle course ID',
ADD COLUMN moodle_course_name VARCHAR(255) NULL COMMENT 'Moodle course name',
ADD COLUMN moodle_shortname VARCHAR(100) NULL COMMENT 'Moodle course shortname',
ADD COLUMN auto_enroll BOOLEAN DEFAULT TRUE COMMENT 'Auto-enroll students when they register/pay',
ADD COLUMN moodle_sync_enabled BOOLEAN DEFAULT FALSE COMMENT 'Enable sync with Moodle',
ADD INDEX idx_moodle_course (moodle_course_id);

-- =====================================================
-- STEP 2: Add Moodle configuration to tenant_settings
-- =====================================================

-- Check if moodle_url column exists before adding
ALTER TABLE tenant_settings
ADD COLUMN IF NOT EXISTS moodle_url VARCHAR(255) NULL COMMENT 'Moodle site URL',
ADD COLUMN IF NOT EXISTS moodle_token VARCHAR(255) NULL COMMENT 'Moodle web service token',
ADD COLUMN IF NOT EXISTS moodle_enabled BOOLEAN DEFAULT FALSE COMMENT 'Enable Moodle integration',
ADD COLUMN IF NOT EXISTS moodle_service_name VARCHAR(100) DEFAULT 'moodle_mobile_app' COMMENT 'Moodle web service name';

-- =====================================================
-- STEP 3: Create Moodle enrollment log table
-- =====================================================

CREATE TABLE IF NOT EXISTS moodle_enrollments (
    id BIGINT AUTO_INCREMENT PRIMARY KEY,
    student_id INT NOT NULL,
    subject_id INT NOT NULL,
    moodle_course_id INT NOT NULL,
    moodle_user_id INT NULL COMMENT 'Student user ID in Moodle',
    enrollment_status ENUM('pending', 'enrolled', 'failed', 'unenrolled') DEFAULT 'pending',
    enrollment_method ENUM('manual', 'auto_payment', 'auto_registration', 'bulk') DEFAULT 'auto_payment',
    enrolled_at TIMESTAMP NULL,
    error_message TEXT NULL,
    retry_count INT DEFAULT 0,
    last_retry_at TIMESTAMP NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    academy_reference VARCHAR(50) NULL,
    
    INDEX idx_student (student_id),
    INDEX idx_subject (subject_id),
    INDEX idx_moodle_course (moodle_course_id),
    INDEX idx_status (enrollment_status),
    INDEX idx_academy (academy_reference),
    UNIQUE KEY unique_enrollment (student_id, moodle_course_id),
    
    FOREIGN KEY (student_id) REFERENCES students(id) ON DELETE CASCADE,
    FOREIGN KEY (subject_id) REFERENCES subjects(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- STEP 4: Create Moodle sync log table
-- =====================================================

CREATE TABLE IF NOT EXISTS moodle_sync_log (
    id BIGINT AUTO_INCREMENT PRIMARY KEY,
    sync_type ENUM('enrollment', 'unenrollment', 'user_create', 'course_sync', 'grade_sync') NOT NULL,
    student_id INT NULL,
    subject_id INT NULL,
    moodle_course_id INT NULL,
    status ENUM('success', 'failed', 'pending') NOT NULL,
    request_data TEXT NULL COMMENT 'JSON data sent to Moodle',
    response_data TEXT NULL COMMENT 'JSON response from Moodle',
    error_message TEXT NULL,
    execution_time DECIMAL(10,3) NULL COMMENT 'Time in seconds',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    academy_reference VARCHAR(50) NULL,
    
    INDEX idx_sync_type (sync_type),
    INDEX idx_status (status),
    INDEX idx_created (created_at),
    INDEX idx_academy (academy_reference)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- STEP 5: Add Moodle user mapping to students table
-- =====================================================

ALTER TABLE students
ADD COLUMN IF NOT EXISTS moodle_user_id INT NULL COMMENT 'User ID in Moodle',
ADD COLUMN IF NOT EXISTS moodle_username VARCHAR(100) NULL COMMENT 'Username in Moodle',
ADD COLUMN IF NOT EXISTS moodle_synced BOOLEAN DEFAULT FALSE COMMENT 'User created in Moodle',
ADD COLUMN IF NOT EXISTS moodle_last_sync TIMESTAMP NULL COMMENT 'Last sync with Moodle',
ADD INDEX idx_moodle_user (moodle_user_id),
ADD INDEX idx_moodle_synced (moodle_synced);

-- =====================================================
-- STEP 6: Create Moodle course cache table
-- =====================================================

CREATE TABLE IF NOT EXISTS moodle_course_cache (
    id INT AUTO_INCREMENT PRIMARY KEY,
    moodle_course_id INT NOT NULL UNIQUE,
    course_name VARCHAR(255) NOT NULL,
    shortname VARCHAR(100) NOT NULL,
    category_id INT NULL,
    category_name VARCHAR(255) NULL,
    visible BOOLEAN DEFAULT TRUE,
    startdate BIGINT NULL COMMENT 'Unix timestamp',
    enddate BIGINT NULL COMMENT 'Unix timestamp',
    last_synced TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    academy_reference VARCHAR(50) NULL,
    
    INDEX idx_shortname (shortname),
    INDEX idx_visible (visible),
    INDEX idx_academy (academy_reference)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- VERIFICATION QUERIES
-- =====================================================

-- Check if Moodle fields were added to subjects
-- SHOW COLUMNS FROM subjects LIKE 'moodle%';

-- Check if Moodle fields were added to tenant_settings
-- SHOW COLUMNS FROM tenant_settings LIKE 'moodle%';

-- Check if enrollment tables were created
-- SHOW TABLES LIKE 'moodle_%';

-- Check students with Moodle mapping
-- SELECT COUNT(*) as moodle_synced_students FROM students WHERE moodle_synced = TRUE;

-- =====================================================
-- COMPLETE!
-- =====================================================
-- Moodle integration schema is ready!
-- 
-- Next steps:
-- 1. Configure Moodle URL and token in tenant_settings
-- 2. Map subjects to Moodle courses in admin panel
-- 3. Enable auto-enrollment
-- 4. Test enrollment for a student
-- =====================================================

