<?php
/**
 * Automated Tenant Backup Cron Job
 * 
 * Run this via cron for automatic backups
 * 
 * CRON EXAMPLES:
 * Daily (2 AM):     0 2 * * * /usr/bin/php /path/to/cron/tenant_backups.php daily
 * Weekly (Sunday):  0 3 * * 0 /usr/bin/php /path/to/cron/tenant_backups.php weekly
 * Monthly (1st):    0 4 1 * * /usr/bin/php /path/to/cron/tenant_backups.php monthly
 */

// Only run from CLI
if (php_sapi_name() !== 'cli') {
    die("This script must be run from command line\n");
}

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../includes/tenant_directory_manager.php';
require_once __DIR__ . '/../includes/tenant_backup.php';

// Get backup type from command line
$backupType = $argv[1] ?? 'daily';

echo "\n";
echo str_repeat("=", 60) . "\n";
echo "TENANT BACKUP SYSTEM\n";
echo "Type: {$backupType}\n";
echo "Date: " . date('Y-m-d H:i:s') . "\n";
echo str_repeat("=", 60) . "\n\n";

// Database connection
try {
    $conn = new PDO(
        "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME,
        DB_USER,
        DB_PASS,
        [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
    );
} catch (PDOException $e) {
    echo "✗ Database connection failed: " . $e->getMessage() . "\n";
    exit(1);
}

// Initialize backup system
$dirManager = new TenantDirectoryManager($conn);
$backup = new TenantBackup($conn, $dirManager);

// Get all tenants
$stmt = $conn->query("SELECT academy_reference, name FROM tenants");
$tenants = $stmt->fetchAll(PDO::FETCH_ASSOC);

echo "Found " . count($tenants) . " tenant(s)\n\n";

$results = [
    'success' => [],
    'failed' => []
];

// Backup each tenant
foreach ($tenants as $tenant) {
    $tenant_id = $tenant['academy_reference'];
    $tenant_name = $tenant['name'];
    
    echo "Processing: {$tenant_name} ({$tenant_id})...\n";
    
    try {
        // Determine what to backup based on schedule
        switch ($backupType) {
            case 'daily':
                // Daily: Database only
                $result = $backup->backupTenantDatabase($tenant_id);
                $backupDesc = "database";
                break;
                
            case 'weekly':
                // Weekly: Files only
                $result = $backup->backupTenantFiles($tenant_id);
                $backupDesc = "files";
                break;
                
            case 'monthly':
                // Monthly: Full backup
                $result = $backup->fullTenantBackup($tenant_id);
                $backupDesc = "full";
                break;
                
            default:
                throw new Exception("Invalid backup type");
        }
        
        if ($result['success']) {
            echo "  ✓ {$backupDesc} backup: {$result['size_formatted']}\n";
            $results['success'][] = [
                'tenant_id' => $tenant_id,
                'tenant_name' => $tenant_name,
                'filename' => $result['filename'],
                'size' => $result['size_formatted']
            ];
        } else {
            throw new Exception($result['error']);
        }
        
        // Cleanup old backups (keep last 10)
        if ($backupType === 'monthly') {
            $cleanup = $backup->cleanupOldBackups($tenant_id, 10);
            if ($cleanup['success'] && $cleanup['deleted'] > 0) {
                echo "  ✓ Cleaned up {$cleanup['deleted']} old backup(s)\n";
            }
        }
        
    } catch (Exception $e) {
        echo "  ✗ Failed: " . $e->getMessage() . "\n";
        $results['failed'][] = [
            'tenant_id' => $tenant_id,
            'tenant_name' => $tenant_name,
            'error' => $e->getMessage()
        ];
    }
    
    echo "\n";
}

// Summary
echo str_repeat("=", 60) . "\n";
echo "BACKUP SUMMARY\n";
echo str_repeat("=", 60) . "\n";
echo "Successful: " . count($results['success']) . "\n";
echo "Failed: " . count($results['failed']) . "\n";

if (!empty($results['failed'])) {
    echo "\nFailed Backups:\n";
    foreach ($results['failed'] as $fail) {
        echo "  ✗ {$fail['tenant_name']}: {$fail['error']}\n";
    }
}

echo "\n";

// Send email notification if configured
if (defined('SMTP_FROM_EMAIL') && count($results['failed']) > 0) {
    $subject = "Tenant Backup Report - " . count($results['failed']) . " Failed";
    $message = "Backup Summary for " . date('Y-m-d') . "\n\n";
    $message .= "Successful: " . count($results['success']) . "\n";
    $message .= "Failed: " . count($results['failed']) . "\n\n";
    
    if (!empty($results['failed'])) {
        $message .= "Failed Backups:\n";
        foreach ($results['failed'] as $fail) {
            $message .= "  - {$fail['tenant_name']}: {$fail['error']}\n";
        }
    }
    
    @mail(SMTP_FROM_EMAIL, $subject, $message);
}

echo "Backup process complete!\n";
exit(count($results['failed']) > 0 ? 1 : 0);

