<?php
/**
 * Security Scan Cron Job
 * Checks for security issues and vulnerabilities
 */

require_once __DIR__ . '/cron_utils.php';

// Set execution time limit
set_time_limit(300); // 5 minutes

// Log file
$logFile = __DIR__ . '/../logs/security_scan.log';

// Ensure logs directory exists
if (!is_dir(__DIR__ . '/../logs')) {
    mkdir(__DIR__ . '/../logs', 0755, true);
}



function sendSecurityAlert($subject, $message) {
    $adminEmail = 'admin@' . ($_SERVER['SERVER_NAME'] ?? 'localhost');
    $headers = [
        'From: ' . $adminEmail,
        'Reply-To: ' . $adminEmail,
        'X-Mailer: PHP/' . phpversion(),
        'Content-Type: text/html; charset=UTF-8'
    ];
    
    return mail($adminEmail, $subject, $message, implode("\r\n", $headers));
}

logMessage("=== SECURITY SCAN STARTED ===");

try {
    $securityIssues = [];
    $warnings = [];
    
    // Check file permissions
    $criticalFiles = [
        __DIR__ . '/../config.php',
        __DIR__ . '/../includes/functions.php',
        __DIR__ . '/../admin/dashboard.php'
    ];
    
    foreach ($criticalFiles as $file) {
        if (file_exists($file)) {
            $perms = fileperms($file);
            $octal = substr(sprintf('%o', $perms), -4);
            
            if ($octal > '0644') {
                $securityIssues[] = "File $file has overly permissive permissions: $octal";
                logMessage("SECURITY ISSUE: File $file has permissions $octal");
            } else {
                logMessage("File permissions OK for: " . basename($file));
            }
        }
    }
    
    // Check for exposed sensitive files
    $sensitiveFiles = [
        '.env',
        'config.php',
        'database.php',
        'wp-config.php',
        '.htaccess',
        'composer.json',
        'package.json'
    ];
    
    $webRoot = __DIR__ . '/../';
    foreach ($sensitiveFiles as $file) {
        $filePath = $webRoot . $file;
        if (file_exists($filePath)) {
            // Check if file is accessible via web
            $url = 'http://' . ($_SERVER['SERVER_NAME'] ?? 'localhost') . '/' . $file;
            $headers = @get_headers($url);
            
            if ($headers && strpos($headers[0], '200') !== false) {
                $securityIssues[] = "Sensitive file accessible via web: $file";
                logMessage("SECURITY ISSUE: File $file is accessible via web");
            } else {
                logMessage("File $file is not accessible via web");
            }
        }
    }
    
    // Check for SQL injection vulnerabilities in user inputs
    $suspiciousPatterns = [
        'union.*select',
        'drop.*table',
        'delete.*from',
        'insert.*into',
        'update.*set',
        'exec\(',
        'eval\(',
        'system\(',
        'shell_exec\(',
        'passthru\('
    ];
    
    // Check recent log files for suspicious activity
    $logsDir = __DIR__ . '/../logs';
    if (is_dir($logsDir)) {
        $logFiles = glob($logsDir . '/*.log');
        foreach ($logFiles as $logFile) {
            if (filemtime($logFile) > time() - 86400) { // Last 24 hours
                $content = file_get_contents($logFile);
                foreach ($suspiciousPatterns as $pattern) {
                    if (preg_match('/' . $pattern . '/i', $content)) {
                        $warnings[] = "Suspicious pattern found in " . basename($logFile) . ": $pattern";
                        logMessage("WARNING: Suspicious pattern found in " . basename($logFile));
                    }
                }
            }
        }
    }
    
    // Check for weak passwords in database
    try {
        $pdo = getCronDB();\n    if (!$pdo) {\n        throw new Exception("Database connection failed");\n    }
        $stmt = $pdo->query("SELECT id, email, password_hash FROM students WHERE email_verified = 1 LIMIT 100");
        $students = $stmt->fetchAll();
        
        $weakPasswords = 0;
        foreach ($students as $student) {
            // Check if password is too simple (just last name)
            if (strlen($student['password_hash']) < 32) {
                $weakPasswords++;
            }
        }
        
        if ($weakPasswords > 0) {
            $warnings[] = "Found $weakPasswords students with potentially weak passwords";
            logMessage("WARNING: Found $weakPasswords students with weak passwords");
        } else {
            logMessage("Password strength check passed");
        }
        
    } catch (Exception $e) {
        logMessage("WARNING: Could not check password strength: " . $e->getMessage());
    }
    
    // Check for outdated PHP version
    $phpVersion = PHP_VERSION;
    $phpMajor = (int)substr($phpVersion, 0, 1);
    $phpMinor = (int)substr($phpVersion, 2, 1);
    
    if ($phpMajor < 8 || ($phpMajor == 8 && $phpMinor < 1)) {
        $securityIssues[] = "Outdated PHP version: $phpVersion";
        logMessage("SECURITY ISSUE: Outdated PHP version: $phpVersion");
    } else {
        logMessage("PHP version is up to date: $phpVersion");
    }
    
    // Check for dangerous PHP functions
    $dangerousFunctions = [
        'exec', 'system', 'shell_exec', 'passthru', 'eval', 'assert',
        'popen', 'proc_open', 'file_get_contents', 'fopen', 'fwrite'
    ];
    
    $codeFiles = glob(__DIR__ . '/../*.php');
    $codeFiles = array_merge($codeFiles, glob(__DIR__ . '/../admin/*.php'));
    $codeFiles = array_merge($codeFiles, glob(__DIR__ . '/../includes/*.php'));
    
    foreach ($codeFiles as $file) {
        $content = file_get_contents($file);
        foreach ($dangerousFunctions as $func) {
            if (strpos($content, $func . '(') !== false) {
                $warnings[] = "Dangerous function '$func' found in " . basename($file);
                logMessage("WARNING: Dangerous function '$func' found in " . basename($file));
            }
        }
    }
    
    // Check for exposed directories
    $exposedDirs = [
        'admin',
        'includes',
        'cron',
        'logs',
        'backups'
    ];
    
    foreach ($exposedDirs as $dir) {
        $dirPath = __DIR__ . '/../' . $dir;
        if (is_dir($dirPath)) {
            $indexFile = $dirPath . '/index.php';
            if (!file_exists($indexFile)) {
                $warnings[] = "Directory $dir is not protected with index.php";
                logMessage("WARNING: Directory $dir is not protected");
            } else {
                logMessage("Directory $dir is protected");
            }
        }
    }
    
    // Check for HTTPS usage
    $isHttps = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on';
    if (!$isHttps) {
        $warnings[] = "Site is not using HTTPS";
        logMessage("WARNING: Site is not using HTTPS");
    } else {
        logMessage("HTTPS is enabled");
    }
    
    // Check for exposed error messages
    $displayErrors = ini_get('display_errors');
    if ($displayErrors) {
        $warnings[] = "PHP display_errors is enabled";
        logMessage("WARNING: PHP display_errors is enabled");
    } else {
        logMessage("PHP display_errors is disabled");
    }
    
    // Check for exposed PHP version
    $exposePhp = ini_get('expose_php');
    if ($exposePhp) {
        $warnings[] = "PHP version is exposed in headers";
        logMessage("WARNING: PHP version is exposed in headers");
    } else {
        logMessage("PHP version is not exposed");
    }
    
    // Send security alert if critical issues found
    if (count($securityIssues) > 0) {
        $alertSubject = "SECURITY ALERT - " . APP_NAME;
        $alertMessage = "
            <html>
            <body>
                <h2 style='color: red;'>SECURITY ALERT</h2>
                <p>The following critical security issues were detected:</p>
                <ul>
        ";
        
        foreach ($securityIssues as $issue) {
            $alertMessage .= "<li style='color: red;'>$issue</li>";
        }
        
        $alertMessage .= "
                </ul>
                <p>Please address these issues immediately.</p>
                <p>Server: " . ($_SERVER['SERVER_NAME'] ?? 'localhost') . "</p>
                <p>Time: " . date('Y-m-d H:i:s') . "</p>
            </body>
            </html>
        ";
        
        if (sendSecurityAlert($alertSubject, $alertMessage)) {
            logMessage("Security alert email sent to administrator");
        } else {
            logMessage("WARNING: Failed to send security alert email");
        }
    }
    
    // Summary
    logMessage("Security scan completed:");
    logMessage("- Critical issues: " . count($securityIssues));
    logMessage("- Warnings: " . count($warnings));
    logMessage("- PHP version: $phpVersion");
    logMessage("- HTTPS enabled: " . ($isHttps ? 'Yes' : 'No'));
    logMessage("- Display errors: " . ($displayErrors ? 'Yes' : 'No'));
    logMessage("- Expose PHP: " . ($exposePhp ? 'Yes' : 'No'));
    
    if (count($securityIssues) == 0 && count($warnings) == 0) {
        logMessage("No security issues found");
    }
    
} catch (Exception $e) {
    logMessage("ERROR: Security scan failed: " . $e->getMessage());
    exit(1);
}

logMessage("=== SECURITY SCAN COMPLETED ===");
?>
