<?php
/**
 * Cache Clearing Cron Job
 * Clears application cache for better performance
 */

require_once __DIR__ . '/cron_utils.php';

// Log file
$logFile = __DIR__ . '/../logs/cache_clear.log';

function clearDirectory($dir, $pattern = '*') {
    $cleared = 0;
    $size = 0;
    
    if (!is_dir($dir)) {
        return ['cleared' => 0, 'size' => 0];
    }
    
    $files = glob($dir . '/' . $pattern);
    foreach ($files as $file) {
        if (is_file($file)) {
            $size += filesize($file);
            if (unlink($file)) {
                $cleared++;
            }
        }
    }
    
    return ['cleared' => $cleared, 'size' => $size];
}

logMessage("=== CACHE CLEARING STARTED ===");

try {
    $totalCleared = 0;
    $totalSize = 0;
    
    // Clear PHP opcache if available
    if (function_exists('opcache_reset')) {
        if (opcache_reset()) {
            logMessage("PHP opcache cleared successfully");
        } else {
            logMessage("WARNING: Failed to clear PHP opcache");
        }
    } else {
        logMessage("PHP opcache not available");
    }
    
    // Clear application cache directory
    $cacheDir = __DIR__ . '/../cache';
    if (is_dir($cacheDir)) {
        $result = clearDirectory($cacheDir);
        $totalCleared += $result['cleared'];
        $totalSize += $result['size'];
        logMessage("Application cache cleared: {$result['cleared']} files, " . round($result['size']/1024, 2) . "KB");
    } else {
        logMessage("Application cache directory not found");
    }
    
    // Clear temporary files
    $tempDir = sys_get_temp_dir();
    $tempFiles = glob($tempDir . '/php*');
    $tempCleared = 0;
    $tempSize = 0;
    
    foreach ($tempFiles as $file) {
        if (is_file($file) && filemtime($file) < time() - 3600) { // Older than 1 hour
            $tempSize += filesize($file);
            if (unlink($file)) {
                $tempCleared++;
            }
        }
    }
    
    if ($tempCleared > 0) {
        logMessage("Temporary files cleared: $tempCleared files, " . round($tempSize/1024, 2) . "KB");
    }
    
    // Clear session files (older than 24 hours)
    $sessionPath = session_save_path();
    if (empty($sessionPath)) {
        $sessionPath = sys_get_temp_dir();
    }
    
    $sessionFiles = glob($sessionPath . '/sess_*');
    $sessionCleared = 0;
    $sessionSize = 0;
    
    foreach ($sessionFiles as $file) {
        if (is_file($file) && filemtime($file) < time() - 86400) { // Older than 24 hours
            $sessionSize += filesize($file);
            if (unlink($file)) {
                $sessionCleared++;
            }
        }
    }
    
    if ($sessionCleared > 0) {
        logMessage("Old session files cleared: $sessionCleared files, " . round($sessionSize/1024, 2) . "KB");
    }
    
    // Clear log files (older than 7 days, keep last 10)
    $logsDir = __DIR__ . '/../logs';
    if (is_dir($logsDir)) {
        $logFiles = glob($logsDir . '/*.log');
        
        // Sort by modification time (newest first)
        usort($logFiles, function($a, $b) {
            return filemtime($b) - filemtime($a);
        });
        
        $logCleared = 0;
        $logSize = 0;
        
        // Keep the 10 newest files, delete the rest if older than 7 days
        for ($i = 10; $i < count($logFiles); $i++) {
            $file = $logFiles[$i];
            if (filemtime($file) < time() - (7 * 24 * 60 * 60)) { // 7 days
                $logSize += filesize($file);
                if (unlink($file)) {
                    $logCleared++;
                }
            }
        }
        
        if ($logCleared > 0) {
            logMessage("Old log files cleared: $logCleared files, " . round($logSize/1024, 2) . "KB");
        }
    }
    
    // Clear browser cache headers (if running in web context)
    if (php_sapi_name() !== 'cli') {
        header('Cache-Control: no-cache, no-store, must-revalidate');
        header('Pragma: no-cache');
        header('Expires: 0');
    }
    
    // Clear any custom cache directories
    $customCacheDirs = [
        __DIR__ . '/../tmp',
        __DIR__ . '/../temp',
        __DIR__ . '/../uploads/temp'
    ];
    
    foreach ($customCacheDirs as $dir) {
        if (is_dir($dir)) {
            $result = clearDirectory($dir);
            if ($result['cleared'] > 0) {
                $totalCleared += $result['cleared'];
                $totalSize += $result['size'];
                logMessage("Custom cache cleared: {$result['cleared']} files, " . round($result['size']/1024, 2) . "KB");
            }
        }
    }
    
    // Check memory usage after clearing
    $memoryUsage = memory_get_usage(true);
    $memoryPeak = memory_get_peak_usage(true);
    
    logMessage("Memory usage after clearing: " . round($memoryUsage/1024/1024, 2) . "MB");
    logMessage("Memory peak: " . round($memoryPeak/1024/1024, 2) . "MB");
    
    // Summary
    logMessage("Cache clearing completed:");
    logMessage("- Total files cleared: $totalCleared");
    logMessage("- Total size cleared: " . round($totalSize/1024/1024, 2) . "MB");
    logMessage("- Temporary files: $tempCleared");
    logMessage("- Session files: $sessionCleared");
    logMessage("- Log files: $logCleared");
    
    if ($totalCleared > 0) {
        logMessage("Cache cleared successfully");
    } else {
        logMessage("No cache files found to clear");
    }
    
} catch (Exception $e) {
    logMessage("ERROR: Cache clearing failed: " . $e->getMessage());
    exit(1);
}

logMessage("=== CACHE CLEARING COMPLETED ===");
?>
