<?php
/**
 * Automated Backup System for Skolo-Kine
 */

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../includes/functions.php';

// Backup configuration
define('BACKUP_DIR', __DIR__ . '/../backups');
define('MAX_BACKUPS', 30); // Keep last 30 backups
define('BACKUP_COMPRESSION', 'gzip');

class BackupSystem {
    private $pdo;
    private $backup_dir;
    
    public function __construct() {
        $this->pdo = getDB();
        $this->backup_dir = BACKUP_DIR;
        $this->ensureBackupDirectory();
    }
    
    /**
     * Create full system backup
     */
    public function createFullBackup() {
        $timestamp = date('Y-m-d_H-i-s');
        $backup_name = "skolo_kine_full_backup_{$timestamp}";
        $backup_path = $this->backup_dir . '/' . $backup_name;
        
        try {
            // Create backup directory
            mkdir($backup_path, 0755, true);
            
            // Backup database
            $this->backupDatabase($backup_path);
            
            // Backup files
            $this->backupFiles($backup_path);
            
            // Create backup manifest
            $this->createBackupManifest($backup_path, $backup_name);
            
            // Compress backup
            $this->compressBackup($backup_path);
            
            // Clean old backups
            $this->cleanOldBackups();
            
            // Log backup
            $this->logBackup($backup_name, 'FULL', 'SUCCESS');
            
            return [
                'success' => true,
                'backup_name' => $backup_name,
                'size' => $this->getBackupSize($backup_path . '.tar.gz')
            ];
            
        } catch (Exception $e) {
            $this->logBackup($backup_name, 'FULL', 'FAILED', $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Create database backup
     */
    private function backupDatabase($backup_path) {
        $db_file = $backup_path . '/database.sql';
        
        // Get database credentials
        $host = DB_HOST;
        $port = DB_PORT;
        $database = DB_NAME;
        $username = DB_USER;
        $password = DB_PASS;
        
        // Create mysqldump command
        $command = sprintf(
            'mysqldump --host=%s --port=%s --user=%s --password=%s --single-transaction --routines --triggers %s > %s',
            escapeshellarg($host),
            escapeshellarg($port),
            escapeshellarg($username),
            escapeshellarg($password),
            escapeshellarg($database),
            escapeshellarg($db_file)
        );
        
        // Execute backup
        $output = [];
        $return_code = 0;
        exec($command, $output, $return_code);
        
        if ($return_code !== 0) {
            throw new Exception('Database backup failed: ' . implode("\n", $output));
        }
        
        // Verify backup file
        if (!file_exists($db_file) || filesize($db_file) === 0) {
            throw new Exception('Database backup file is empty or missing');
        }
    }
    
    /**
     * Backup important files
     */
    private function backupFiles($backup_path) {
        $files_dir = $backup_path . '/files';
        mkdir($files_dir, 0755, true);
        
        // Files to backup
        $files_to_backup = [
            'config.php',
            'includes/',
            'admin/',
            'payments/',
            'uploads/',
            'logs/',
            'exports/',
            'bot/',
            'vendor/',
            'composer.json',
            'composer.lock'
        ];
        
        foreach ($files_to_backup as $file) {
            $source = __DIR__ . '/../' . $file;
            $destination = $files_dir . '/' . $file;
            
            if (file_exists($source)) {
                if (is_dir($source)) {
                    $this->copyDirectory($source, $destination);
                } else {
                    $this->copyFile($source, $destination);
                }
            }
        }
    }
    
    /**
     * Copy directory recursively
     */
    private function copyDirectory($source, $destination) {
        if (!is_dir($destination)) {
            mkdir($destination, 0755, true);
        }
        
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($source, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::SELF_FIRST
        );
        
        foreach ($iterator as $item) {
            $target = $destination . DIRECTORY_SEPARATOR . $iterator->getSubPathName();
            
            if ($item->isDir()) {
                mkdir($target, 0755, true);
            } else {
                copy($item, $target);
            }
        }
    }
    
    /**
     * Copy file
     */
    private function copyFile($source, $destination) {
        $dir = dirname($destination);
        if (!is_dir($dir)) {
            mkdir($dir, 0755, true);
        }
        copy($source, $destination);
    }
    
    /**
     * Create backup manifest
     */
    private function createBackupManifest($backup_path, $backup_name) {
        $manifest = [
            'backup_name' => $backup_name,
            'created_at' => date('Y-m-d H:i:s'),
            'version' => '1.0',
            'database' => [
                'host' => DB_HOST,
                'name' => DB_NAME,
                'charset' => DB_CHARSET
            ],
            'files' => $this->getBackupFileList($backup_path),
            'system' => [
                'php_version' => PHP_VERSION,
                'mysql_version' => $this->getMySQLVersion(),
                'server' => $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown'
            ]
        ];
        
        file_put_contents(
            $backup_path . '/manifest.json',
            json_encode($manifest, JSON_PRETTY_PRINT)
        );
    }
    
    /**
     * Get list of backed up files
     */
    private function getBackupFileList($backup_path) {
        $files = [];
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($backup_path, RecursiveDirectoryIterator::SKIP_DOTS)
        );
        
        foreach ($iterator as $file) {
            if ($file->isFile()) {
                $files[] = [
                    'path' => str_replace($backup_path . '/', '', $file->getPathname()),
                    'size' => $file->getSize(),
                    'modified' => date('Y-m-d H:i:s', $file->getMTime())
                ];
            }
        }
        
        return $files;
    }
    
    /**
     * Compress backup
     */
    private function compressBackup($backup_path) {
        $archive_name = $backup_path . '.tar.gz';
        
        $command = sprintf(
            'tar -czf %s -C %s %s',
            escapeshellarg($archive_name),
            escapeshellarg(dirname($backup_path)),
            escapeshellarg(basename($backup_path))
        );
        
        exec($command, $output, $return_code);
        
        if ($return_code !== 0) {
            throw new Exception('Backup compression failed');
        }
        
        // Remove uncompressed directory
        $this->removeDirectory($backup_path);
    }
    
    /**
     * Remove directory recursively
     */
    private function removeDirectory($dir) {
        if (!is_dir($dir)) {
            return;
        }
        
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($dir, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::CHILD_FIRST
        );
        
        foreach ($iterator as $file) {
            if ($file->isDir()) {
                rmdir($file->getPathname());
            } else {
                unlink($file->getPathname());
            }
        }
        
        rmdir($dir);
    }
    
    /**
     * Clean old backups
     */
    private function cleanOldBackups() {
        $backups = glob($this->backup_dir . '/skolo_kine_*_backup_*.tar.gz');
        
        if (count($backups) > MAX_BACKUPS) {
            // Sort by modification time (oldest first)
            usort($backups, function($a, $b) {
                return filemtime($a) - filemtime($b);
            });
            
            // Remove oldest backups
            $to_remove = array_slice($backups, 0, count($backups) - MAX_BACKUPS);
            foreach ($to_remove as $backup) {
                unlink($backup);
            }
        }
    }
    
    /**
     * Get backup size
     */
    private function getBackupSize($backup_file) {
        if (file_exists($backup_file)) {
            return $this->formatBytes(filesize($backup_file));
        }
        return '0 B';
    }
    
    /**
     * Format bytes
     */
    private function formatBytes($bytes, $precision = 2) {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        
        for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
            $bytes /= 1024;
        }
        
        return round($bytes, $precision) . ' ' . $units[$i];
    }
    
    /**
     * Get MySQL version
     */
    private function getMySQLVersion() {
        try {
            $stmt = $this->pdo->query('SELECT VERSION() as version');
            $result = $stmt->fetch();
            return $result['version'] ?? 'Unknown';
        } catch (Exception $e) {
            return 'Unknown';
        }
    }
    
    /**
     * Ensure backup directory exists
     */
    private function ensureBackupDirectory() {
        if (!is_dir($this->backup_dir)) {
            mkdir($this->backup_dir, 0755, true);
        }
    }
    
    /**
     * Log backup operation
     */
    private function logBackup($backup_name, $type, $status, $error = null) {
        try {
            $stmt = $this->pdo->prepare("
                INSERT INTO backup_logs (backup_name, type, status, error_message, created_at) 
                VALUES (?, ?, ?, ?, NOW())
            ");
            $stmt->execute([$backup_name, $type, $status, $error]);
        } catch (Exception $e) {
            error_log("Failed to log backup: " . $e->getMessage());
        }
    }
    
    /**
     * Restore from backup
     */
    public function restoreFromBackup($backup_file) {
        try {
            // Extract backup
            $extract_dir = $this->backup_dir . '/restore_' . time();
            mkdir($extract_dir, 0755, true);
            
            $command = sprintf(
                'tar -xzf %s -C %s',
                escapeshellarg($backup_file),
                escapeshellarg($extract_dir)
            );
            
            exec($command, $output, $return_code);
            
            if ($return_code !== 0) {
                throw new Exception('Backup extraction failed');
            }
            
            // Restore database
            $this->restoreDatabase($extract_dir);
            
            // Restore files
            $this->restoreFiles($extract_dir);
            
            // Clean up
            $this->removeDirectory($extract_dir);
            
            $this->logBackup(basename($backup_file), 'RESTORE', 'SUCCESS');
            
            return ['success' => true];
            
        } catch (Exception $e) {
            $this->logBackup(basename($backup_file), 'RESTORE', 'FAILED', $e->getMessage());
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Restore database
     */
    private function restoreDatabase($extract_dir) {
        $db_file = $extract_dir . '/database.sql';
        
        if (!file_exists($db_file)) {
            throw new Exception('Database backup file not found');
        }
        
        $command = sprintf(
            'mysql --host=%s --port=%s --user=%s --password=%s %s < %s',
            escapeshellarg(DB_HOST),
            escapeshellarg(DB_PORT),
            escapeshellarg(DB_USER),
            escapeshellarg(DB_PASS),
            escapeshellarg(DB_NAME),
            escapeshellarg($db_file)
        );
        
        exec($command, $output, $return_code);
        
        if ($return_code !== 0) {
            throw new Exception('Database restore failed: ' . implode("\n", $output));
        }
    }
    
    /**
     * Restore files
     */
    private function restoreFiles($extract_dir) {
        $files_dir = $extract_dir . '/files';
        
        if (!is_dir($files_dir)) {
            return;
        }
        
        $this->copyDirectory($files_dir, __DIR__ . '/..');
    }
}

// Create backup logs table if it doesn't exist
try {
    $pdo = getDB();
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS backup_logs (
            id INT AUTO_INCREMENT PRIMARY KEY,
            backup_name VARCHAR(255) NOT NULL,
            type ENUM('FULL', 'INCREMENTAL', 'RESTORE') NOT NULL,
            status ENUM('SUCCESS', 'FAILED', 'IN_PROGRESS') NOT NULL,
            error_message TEXT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            INDEX idx_backup_name (backup_name),
            INDEX idx_status (status),
            INDEX idx_created_at (created_at)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
    ");
} catch (Exception $e) {
    error_log("Failed to create backup_logs table: " . $e->getMessage());
}

// Run backup if called directly
if (basename(__FILE__) === basename($_SERVER['SCRIPT_NAME'])) {
    $backup = new BackupSystem();
    $result = $backup->createFullBackup();
    
    if ($result['success']) {
        echo "Backup completed successfully: " . $result['backup_name'] . " (" . $result['size'] . ")\n";
    } else {
        echo "Backup failed: " . $result['error'] . "\n";
        exit(1);
    }
}
?>
