"""
Database utility functions for Python Moodle Sync Bot
Handles database connections and common database operations
"""

import logging
import pymysql
from contextlib import contextmanager
from typing import List, Dict, Optional, Any
from config import DB_CONFIG

# Configure logging
logger = logging.getLogger(__name__)

class DatabaseError(Exception):
    """Custom exception for database errors"""
    pass

@contextmanager
def get_db_connection():
    """
    Context manager for database connections
    Ensures proper connection cleanup
    """
    connection = None
    try:
        connection = pymysql.connect(
            host=DB_CONFIG['host'],
            port=DB_CONFIG['port'],
            user=DB_CONFIG['user'],
            password=DB_CONFIG['password'],
            database=DB_CONFIG['database'],
            charset=DB_CONFIG['charset'],
            autocommit=True,
            cursorclass=pymysql.cursors.DictCursor
        )
        yield connection
    except pymysql.Error as e:
        logger.error(f"Database connection error: {e}")
        raise DatabaseError(f"Database connection failed: {e}")
    finally:
        if connection:
            connection.close()

def execute_query(query: str, params: tuple = None) -> List[Dict[str, Any]]:
    """
    Execute a SELECT query and return results
    """
    try:
        with get_db_connection() as conn:
            with conn.cursor() as cursor:
                cursor.execute(query, params)
                return cursor.fetchall()
    except pymysql.Error as e:
        logger.error(f"Query execution error: {e}")
        raise DatabaseError(f"Query execution failed: {e}")

def execute_update(query: str, params: tuple = None) -> int:
    """
    Execute an INSERT/UPDATE/DELETE query and return affected rows
    """
    try:
        with get_db_connection() as conn:
            with conn.cursor() as cursor:
                cursor.execute(query, params)
                conn.commit()
                return cursor.rowcount
    except pymysql.Error as e:
        logger.error(f"Update execution error: {e}")
        raise DatabaseError(f"Update execution failed: {e}")

def get_students(limit: Optional[int] = None, offset: int = 0) -> List[Dict[str, Any]]:
    """
    Get all students with their grade information
    """
    query = """
        SELECT s.*, g.name as grade_name 
        FROM students s 
        JOIN grades g ON s.grade_id = g.id 
        ORDER BY s.created_at DESC
    """
    if limit:
        query += f" LIMIT {limit} OFFSET {offset}"
    
    return execute_query(query)

def get_verified_students() -> List[Dict[str, Any]]:
    """
    Get only verified students for CSV export
    """
    query = """
        SELECT s.*, g.name as grade_name 
        FROM students s 
        JOIN grades g ON s.grade_id = g.id 
        WHERE s.email_verified = TRUE 
        ORDER BY s.created_at DESC
    """
    return execute_query(query)

def get_student_by_id(student_id: int) -> Optional[Dict[str, Any]]:
    """
    Get a specific student by ID
    """
    query = """
        SELECT s.*, g.name as grade_name 
        FROM students s 
        JOIN grades g ON s.grade_id = g.id 
        WHERE s.id = %s
    """
    results = execute_query(query, (student_id,))
    return results[0] if results else None

def get_student_by_username(username: str) -> Optional[Dict[str, Any]]:
    """
    Get a specific student by username
    """
    query = """
        SELECT s.*, g.name as grade_name 
        FROM students s 
        JOIN grades g ON s.grade_id = g.id 
        WHERE s.username = %s
    """
    results = execute_query(query, (username,))
    return results[0] if results else None

def get_subjects_for_student(student_id: int) -> List[Dict[str, Any]]:
    """
    Get all subjects for a specific student
    """
    query = """
        SELECT s.* 
        FROM student_subject ss 
        JOIN subjects s ON ss.subject_id = s.id 
        WHERE ss.student_id = %s 
        ORDER BY s.name
    """
    return execute_query(query, (student_id,))

def get_subjects_for_grade(grade_id: int) -> List[Dict[str, Any]]:
    """
    Get all subjects for a specific grade
    """
    query = """
        SELECT s.* 
        FROM subjects s 
        JOIN grade_subject gs ON s.id = gs.subject_id 
        WHERE gs.grade_id = %s
        ORDER BY s.name
    """
    return execute_query(query, (grade_id,))

def get_all_subjects() -> List[Dict[str, Any]]:
    """
    Get all subjects
    """
    query = "SELECT * FROM subjects ORDER BY name"
    return execute_query(query)

def get_all_grades() -> List[Dict[str, Any]]:
    """
    Get all grades
    """
    query = "SELECT * FROM grades ORDER BY name"
    return execute_query(query)

def get_student_count() -> int:
    """
    Get total number of students
    """
    query = "SELECT COUNT(*) as count FROM students"
    result = execute_query(query)
    return result[0]['count'] if result else 0

def get_verified_student_count() -> int:
    """
    Get number of verified students
    """
    query = "SELECT COUNT(*) as count FROM students WHERE email_verified = TRUE"
    result = execute_query(query)
    return result[0]['count'] if result else 0

def get_unverified_student_count() -> int:
    """
    Get number of unverified students
    """
    query = "SELECT COUNT(*) as count FROM students WHERE email_verified = FALSE"
    result = execute_query(query)
    return result[0]['count'] if result else 0

def get_dashboard_stats() -> Dict[str, Any]:
    """
    Get dashboard statistics
    """
    stats = {}
    
    # Total students
    stats['total_students'] = get_student_count()
    
    # Students by grade
    query = """
        SELECT g.name as grade, COUNT(s.id) as count 
        FROM grades g 
        LEFT JOIN students s ON g.id = s.grade_id 
        GROUP BY g.id, g.name 
        ORDER BY g.name
    """
    stats['students_by_grade'] = execute_query(query)
    
    # Recent registrations (last 7 days)
    query = """
        SELECT COUNT(*) as count 
        FROM students 
        WHERE created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)
    """
    result = execute_query(query)
    stats['recent_registrations'] = result[0]['count'] if result else 0
    
    # Email verification statistics
    stats['verified_emails'] = get_verified_student_count()
    stats['pending_verifications'] = get_unverified_student_count()
    
    return stats

def test_database_connection() -> bool:
    """
    Test database connection
    """
    try:
        with get_db_connection() as conn:
            with conn.cursor() as cursor:
                cursor.execute("SELECT 1")
                result = cursor.fetchone()
                return result is not None
    except Exception as e:
        logger.error(f"Database connection test failed: {e}")
        return False

def cleanup_unverified_users() -> Dict[str, Any]:
    """
    Clean up unverified users older than 24 hours
    """
    try:
        query = """
            DELETE FROM students 
            WHERE email_verified = FALSE 
            AND created_at < DATE_SUB(NOW(), INTERVAL 24 HOUR)
        """
        deleted_count = execute_update(query)
        
        logger.info(f"Cleanup: Deleted {deleted_count} unverified users older than 24 hours")
        
        return {
            'success': True,
            'deleted_count': deleted_count,
            'message': f"Successfully deleted {deleted_count} unverified users"
        }
    except Exception as e:
        logger.error(f"Cleanup error: {e}")
        return {
            'success': False,
            'deleted_count': 0,
            'message': f"Error during cleanup: {e}"
        }

def get_unverified_user_stats() -> Dict[str, Any]:
    """
    Get statistics about unverified users
    """
    # Count unverified users
    query = """
        SELECT COUNT(*) as total_unverified 
        FROM students 
        WHERE email_verified = FALSE
    """
    result = execute_query(query)
    total_unverified = result[0]['total_unverified'] if result else 0
    
    # Count unverified users older than 24 hours
    query = """
        SELECT COUNT(*) as expired_unverified 
        FROM students 
        WHERE email_verified = FALSE 
        AND created_at < DATE_SUB(NOW(), INTERVAL 24 HOUR)
    """
    result = execute_query(query)
    expired_unverified = result[0]['expired_unverified'] if result else 0
    
    # Count unverified users by age
    query = """
        SELECT 
            COUNT(CASE WHEN created_at >= DATE_SUB(NOW(), INTERVAL 1 HOUR) THEN 1 END) as last_hour,
            COUNT(CASE WHEN created_at >= DATE_SUB(NOW(), INTERVAL 6 HOUR) THEN 1 END) as last_6_hours,
            COUNT(CASE WHEN created_at >= DATE_SUB(NOW(), INTERVAL 12 HOUR) THEN 1 END) as last_12_hours,
            COUNT(CASE WHEN created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR) THEN 1 END) as last_24_hours
        FROM students 
        WHERE email_verified = FALSE
    """
    result = execute_query(query)
    age_stats = result[0] if result else {}
    
    return {
        'total_unverified': total_unverified,
        'expired_unverified': expired_unverified,
        'age_breakdown': age_stats
    }
