#!/bin/bash
###############################################################################
# Automated Database Backup Script
# Purpose: Daily automated backups with rotation
# Schedule: Run via cron at 2 AM daily
###############################################################################

# Configuration
DATE=$(date +%Y%m%d_%H%M%S)
DAY=$(date +%A)
BACKUP_DIR="/home/melane/backups/database"
LOG_DIR="/home/melane/backups/logs"
DB_NAME="melane_kine"
DB_USER="melane_user"
DB_PASS="YOUR_DATABASE_PASSWORD_HERE"  # CHANGE THIS!

# Email notification (optional)
NOTIFY_EMAIL="admin@melanegroup.com"

# Create directories if they don't exist
mkdir -p "$BACKUP_DIR"
mkdir -p "$BACKUP_DIR/daily"
mkdir -p "$BACKUP_DIR/weekly"
mkdir -p "$BACKUP_DIR/monthly"
mkdir -p "$LOG_DIR"

# Log file
LOG_FILE="$LOG_DIR/backup_$DATE.log"

# Function to log messages
log_message() {
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] $1" | tee -a "$LOG_FILE"
}

log_message "===== Starting Database Backup ====="
log_message "Database: $DB_NAME"
log_message "Backup Type: Daily"

# Daily Backup
log_message "Creating daily backup..."
DAILY_BACKUP="$BACKUP_DIR/daily/backup_$DATE.sql.gz"

if mysqldump -u"$DB_USER" -p"$DB_PASS" "$DB_NAME" | gzip > "$DAILY_BACKUP"; then
    BACKUP_SIZE=$(du -h "$DAILY_BACKUP" | cut -f1)
    log_message "✓ Daily backup completed successfully: $DAILY_BACKUP ($BACKUP_SIZE)"
else
    log_message "✗ ERROR: Daily backup failed!"
    echo "Database backup failed on $(date)" | mail -s "BACKUP FAILURE" "$NOTIFY_EMAIL"
    exit 1
fi

# Weekly Backup (Every Sunday)
if [ "$DAY" == "Sunday" ]; then
    log_message "Creating weekly backup..."
    WEEK_NUM=$(date +%U)
    WEEKLY_BACKUP="$BACKUP_DIR/weekly/backup_week${WEEK_NUM}_$DATE.sql.gz"
    
    cp "$DAILY_BACKUP" "$WEEKLY_BACKUP"
    log_message "✓ Weekly backup created: $WEEKLY_BACKUP"
fi

# Monthly Backup (First day of month)
if [ $(date +%d) -eq 01 ]; then
    log_message "Creating monthly backup..."
    MONTH=$(date +%Y%m)
    MONTHLY_BACKUP="$BACKUP_DIR/monthly/backup_${MONTH}.sql.gz"
    
    cp "$DAILY_BACKUP" "$MONTHLY_BACKUP"
    log_message "✓ Monthly backup created: $MONTHLY_BACKUP"
fi

# Cleanup old backups
log_message "Cleaning up old backups..."

# Keep daily backups for 7 days
find "$BACKUP_DIR/daily" -name "backup_*.sql.gz" -mtime +7 -delete
DAILY_COUNT=$(find "$BACKUP_DIR/daily" -name "backup_*.sql.gz" | wc -l)
log_message "Daily backups retained: $DAILY_COUNT"

# Keep weekly backups for 4 weeks
find "$BACKUP_DIR/weekly" -name "backup_*.sql.gz" -mtime +28 -delete
WEEKLY_COUNT=$(find "$BACKUP_DIR/weekly" -name "backup_*.sql.gz" | wc -l)
log_message "Weekly backups retained: $WEEKLY_COUNT"

# Keep monthly backups for 12 months
find "$BACKUP_DIR/monthly" -name "backup_*.sql.gz" -mtime +365 -delete
MONTHLY_COUNT=$(find "$BACKUP_DIR/monthly" -name "backup_*.sql.gz" | wc -l)
log_message "Monthly backups retained: $MONTHLY_COUNT"

# Keep logs for 30 days
find "$LOG_DIR" -name "backup_*.log" -mtime +30 -delete

# Verify backup integrity
log_message "Verifying backup integrity..."
if gzip -t "$DAILY_BACKUP" 2>/dev/null; then
    log_message "✓ Backup integrity verified"
else
    log_message "✗ WARNING: Backup may be corrupted!"
    echo "Backup integrity check failed on $(date)" | mail -s "BACKUP WARNING" "$NOTIFY_EMAIL"
fi

# Calculate total backup size
TOTAL_SIZE=$(du -sh "$BACKUP_DIR" | cut -f1)
log_message "Total backup storage used: $TOTAL_SIZE"

log_message "===== Backup Process Completed Successfully ====="

# Optional: Upload to cloud storage (uncomment and configure)
# log_message "Uploading to cloud storage..."
# rclone copy "$DAILY_BACKUP" remote:backups/database/
# log_message "✓ Cloud upload completed"

exit 0

