<?php
require_once '../includes/functions.php';

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

// Validate CSRF token
if (!validateCSRFToken($_POST['csrf_token'] ?? '')) {
    http_response_code(403);
    echo json_encode(['success' => false, 'message' => 'Invalid CSRF token']);
    exit;
}

$studentId = (int)($_POST['student_id'] ?? 0);
$statusType = $_POST['status_type'] ?? '';
$newStatus = $_POST['new_status'] ?? '';

if (!$studentId || !$statusType || $newStatus === '') {
    echo json_encode(['success' => false, 'message' => 'Missing required parameters']);
    exit;
}

try {
    $pdo = getDB();
    
    if ($statusType === 'email') {
        // Update email verification status
        $verified = $newStatus === '1' ? 1 : 0;
        $stmt = $pdo->prepare('UPDATE students SET email_verified = ? WHERE id = ?');
        $stmt->execute([$verified, $studentId]);
        
        echo json_encode([
            'success' => true, 
            'message' => 'Email status updated successfully',
            'new_status' => $verified ? 'verified' : 'pending'
        ]);
        
    } elseif ($statusType === 'payment') {
        // Update payment status
        
        // Get student details for academy_reference
        $stmt = $pdo->prepare('SELECT academy_reference FROM students WHERE id = ?');
        $stmt->execute([$studentId]);
        $student = $stmt->fetch();
        $academy_reference = $student['academy_reference'] ?? getCurrentSchool();
        
        if ($newStatus === 'paid') {
            // Check if student already has completed payments
            $stmt = $pdo->prepare('SELECT COUNT(*) as count FROM payments WHERE student_id = ? AND status = "completed"');
            $stmt->execute([$studentId]);
            $hasPaid = $stmt->fetch()['count'] > 0;
            
            if (!$hasPaid) {
                // Create a manual payment record
                $stmt = $pdo->prepare('
                    INSERT INTO payments 
                    (student_id, amount, payment_reference, status, payment_method, notes, academy_reference, payment_date) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, NOW())
                ');
                $stmt->execute([
                    $studentId, 
                    0.00, 
                    'MANUAL_' . time(), 
                    'completed',
                    'manual',
                    'Manually marked as paid by admin',
                    $academy_reference
                ]);
            }
            
            // Always enroll student in subjects when marking as paid
            $subjects = getSubjectsForStudent($studentId);
            $subjectIds = array_column($subjects, 'id');
            if (!empty($subjectIds)) {
                enrollStudentInSubjects($studentId, $subjectIds);
                
                // Trigger bot to sync enrollment to Moodle
                if (function_exists('triggerMoodleSyncBot')) {
                    triggerMoodleSyncBot();
                }
            }
            
        } elseif ($newStatus === 'unpaid') {
            // Mark all payments as failed
            $stmt = $pdo->prepare('UPDATE payments SET status = "failed" WHERE student_id = ?');
            $stmt->execute([$studentId]);
        }
        
        echo json_encode([
            'success' => true, 
            'message' => 'Payment status updated successfully',
            'new_status' => $newStatus
        ]);
        
    } else {
        echo json_encode(['success' => false, 'message' => 'Invalid status type']);
    }
    
} catch (Exception $e) {
    error_log("Status update error: " . $e->getMessage());
    echo json_encode(['success' => false, 'message' => 'Database error occurred']);
}
?>
