<?php
/**
 * Student Payment Details
 * View complete payment history and details for a specific student
 */

session_start();
require_once '../../includes/functions.php';
require_once '../../includes/school_context.php';
require_once '../../includes/favicon_helper.php';
requireAdminLogin();

$pdo = getDB();
$academy_reference = getCurrentSchool();
$student_id = (int)($_GET['id'] ?? 0);

// Get student with payment schedule
$stmt = $pdo->prepare("
    SELECT 
        s.*, sps.*, g.name as grade_name, ar.academy_name
    FROM students s
    LEFT JOIN student_payment_schedules sps ON s.id = sps.student_id
    LEFT JOIN grades g ON s.grade_id = g.id
    LEFT JOIN academy_references ar ON s.academy_reference = ar.reference_code
    WHERE s.id = ? AND s.academy_reference = ?
");
$stmt->execute([$student_id, $academy_reference]);
$student = $stmt->fetch();

if (!$student) {
    die("Student not found or access denied.");
}

// Get payment history
$stmt = $pdo->prepare("
    SELECT * FROM payment_transactions 
    WHERE student_id = ? 
    ORDER BY created_at DESC
");
$stmt->execute([$student_id]);
$transactions = $stmt->fetchAll();

// Get reminder history
$stmt = $pdo->prepare("
    SELECT * FROM payment_reminders 
    WHERE student_id = ? 
    ORDER BY sent_at DESC
");
$stmt->execute([$student_id]);
$reminders = $stmt->fetchAll();

// Get suspension history
$stmt = $pdo->prepare("
    SELECT * FROM suspension_logs 
    WHERE student_id = ? 
    ORDER BY suspended_at DESC
");
$stmt->execute([$student_id]);
$suspensions = $stmt->fetchAll();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Payment Details - <?= htmlspecialchars($student['full_name']) ?></title>
    <?php echoFavicon($academy_reference); ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.min.css" rel="stylesheet">
    <style>
        .detail-card {
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
            margin-bottom: 20px;
        }
        .status-badge {
            padding: 5px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: bold;
        }
        .status-active { background: #d4edda; color: #155724; }
        .status-overdue { background: #f8d7da; color: #721c24; }
        .status-suspended { background: #d6d8db; color: #383d41; }
        
        @media print {
            .no-print { display: none !important; }
            body { background: white !important; }
            .detail-card { box-shadow: none; border: 1px solid #ddd; }
        }
    </style>
</head>
<body class="bg-light">
    <div class="container mt-4 mb-5">
        <!-- Header -->
        <div class="d-flex justify-content-between align-items-center mb-4 no-print">
            <div>
                <h2><i class="bi bi-person-circle text-primary"></i> Payment Details</h2>
                <p class="text-muted mb-0"><?= htmlspecialchars($student['full_name']) ?></p>
            </div>
            <div>
                <button onclick="window.print()" class="btn btn-primary me-2">
                    <i class="bi bi-printer"></i> Print Receipt
                </button>
                <a href="manage_student_payments.php" class="btn btn-outline-secondary">
                    <i class="bi bi-arrow-left"></i> Back to Students
                </a>
            </div>
        </div>
        
        <!-- Print Header (only shows when printing) -->
        <div class="d-none" id="printHeader" style="text-align: center; margin-bottom: 30px;">
            <h2><?= htmlspecialchars($student['academy_name']) ?></h2>
            <h4>Payment Receipt</h4>
            <p>Date: <?= date('F j, Y') ?></p>
            <hr>
        </div>

        <!-- Student Info Card -->
        <div class="card detail-card">
            <div class="card-body">
                <div class="row">
                    <div class="col-md-6">
                        <h5>Student Information</h5>
                        <p><strong>Name:</strong> <?= htmlspecialchars($student['full_name']) ?></p>
                        <p><strong>Email:</strong> <?= htmlspecialchars($student['email']) ?></p>
                        <p><strong>Phone:</strong> <?= htmlspecialchars($student['phone']) ?></p>
                        <p><strong>Grade:</strong> <?= htmlspecialchars($student['grade_name']) ?></p>
                    </div>
                    <div class="col-md-6">
                        <h5>Payment Status</h5>
                        <p>
                            <strong>Status:</strong> 
                            <span class="status-badge status-<?= $student['payment_status'] ?? 'active' ?>">
                                <?= strtoupper($student['payment_status'] ?? 'ACTIVE') ?>
                            </span>
                        </p>
                        <p><strong>Balance:</strong> SZL <?= number_format($student['balance'] ?? 0, 2) ?></p>
                        <p><strong>Total Paid:</strong> SZL <?= number_format($student['total_amount_paid'] ?? 0, 2) ?></p>
                        <p><strong>Last Payment:</strong> <?= $student['last_payment_date'] ? date('M j, Y', strtotime($student['last_payment_date'])) : 'Never' ?></p>
                        <p><strong>Next Due:</strong> <?= $student['next_due_date'] ? date('M j, Y', strtotime($student['next_due_date'])) : 'N/A' ?></p>
                    </div>
                </div>
            </div>
        </div>

        <!-- Payment Transactions -->
        <div class="card detail-card">
            <div class="card-header bg-white">
                <h5 class="mb-0"><i class="bi bi-cash-stack"></i> Payment History</h5>
            </div>
            <div class="card-body p-0">
                <div class="table-responsive">
                    <table class="table table-hover mb-0">
                        <thead class="table-light">
                            <tr>
                                <th>Date</th>
                                <th>Reference</th>
                                <th>Amount</th>
                                <th>Method</th>
                                <th>Status</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($transactions as $txn): ?>
                            <tr>
                                <td><?= date('M j, Y', strtotime($txn['created_at'])) ?></td>
                                <td><code><?= htmlspecialchars($txn['transaction_reference']) ?></code></td>
                                <td>SZL <?= number_format($txn['amount'], 2) ?></td>
                                <td><?= strtoupper($txn['payment_method']) ?></td>
                                <td>
                                    <span class="badge bg-<?= $txn['status'] === 'completed' ? 'success' : ($txn['status'] === 'failed' ? 'danger' : 'warning') ?>">
                                        <?= strtoupper($txn['status']) ?>
                                    </span>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                            <?php if (empty($transactions)): ?>
                            <tr><td colspan="5" class="text-center text-muted py-4">No payment transactions</td></tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>

        <!-- Reminders Sent -->
        <div class="card detail-card">
            <div class="card-header bg-white">
                <h5 class="mb-0"><i class="bi bi-bell"></i> Reminders Sent</h5>
            </div>
            <div class="card-body p-0">
                <div class="table-responsive">
                    <table class="table table-hover mb-0">
                        <thead class="table-light">
                            <tr>
                                <th>Date</th>
                                <th>Type</th>
                                <th>Email</th>
                                <th>SMS</th>
                                <th>Paid</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($reminders as $reminder): ?>
                            <tr>
                                <td><?= date('M j, Y', strtotime($reminder['sent_at'])) ?></td>
                                <td><?= ucwords(str_replace('_', ' ', $reminder['reminder_type'])) ?></td>
                                <td>
                                    <?= $reminder['email_sent'] ? '<i class="bi bi-check-circle text-success"></i>' : '<i class="bi bi-x-circle text-danger"></i>' ?>
                                </td>
                                <td>
                                    <?= $reminder['sms_sent'] ? '<i class="bi bi-check-circle text-success"></i>' : '<i class="bi bi-x-circle text-danger"></i>' ?>
                                </td>
                                <td>
                                    <?= $reminder['payment_completed'] ? '<i class="bi bi-check-circle text-success"></i> Yes' : '<i class="bi bi-x-circle text-muted"></i> No' ?>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                            <?php if (empty($reminders)): ?>
                            <tr><td colspan="5" class="text-center text-muted py-4">No reminders sent</td></tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>

        <!-- Suspension History -->
        <?php if (!empty($suspensions)): ?>
        <div class="card detail-card">
            <div class="card-header bg-white">
                <h5 class="mb-0"><i class="bi bi-ban"></i> Suspension History</h5>
            </div>
            <div class="card-body p-0">
                <div class="table-responsive">
                    <table class="table table-hover mb-0">
                        <thead class="table-light">
                            <tr>
                                <th>Suspended</th>
                                <th>Unsuspended</th>
                                <th>Duration</th>
                                <th>Reason</th>
                                <th>Moodle</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($suspensions as $suspension): ?>
                            <tr>
                                <td><?= date('M j, Y', strtotime($suspension['suspended_at'])) ?></td>
                                <td><?= $suspension['unsuspended_at'] ? date('M j, Y', strtotime($suspension['unsuspended_at'])) : '<span class="badge bg-danger">Active</span>' ?></td>
                                <td><?= $suspension['duration_days'] ? $suspension['duration_days'] . ' days' : 'N/A' ?></td>
                                <td><?= ucwords(str_replace('_', ' ', $suspension['suspension_reason'])) ?></td>
                                <td>
                                    <?= $suspension['moodle_suspended'] ? '<i class="bi bi-check-circle text-success"></i>' : '<i class="bi bi-x-circle text-danger"></i>' ?>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        <?php endif; ?>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Show print header only when printing
        window.addEventListener('beforeprint', function() {
            document.getElementById('printHeader').classList.remove('d-none');
        });
        
        window.addEventListener('afterprint', function() {
            document.getElementById('printHeader').classList.add('d-none');
        });
    </script>
</body>
</html>

