<?php
/**
 * Manage Student Payments
 * View and manage all student payment statuses and suspensions
 */

session_start();
require_once '../../includes/functions.php';
require_once '../../includes/school_context.php';
require_once '../../includes/payment_reminder_manager.php';
require_once '../../includes/moodle_suspension_handler.php';
require_once '../../includes/favicon_helper.php';
requireAdminLogin();

$pdo = getDB();
$academy_reference = getCurrentSchool();
$success = '';
$error = '';

// Handle manual actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $manager = new PaymentReminderManager();
    $suspension_handler = new MoodleSuspensionHandler();
    
    if (isset($_POST['send_reminder'])) {
        $student_id = (int)$_POST['student_id'];
        $result = $manager->sendPaymentReminder($student_id, 'initial_reminder');
        
        if ($result['success']) {
            $success = 'Reminder sent successfully!';
        } else {
            $error = 'Failed to send reminder: ' . $result['error'];
        }
    }
    
    if (isset($_POST['suspend_student'])) {
        $student_id = (int)$_POST['student_id'];
        $result = $suspension_handler->suspendUser($student_id, 'manual_admin');
        
        if ($result['success']) {
            $success = 'Student suspended successfully!';
        } else {
            $error = 'Failed to suspend student: ' . $result['error'];
        }
    }
    
    if (isset($_POST['unsuspend_student'])) {
        $student_id = (int)$_POST['student_id'];
        $result = $suspension_handler->unsuspendUser($student_id);
        
        if ($result['success']) {
            $success = 'Student unsuspended successfully!';
        } else {
            $error = 'Failed to unsuspend student: ' . $result['error'];
        }
    }
}

// Get filter parameters
$status_filter = $_GET['status'] ?? 'all';
$search = $_GET['search'] ?? '';

// Build query
$where_clauses = ["s.academy_reference = ?"];
$params = [$academy_reference];

if ($status_filter !== 'all') {
    $where_clauses[] = "sps.payment_status = ?";
    $params[] = $status_filter;
}

if (!empty($search)) {
    $where_clauses[] = "(s.full_name LIKE ? OR s.email LIKE ? OR s.phone LIKE ?)";
    $search_param = '%' . $search . '%';
    $params[] = $search_param;
    $params[] = $search_param;
    $params[] = $search_param;
}

$where_sql = implode(' AND ', $where_clauses);

// Get students with payment info (including those without payment schedules)
$stmt = $pdo->prepare("
    SELECT 
        s.id, s.full_name, s.email, s.phone, s.suspended,
        COALESCE(sps.payment_status, 'no_schedule') as payment_status, 
        sps.last_payment_date, sps.next_due_date,
        sps.total_amount_due, sps.balance, sps.reminder_sent_at,
        sps.warning_sent_at, sps.suspended_at,
        DATEDIFF(CURDATE(), sps.last_payment_date) as days_since_payment,
        DATEDIFF(sps.next_due_date, CURDATE()) as days_until_due,
        g.name as grade_name
    FROM students s
    LEFT JOIN student_payment_schedules sps ON s.id = sps.student_id
    LEFT JOIN grades g ON s.grade_id = g.id
    WHERE {$where_sql}
    ORDER BY 
        CASE COALESCE(sps.payment_status, 'no_schedule')
            WHEN 'no_schedule' THEN 0
            WHEN 'suspended' THEN 1
            WHEN 'overdue' THEN 2
            WHEN 'pending' THEN 3
            WHEN 'active' THEN 4
            ELSE 5
        END,
        sps.next_due_date ASC
");
$stmt->execute($params);
$students = $stmt->fetchAll();

// Get summary statistics
$stmt = $pdo->prepare("
    SELECT 
        COUNT(*) as total,
        SUM(CASE WHEN payment_status = 'active' THEN 1 ELSE 0 END) as active,
        SUM(CASE WHEN payment_status = 'overdue' THEN 1 ELSE 0 END) as overdue,
        SUM(CASE WHEN payment_status = 'suspended' THEN 1 ELSE 0 END) as suspended,
        SUM(total_amount_due) as total_due,
        SUM(balance) as total_balance
    FROM student_payment_schedules
    WHERE academy_reference = ?
");
$stmt->execute([$academy_reference]);
$summary = $stmt->fetch();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Student Payments - <?= APP_NAME ?></title>
    <?php echoFavicon($academy_reference); ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.min.css" rel="stylesheet">
    <style>
        .status-badge {
            padding: 5px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: bold;
        }
        .status-active { background: #d4edda; color: #155724; }
        .status-pending { background: #fff3cd; color: #856404; }
        .status-overdue { background: #f8d7da; color: #721c24; }
        .status-suspended { background: #d6d8db; color: #383d41; }
        .status-no_schedule { background: #cfe2ff; color: #084298; }
        .stat-card {
            border-radius: 10px;
            padding: 20px;
            text-align: center;
            background: white;
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
        }
        .stat-number {
            font-size: 32px;
            font-weight: bold;
        }
        .action-btn {
            padding: 4px 12px;
            font-size: 12px;
        }
        
        @media print {
            .no-print { display: none !important; }
            body { background: white !important; }
            .stat-card, .card { box-shadow: none; border: 1px solid #ddd; }
            table { font-size: 12px; }
            .action-btn { display: none !important; }
        }
    </style>
</head>
<body class="bg-light">
    <div class="container-fluid mt-4 mb-5">
        <!-- Header -->
        <div class="d-flex justify-content-between align-items-center mb-4 no-print">
            <div>
                <h2><i class="bi bi-people-fill text-primary"></i> Manage Student Payments</h2>
                <p class="text-muted mb-0">Monitor and manage student payment statuses</p>
            </div>
            <div>
                <div class="btn-group me-2">
                    <button type="button" class="btn btn-outline-info dropdown-toggle" data-bs-toggle="dropdown">
                        <i class="bi bi-mortarboard"></i> Moodle
                    </button>
                    <ul class="dropdown-menu">
                        <li><a class="dropdown-item" href="../moodle/index.php">
                            <i class="bi bi-house"></i> Moodle Hub
                        </a></li>
                        <li><a class="dropdown-item" href="../moodle/configure_mapping.php">
                            <i class="bi bi-diagram-3"></i> Configure Mappings
                        </a></li>
                        <li><hr class="dropdown-divider"></li>
                        <li><a class="dropdown-item" href="/Multi-Tanent/test_moodle_functions.php?key=test123" target="_blank">
                            <i class="bi bi-clipboard-check"></i> Test API Functions
                        </a></li>
                        <li><a class="dropdown-item" href="/Multi-Tanent/test_moodle_complete_workflow.php?key=test123" target="_blank">
                            <i class="bi bi-gear-wide-connected"></i> Test Complete Workflow
                        </a></li>
                    </ul>
                </div>
                <button onclick="window.print()" class="btn btn-success me-2">
                    <i class="bi bi-printer"></i> Print List
                </button>
                <a href="payment_settings.php" class="btn btn-outline-primary me-2">
                    <i class="bi bi-gear"></i> Settings
                </a>
                <a href="../dashboard.php" class="btn btn-outline-secondary">
                    <i class="bi bi-arrow-left"></i> Back
                </a>
            </div>
        </div>
        
        <!-- Print Header (only shows when printing) -->
        <div class="d-none" id="printHeader" style="text-align: center; margin-bottom: 30px;">
            <h2><?= htmlspecialchars(getSchoolName()) ?></h2>
            <h4>Student Payment Status Report</h4>
            <p>Generated: <?= date('F j, Y g:i A') ?></p>
            <?php if ($status_filter !== 'all'): ?>
            <p><strong>Filter:</strong> <?= strtoupper($status_filter) ?> Students Only</p>
            <?php endif; ?>
            <hr>
        </div>

        <?php if ($success): ?>
        <div class="alert alert-success alert-dismissible fade show">
            <i class="bi bi-check-circle-fill"></i> <?= htmlspecialchars($success) ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php endif; ?>

        <?php if ($error): ?>
        <div class="alert alert-danger alert-dismissible fade show">
            <i class="bi bi-exclamation-triangle-fill"></i> <?= htmlspecialchars($error) ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php endif; ?>

        <!-- Summary Cards -->
        <div class="row mb-4 no-print">
            <div class="col-md-2">
                <div class="stat-card">
                    <i class="bi bi-people text-primary" style="font-size: 24px;"></i>
                    <div class="stat-number text-primary"><?= $summary['total'] ?></div>
                    <small class="text-muted">Total Students</small>
                </div>
            </div>
            <div class="col-md-2">
                <div class="stat-card">
                    <i class="bi bi-check-circle text-success" style="font-size: 24px;"></i>
                    <div class="stat-number text-success"><?= $summary['active'] ?></div>
                    <small class="text-muted">Active</small>
                </div>
            </div>
            <div class="col-md-2">
                <div class="stat-card">
                    <i class="bi bi-exclamation-circle text-warning" style="font-size: 24px;"></i>
                    <div class="stat-number text-warning"><?= $summary['overdue'] ?></div>
                    <small class="text-muted">Overdue</small>
                </div>
            </div>
            <div class="col-md-2">
                <div class="stat-card">
                    <i class="bi bi-ban text-danger" style="font-size: 24px;"></i>
                    <div class="stat-number text-danger"><?= $summary['suspended'] ?></div>
                    <small class="text-muted">Suspended</small>
                </div>
            </div>
            <div class="col-md-2">
                <div class="stat-card">
                    <i class="bi bi-currency-dollar text-info" style="font-size: 24px;"></i>
                    <div class="stat-number text-info">E<?= number_format($summary['total_balance'], 0) ?></div>
                    <small class="text-muted">Total Outstanding</small>
                </div>
            </div>
        </div>

        <!-- Filters -->
        <div class="card mb-4 no-print">
            <div class="card-body">
                <form method="GET" class="row g-3">
                    <div class="col-md-3">
                        <label class="form-label">Filter by Status</label>
                        <select name="status" class="form-select" onchange="this.form.submit()">
                            <option value="all" <?= $status_filter === 'all' ? 'selected' : '' ?>>All Statuses</option>
                            <option value="active" <?= $status_filter === 'active' ? 'selected' : '' ?>>Active</option>
                            <option value="pending" <?= $status_filter === 'pending' ? 'selected' : '' ?>>Pending</option>
                            <option value="overdue" <?= $status_filter === 'overdue' ? 'selected' : '' ?>>Overdue</option>
                            <option value="suspended" <?= $status_filter === 'suspended' ? 'selected' : '' ?>>Suspended</option>
                        </select>
                    </div>
                    <div class="col-md-6">
                        <label class="form-label">Search Students</label>
                        <input type="text" name="search" class="form-control" placeholder="Search by name, email, or phone" value="<?= htmlspecialchars($search) ?>">
                    </div>
                    <div class="col-md-3 d-flex align-items-end">
                        <button type="submit" class="btn btn-primary me-2">
                            <i class="bi bi-search"></i> Search
                        </button>
                        <a href="?" class="btn btn-outline-secondary">
                            <i class="bi bi-x"></i> Clear
                        </a>
                    </div>
                </form>
            </div>
        </div>

        <!-- Students Table -->
        <div class="card">
            <div class="card-header bg-white">
                <h5 class="mb-0">Students (<?= count($students) ?>)</h5>
            </div>
            <div class="card-body p-0">
                <div class="table-responsive">
                    <table class="table table-hover mb-0">
                        <thead class="table-light">
                            <tr>
                                <th>Student</th>
                                <th>Grade</th>
                                <th>Status</th>
                                <th>Last Payment</th>
                                <th>Next Due</th>
                                <th>Balance</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($students as $student): ?>
                            <tr>
                                <td>
                                    <strong><?= htmlspecialchars($student['full_name']) ?></strong><br>
                                    <small class="text-muted">
                                        <?= htmlspecialchars($student['email']) ?><br>
                                        <?= htmlspecialchars($student['phone']) ?>
                                    </small>
                                </td>
                                <td><?= htmlspecialchars($student['grade_name']) ?></td>
                                <td>
                                    <span class="status-badge status-<?= $student['payment_status'] ?>">
                                        <?php
                                        if ($student['payment_status'] === 'no_schedule') {
                                            echo 'NEW STUDENT';
                                        } else {
                                            echo strtoupper($student['payment_status']);
                                        }
                                        ?>
                                    </span>
                                    <?php if ($student['suspended']): ?>
                                    <br><small class="text-danger"><i class="bi bi-ban"></i> Moodle Suspended</small>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?= $student['last_payment_date'] ? date('M j, Y', strtotime($student['last_payment_date'])) : 'Never' ?>
                                    <?php if ($student['days_since_payment']): ?>
                                    <br><small class="text-muted"><?= $student['days_since_payment'] ?> days ago</small>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php if ($student['next_due_date']): ?>
                                        <?= date('M j, Y', strtotime($student['next_due_date'])) ?>
                                        <?php if ($student['days_until_due'] < 0): ?>
                                        <br><small class="text-danger"><?= abs($student['days_until_due']) ?> days overdue</small>
                                        <?php elseif ($student['days_until_due'] <= 7): ?>
                                        <br><small class="text-warning">Due in <?= $student['days_until_due'] ?> days</small>
                                        <?php endif; ?>
                                    <?php else: ?>
                                        <span class="text-muted">Not set</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <strong>SZL <?= number_format((float)($student['balance'] ?? 0), 2) ?></strong>
                                </td>
                                <td>
                                    <div class="btn-group btn-group-sm">
                                        <!-- Send Reminder -->
                                        <form method="POST" class="d-inline" onsubmit="return confirm('Send payment reminder to this student?')">
                                            <input type="hidden" name="student_id" value="<?= $student['id'] ?>">
                                            <button type="submit" name="send_reminder" class="btn btn-sm btn-outline-primary action-btn" title="Send Reminder">
                                                <i class="bi bi-envelope"></i>
                                            </button>
                                        </form>
                                        
                                        <?php if (!$student['suspended']): ?>
                                        <!-- Suspend -->
                                        <form method="POST" class="d-inline" onsubmit="return confirm('Suspend this student in Moodle?')">
                                            <input type="hidden" name="student_id" value="<?= $student['id'] ?>">
                                            <button type="submit" name="suspend_student" class="btn btn-sm btn-outline-danger action-btn" title="Suspend">
                                                <i class="bi bi-ban"></i>
                                            </button>
                                        </form>
                                        <?php else: ?>
                                        <!-- Unsuspend -->
                                        <form method="POST" class="d-inline" onsubmit="return confirm('Unsuspend this student in Moodle?')">
                                            <input type="hidden" name="student_id" value="<?= $student['id'] ?>">
                                            <button type="submit" name="unsuspend_student" class="btn btn-sm btn-outline-success action-btn" title="Unsuspend">
                                                <i class="bi bi-check-circle"></i>
                                            </button>
                                        </form>
                                        <?php endif; ?>
                                        
                                        <!-- View Details -->
                                        <a href="student_payment_details.php?id=<?= $student['id'] ?>" class="btn btn-sm btn-outline-secondary action-btn" title="View Details">
                                            <i class="bi bi-eye"></i>
                                        </a>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                            
                            <?php if (empty($students)): ?>
                            <tr>
                                <td colspan="7" class="text-center py-4 text-muted">
                                    <i class="bi bi-inbox" style="font-size: 48px;"></i>
                                    <p class="mt-2">No students found</p>
                                </td>
                            </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Show/hide print header
        window.addEventListener('beforeprint', function() {
            document.getElementById('printHeader').classList.remove('d-none');
        });
        
        window.addEventListener('afterprint', function() {
            document.getElementById('printHeader').classList.add('d-none');
        });
    </script>
</body>
</html>

