<?php
/**
 * School Settings & Branding
 * Manage school information, branding, and preferences
 */

require_once '../includes/functions.php';
require_once '../includes/school_context.php';
requireAdminLogin();
requireSchool();

$currentSchool = getCurrentSchool();
$schoolName = getSchoolName();
$pdo = getDB();

// Get school data
$stmt = $pdo->prepare("SELECT * FROM academy_references WHERE reference_code = ?");
$stmt->execute([$currentSchool]);
$school = $stmt->fetch();

// Get tenant settings
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM tenant_settings WHERE academy_reference = ?");
$stmt->execute([$currentSchool]);
$settings = [];
while ($row = $stmt->fetch()) {
    $settings[$row['setting_key']] = $row['setting_value'];
}

$errors = [];
$success = false;

// Handle AJAX test requests
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    header('Content-Type: application/json');
    
    if ($_POST['action'] === 'test_moodle') {
        $moodleUrl = trim($_POST['moodle_url'] ?? '');
        $moodleToken = trim($_POST['moodle_token'] ?? '');
        
        if (empty($moodleUrl) || empty($moodleToken)) {
            echo json_encode(['success' => false, 'message' => 'Moodle URL and token are required']);
            exit;
        }
        
        // Test Moodle API connection
        try {
            $testUrl = rtrim($moodleUrl, '/') . '/webservice/rest/server.php';
            $params = [
                'wstoken' => $moodleToken,
                'wsfunction' => 'core_webservice_get_site_info',
                'moodlewsrestformat' => 'json'
            ];
            
            $ch = curl_init($testUrl . '?' . http_build_query($params));
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 10);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
            
            if ($httpCode === 200) {
                $data = json_decode($response, true);
                if (isset($data['sitename'])) {
                    echo json_encode([
                        'success' => true, 
                        'message' => 'Successfully connected to Moodle!',
                        'site_name' => $data['sitename'],
                        'version' => $data['release'] ?? 'Unknown'
                    ]);
                } elseif (isset($data['exception'])) {
                    echo json_encode(['success' => false, 'message' => 'Moodle Error: ' . $data['message']]);
                } else {
                    echo json_encode(['success' => false, 'message' => 'Invalid response from Moodle']);
                }
            } else {
                echo json_encode(['success' => false, 'message' => 'Failed to connect to Moodle (HTTP ' . $httpCode . ')']);
            }
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => 'Connection error: ' . $e->getMessage()]);
        }
        exit;
    }
    
    if ($_POST['action'] === 'test_email') {
        $smtpHost = trim($_POST['smtp_host'] ?? '');
        $smtpPort = trim($_POST['smtp_port'] ?? '587');
        $smtpUsername = trim($_POST['smtp_username'] ?? '');
        $smtpPassword = trim($_POST['smtp_password'] ?? '');
        $smtpEncryption = trim($_POST['smtp_encryption'] ?? 'tls');
        $testEmail = trim($_POST['test_email_address'] ?? '');
        
        if (empty($smtpHost) || empty($smtpUsername) || empty($smtpPassword)) {
            echo json_encode(['success' => false, 'message' => 'SMTP host, username, and password are required']);
            exit;
        }
        
        if (empty($testEmail) || !filter_var($testEmail, FILTER_VALIDATE_EMAIL)) {
            echo json_encode(['success' => false, 'message' => 'Valid test email address is required']);
            exit;
        }
        
        // Test SMTP connection
        try {
            require_once '../vendor/autoload.php'; // PHPMailer
            
            $mail = new PHPMailer\PHPMailer\PHPMailer(true);
            
            // SMTP configuration
            $mail->isSMTP();
            $mail->Host = $smtpHost;
            $mail->SMTPAuth = true;
            $mail->Username = $smtpUsername;
            $mail->Password = $smtpPassword;
            $mail->Port = $smtpPort;
            
            if ($smtpEncryption === 'tls') {
                $mail->SMTPSecure = PHPMailer\PHPMailer\PHPMailer::ENCRYPTION_STARTTLS;
            } elseif ($smtpEncryption === 'ssl') {
                $mail->SMTPSecure = PHPMailer\PHPMailer\PHPMailer::ENCRYPTION_SMTPS;
            }
            
            $mail->setFrom($smtpUsername, $schoolName);
            $mail->addAddress($testEmail);
            $mail->Subject = 'Test Email from ' . $schoolName;
            $mail->Body = "This is a test email from your school management system.\n\nIf you received this, your email configuration is working correctly!\n\nSchool: " . $schoolName . "\nTime: " . date('Y-m-d H:i:s');
            
            if ($mail->send()) {
                echo json_encode([
                    'success' => true, 
                    'message' => 'Test email sent successfully to ' . $testEmail
                ]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Failed to send email: ' . $mail->ErrorInfo]);
            }
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => 'SMTP Error: ' . $e->getMessage()]);
        }
        exit;
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && !isset($_POST['action'])) {
    // Handle form submission
    $schoolName = trim($_POST['school_name'] ?? '');
    $schoolEmail = trim($_POST['school_email'] ?? '');
    $schoolPhone = trim($_POST['school_phone'] ?? '');
    $schoolAddress = trim($_POST['school_address'] ?? '');
    $academicYear = trim($_POST['academic_year'] ?? '');
    $primaryColor = trim($_POST['primary_color'] ?? '#4F46E5');
    $secondaryColor = trim($_POST['secondary_color'] ?? '#10B981');
    $emailFromName = trim($_POST['email_from_name'] ?? '');
    $emailSignature = trim($_POST['email_signature'] ?? '');
    
    // Payment Pricing Settings
    $subjectFee = floatval($_POST['subject_fee'] ?? 350);
    $yearlyDiscount = floatval($_POST['yearly_discount'] ?? 10);
    $currencySymbol = trim($_POST['currency_symbol'] ?? 'E');
    $minimumRequiredSubjects = intval($_POST['minimum_required_subjects'] ?? 0);
    
    // Card Payment Settings
    $cardPaymentsEnabled = isset($_POST['card_payments_enabled']) ? 1 : 0;
    $paygateId = trim($_POST['paygate_id'] ?? '');
    $paygateSecret = trim($_POST['paygate_secret'] ?? '');
    $paymentGateway = trim($_POST['payment_gateway'] ?? 'paygate');
    
    // Moodle API Settings (Super Admin Only)
    if (isSuperAdmin()) {
        $moodleUrl = trim($_POST['moodle_url'] ?? '');
        $moodleToken = trim($_POST['moodle_token'] ?? '');
        $moodleServiceName = trim($_POST['moodle_service_name'] ?? 'moodle_mobile_app');
        $moodleEnabled = isset($_POST['moodle_enabled']) ? 1 : 0;
    } else {
        // Keep existing Moodle settings for non-super admins
        $moodleUrl = $settings['moodle_url'] ?? '';
        $moodleToken = $settings['moodle_token'] ?? '';
        $moodleServiceName = $settings['moodle_service_name'] ?? 'moodle_mobile_app';
        $moodleEnabled = $settings['moodle_enabled'] ?? 0;
    }
    
    // Email SMTP Settings
    $useCustomEmail = isset($_POST['use_custom_email']) ? 1 : 0;
    $smtpHost = trim($_POST['smtp_host'] ?? '');
    $smtpPort = trim($_POST['smtp_port'] ?? '587');
    $smtpUsername = trim($_POST['smtp_username'] ?? '');
    $smtpPassword = trim($_POST['smtp_password'] ?? '');
    $smtpEncryption = trim($_POST['smtp_encryption'] ?? 'tls');
    
    // Validation
    if (empty($schoolName)) {
        $errors[] = 'School name is required';
    }
    
    if (!empty($schoolEmail) && !filter_var($schoolEmail, FILTER_VALIDATE_EMAIL)) {
        $errors[] = 'Valid email address required';
    }
    
    // Validate Moodle settings if enabled (Super Admin Only)
    if (isSuperAdmin() && $moodleEnabled) {
        if (empty($moodleUrl)) {
            $errors[] = 'Moodle URL is required when Moodle integration is enabled';
        } elseif (!filter_var($moodleUrl, FILTER_VALIDATE_URL)) {
            $errors[] = 'Valid Moodle URL is required';
        }
        
        if (empty($moodleToken)) {
            $errors[] = 'Moodle API token is required when Moodle integration is enabled';
        }
    }
    
    // Validate SMTP settings if custom email is enabled
    if ($useCustomEmail) {
        if (empty($smtpHost)) {
            $errors[] = 'SMTP host is required when using custom email';
        }
        if (empty($smtpUsername)) {
            $errors[] = 'SMTP username is required when using custom email';
        }
        if (empty($smtpPassword) && empty($settings['smtp_password'])) {
            $errors[] = 'SMTP password is required when using custom email';
        }
    }
    
    // Handle logo upload using tenant file system
    $logoUrl = $settings['logo_url'] ?? '';
    if (isset($_FILES['logo']) && $_FILES['logo']['error'] === UPLOAD_ERR_OK) {
        require_once '../includes/tenant_directory_manager.php';
        require_once '../includes/tenant_file_upload.php';
        
        try {
            $dirManager = new TenantDirectoryManager($pdo);
            $uploader = new TenantFileUpload($pdo, $dirManager);
            
            // Upload to tenant-specific directory
            $result = $uploader->uploadDocument(
                $currentSchool,
                $_FILES['logo'],
                [
                    'description' => 'School logo',
                    'tags' => 'logo,branding',
                    'related_type' => 'school_settings'
                ]
            );
            
            if ($result['success']) {
                $logoUrl = $result['url'];
            } else {
                $errors[] = 'Failed to upload logo: ' . $result['error'];
            }
        } catch (Exception $e) {
            $errors[] = 'Upload error: ' . $e->getMessage();
        }
    }
    
    if (empty($errors)) {
        try {
            $pdo->beginTransaction();
            
            // Update academy_references
            $stmt = $pdo->prepare("
                UPDATE academy_references 
                SET academy_name = ?,
                    contact_email = ?,
                    contact_phone = ?,
                    address = ?
                WHERE reference_code = ?
            ");
            $stmt->execute([
                $schoolName,
                $schoolEmail,
                $schoolPhone,
                $schoolAddress,
                $currentSchool
            ]);
            
            // Update tenant settings
            $settingsToUpdate = [
                'school_name' => $schoolName,
                'school_email' => $schoolEmail,
                'school_phone' => $schoolPhone,
                'school_address' => $schoolAddress,
                'academic_year' => $academicYear,
                'primary_color' => $primaryColor,
                'secondary_color' => $secondaryColor,
                'logo_url' => $logoUrl,
                'email_from_name' => $emailFromName,
                'email_signature' => $emailSignature,
                // Payment Pricing Settings
                'subject_fee' => $subjectFee,
                'yearly_discount' => $yearlyDiscount,
                'currency_symbol' => $currencySymbol,
                'minimum_required_subjects' => $minimumRequiredSubjects,
                // Card Payment Settings
                'card_payments_enabled' => $cardPaymentsEnabled,
                'paygate_id' => $paygateId,
                'paygate_secret' => $paygateSecret,
                'payment_gateway' => $paymentGateway,
                // Moodle API Settings
                'moodle_url' => $moodleUrl,
                'moodle_token' => $moodleToken,
                'moodle_service_name' => $moodleServiceName,
                'moodle_enabled' => $moodleEnabled,
                // Email SMTP Settings
                'use_custom_email' => $useCustomEmail,
                'smtp_host' => $smtpHost,
                'smtp_port' => $smtpPort,
                'smtp_username' => $smtpUsername,
                'smtp_password' => !empty($smtpPassword) ? $smtpPassword : ($settings['smtp_password'] ?? ''),
                'smtp_encryption' => $smtpEncryption,
                // SMS Gateway Settings
                'use_custom_sms' => isset($_POST['use_custom_sms']) ? 1 : 0,
                'sms_enabled' => isset($_POST['sms_enabled']) ? 1 : 0,
                'sms_provider' => trim($_POST['sms_provider'] ?? ''),
                'sms_username' => trim($_POST['sms_username'] ?? ''),
                'sms_api_key' => !empty($_POST['sms_api_key']) ? trim($_POST['sms_api_key']) : ($settings['sms_api_key'] ?? ''),
                'sms_sender_id' => trim($_POST['sms_sender_id'] ?? ''),
                'sms_twilio_sid' => trim($_POST['sms_twilio_sid'] ?? ''),
                'sms_twilio_token' => !empty($_POST['sms_twilio_token']) ? trim($_POST['sms_twilio_token']) : ($settings['sms_twilio_token'] ?? ''),
                'sms_twilio_from' => trim($_POST['sms_twilio_from'] ?? '')
            ];
            
            foreach ($settingsToUpdate as $key => $value) {
                // Check if setting exists
                $stmt = $pdo->prepare("
                    SELECT COUNT(*) FROM tenant_settings 
                    WHERE academy_reference = ? AND setting_key = ?
                ");
                $stmt->execute([$currentSchool, $key]);
                $exists = $stmt->fetchColumn();
                
                if ($exists) {
                    $stmt = $pdo->prepare("
                        UPDATE tenant_settings 
                        SET setting_value = ? 
                        WHERE academy_reference = ? AND setting_key = ?
                    ");
                    $stmt->execute([$value, $currentSchool, $key]);
                } else {
                    $stmt = $pdo->prepare("
                        INSERT INTO tenant_settings (academy_reference, setting_key, setting_value)
                        VALUES (?, ?, ?)
                    ");
                    $stmt->execute([$currentSchool, $key, $value]);
                }
            }
            
            $pdo->commit();
            $success = true;
            
            // Refresh settings
            $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM tenant_settings WHERE academy_reference = ?");
            $stmt->execute([$currentSchool]);
            $settings = [];
            while ($row = $stmt->fetch()) {
                $settings[$row['setting_key']] = $row['setting_value'];
            }
            
        } catch (Exception $e) {
            $pdo->rollBack();
            $errors[] = 'Error saving settings: ' . $e->getMessage();
        }
    }
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Settings - <?= htmlspecialchars($schoolName) ?></title>
    
    <link rel="icon" type="image/svg+xml" href="../images/img-melanesys-logo.svg">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.css">
</head>
<body>
    <nav class="navbar navbar-expand-lg navbar-dark bg-primary">
        <div class="container-fluid">
            <a class="navbar-brand" href="dashboard.php">
                <i class="bi bi-speedometer2 me-2"></i>Admin Dashboard
            </a>
            <div class="navbar-nav ms-auto">
                <a class="nav-link" href="dashboard.php">
                    <i class="bi bi-arrow-left me-1"></i>Back to Dashboard
                </a>
                <a class="nav-link" href="logout.php">
                    <i class="bi bi-box-arrow-right me-1"></i>Logout
                </a>
            </div>
        </div>
    </nav>

    <!-- School Context Banner -->
    <?php if (isSuperAdmin()): ?>
    <div class="alert alert-warning mb-0 rounded-0 border-0 shadow-sm" style="border-left: 4px solid #f59e0b !important;">
        <div class="container-fluid">
            <div class="d-flex align-items-center justify-content-between">
                <div class="d-flex align-items-center gap-3">
                    <i class="bi bi-shield-check fs-4"></i>
                    <div>
                        <strong class="d-block">Super Admin Mode</strong>
                        <small>Editing settings for: <span class="badge bg-dark"><?= htmlspecialchars($schoolName ?? $currentSchool) ?></span></small>
                    </div>
                </div>
                <div>
                    <a href="../super_admin/dashboard.php" class="btn btn-sm btn-outline-dark">
                        <i class="bi bi-arrow-left me-1"></i>Back to Super Admin
                    </a>
                </div>
            </div>
        </div>
    </div>
    <?php else: ?>
    <div class="alert alert-info mb-0 rounded-0 border-0" style="border-left: 4px solid #3b82f6 !important;">
        <div class="container-fluid">
            <div class="d-flex align-items-center gap-3">
                <i class="bi bi-building fs-4"></i>
                <div>
                    <strong class="d-block"><?= htmlspecialchars($schoolName ?? 'Your School') ?></strong>
                    <small>School Code: <span class="badge bg-primary"><?= htmlspecialchars($currentSchool) ?></span></small>
                </div>
            </div>
        </div>
    </div>
    <?php endif; ?>

    <div class="container-fluid py-4">
        <div class="row">
            <div class="col-lg-10 mx-auto">
                <div class="card shadow">
                    <div class="card-header bg-primary text-white">
                        <h4 class="mb-0">
                            <i class="bi bi-sliders me-2"></i>
                            School Settings & Branding
                        </h4>
                    </div>
                    <div class="card-body">
                        <?php if ($success): ?>
                            <div class="alert alert-success alert-dismissible fade show">
                                <i class="bi bi-check-circle me-2"></i>
                                <strong>Settings saved successfully!</strong>
                                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                            </div>
                        <?php endif; ?>

                        <?php if (!empty($errors)): ?>
                            <div class="alert alert-danger alert-dismissible fade show">
                                <i class="bi bi-exclamation-triangle me-2"></i>
                                <strong>Please fix the following errors:</strong>
                                <ul class="mb-0 mt-2">
                                    <?php foreach ($errors as $error): ?>
                                        <li><?= htmlspecialchars($error) ?></li>
                                    <?php endforeach; ?>
                                </ul>
                                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                            </div>
                        <?php endif; ?>

                        <form method="POST" enctype="multipart/form-data">
                            <!-- School Information -->
                            <div class="row mb-4">
                                <div class="col-12">
                                    <h5 class="border-bottom pb-2 mb-3">
                                        <i class="bi bi-building me-2"></i>School Information
                                    </h5>
                                </div>
                                
                                <div class="col-md-6 mb-3">
                                    <label for="school_name" class="form-label">
                                        School Name <span class="text-danger">*</span>
                                    </label>
                                    <input type="text" class="form-control" id="school_name" name="school_name" 
                                           value="<?= htmlspecialchars($settings['school_name'] ?? $school['academy_name']) ?>" required>
                                </div>

                                <div class="col-md-6 mb-3">
                                    <label for="school_email" class="form-label">
                                        <i class="bi bi-envelope me-1"></i>School Email
                                    </label>
                                    <input type="email" class="form-control" id="school_email" name="school_email" 
                                           value="<?= htmlspecialchars($settings['school_email'] ?? $school['contact_email'] ?? '') ?>"
                                           placeholder="info@school.com">
                                </div>

                                <div class="col-md-6 mb-3">
                                    <label for="school_phone" class="form-label">
                                        <i class="bi bi-telephone me-1"></i>School Phone
                                    </label>
                                    <div class="input-group">
                                        <span class="input-group-text">+268</span>
                                        <input type="text" class="form-control" id="school_phone" name="school_phone" 
                                               value="<?= htmlspecialchars($settings['school_phone'] ?? $school['contact_phone'] ?? '') ?>"
                                               placeholder="7687 3207">
                                    </div>
                                </div>

                                <div class="col-md-6 mb-3">
                                    <label for="academic_year" class="form-label">
                                        <i class="bi bi-calendar me-1"></i>Academic Year
                                    </label>
                                    <input type="text" class="form-control" id="academic_year" name="academic_year" 
                                           value="<?= htmlspecialchars($settings['academic_year'] ?? date('Y') . '/' . (date('Y') + 1)) ?>"
                                           placeholder="2024/2025">
                                </div>

                                <div class="col-12 mb-3">
                                    <label for="school_address" class="form-label">
                                        <i class="bi bi-geo-alt me-1"></i>School Address
                                    </label>
                                    <textarea class="form-control" id="school_address" name="school_address" rows="2" 
                                              placeholder="Physical address"><?= htmlspecialchars($settings['school_address'] ?? $school['address'] ?? '') ?></textarea>
                                </div>
                            </div>

                            <!-- Branding -->
                            <div class="row mb-4">
                                <div class="col-12">
                                    <h5 class="border-bottom pb-2 mb-3">
                                        <i class="bi bi-palette me-2"></i>Branding & Design
                                    </h5>
                                </div>

                                <div class="col-md-12 mb-3">
                                    <label for="logo" class="form-label">
                                        <i class="bi bi-image me-1"></i>School Logo
                                    </label>
                                    
                                    <?php if (!empty($settings['logo_url'])): ?>
                                    <div class="mb-2">
                                        <img src="<?= htmlspecialchars($settings['logo_url']) ?>" 
                                             alt="Current Logo" 
                                             style="max-height: 100px; max-width: 200px;"
                                             class="border rounded p-2">
                                        <small class="d-block text-muted mt-1">Current logo</small>
                                    </div>
                                    <?php endif; ?>
                                    
                                    <input type="file" class="form-control" id="logo" name="logo" 
                                           accept="image/jpeg,image/png,image/svg+xml,image/webp">
                                    <small class="text-muted">Accepted formats: JPG, PNG, SVG, WebP. Max 2MB</small>
                                </div>

                                <div class="col-md-6 mb-3">
                                    <label for="primary_color" class="form-label">
                                        <i class="bi bi-circle-fill me-1"></i>Primary Color
                                    </label>
                                    <div class="input-group">
                                        <input type="color" class="form-control form-control-color" id="primary_color" name="primary_color" 
                                               value="<?= htmlspecialchars($settings['primary_color'] ?? '#4F46E5') ?>">
                                        <input type="text" class="form-control" id="primary_color_text"
                                               value="<?= htmlspecialchars($settings['primary_color'] ?? '#4F46E5') ?>"
                                               readonly>
                                    </div>
                                    <small class="text-muted">Main brand color used throughout the dashboard</small>
                                </div>

                                <div class="col-md-6 mb-3">
                                    <label for="secondary_color" class="form-label">
                                        <i class="bi bi-circle-fill me-1"></i>Secondary Color
                                    </label>
                                    <div class="input-group">
                                        <input type="color" class="form-control form-control-color" id="secondary_color" name="secondary_color" 
                                               value="<?= htmlspecialchars($settings['secondary_color'] ?? '#10B981') ?>">
                                        <input type="text" class="form-control" id="secondary_color_text"
                                               value="<?= htmlspecialchars($settings['secondary_color'] ?? '#10B981') ?>"
                                               readonly>
                                    </div>
                                    <small class="text-muted">Accent color for buttons and highlights</small>
                                </div>
                            </div>

                            <!-- Payment & Pricing Settings -->
                            <div class="row mb-4">
                                <div class="col-12">
                                    <h5 class="border-bottom pb-2 mb-3">
                                        <i class="bi bi-currency-dollar me-2"></i>Payment & Pricing
                                    </h5>
                                </div>
                                
                                <div class="col-md-4 mb-3">
                                    <label for="subject_fee" class="form-label">
                                        <i class="bi bi-cash-stack me-1"></i>Subject Fee (Per Month)
                                    </label>
                                    <div class="input-group">
                                        <span class="input-group-text"><?= htmlspecialchars($settings['currency_symbol'] ?? 'E') ?></span>
                                        <input type="number" class="form-control" id="subject_fee" name="subject_fee" 
                                               value="<?= htmlspecialchars($settings['subject_fee'] ?? '350') ?>"
                                               step="0.01" min="0" required>
                                    </div>
                                    <small class="text-muted">Price charged per subject per month</small>
                                </div>
                                
                                <div class="col-md-4 mb-3">
                                    <label for="yearly_discount" class="form-label">
                                        <i class="bi bi-tag me-1"></i>Yearly Discount (%)
                                    </label>
                                    <div class="input-group">
                                        <input type="number" class="form-control" id="yearly_discount" name="yearly_discount" 
                                               value="<?= htmlspecialchars($settings['yearly_discount'] ?? '10') ?>"
                                               step="1" min="0" max="100" required>
                                        <span class="input-group-text">%</span>
                                    </div>
                                    <small class="text-muted">Discount when parents pay for 12 months</small>
                                </div>
                                
                                <div class="col-md-4 mb-3">
                                    <label for="currency_symbol" class="form-label">
                                        <i class="bi bi-currency-exchange me-1"></i>Currency Symbol
                                    </label>
                                    <select class="form-select" id="currency_symbol" name="currency_symbol">
                                        <?php
                                        $currencies = [
                                            'E' => 'E (Emalangeni - SZL)',
                                            'R' => 'R (Rand - ZAR)',
                                            '$' => '$ (Dollar - USD)',
                                            '€' => '€ (Euro - EUR)',
                                            '£' => '£ (Pound - GBP)',
                                            '₦' => '₦ (Naira - NGN)'
                                        ];
                                        $current_currency = $settings['currency_symbol'] ?? 'E';
                                        foreach ($currencies as $symbol => $label):
                                        ?>
                                            <option value="<?= $symbol ?>" <?= $current_currency === $symbol ? 'selected' : '' ?>>
                                                <?= htmlspecialchars($label) ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                    <small class="text-muted">Used in receipts and parent portal</small>
                                </div>
                                
                                <div class="col-md-4 mb-3">
                                    <label for="minimum_required_subjects" class="form-label">
                                        <i class="bi bi-book-half me-1"></i>Minimum Required Subjects
                                    </label>
                                    <input type="number" class="form-control" id="minimum_required_subjects" 
                                           name="minimum_required_subjects" 
                                           value="<?= htmlspecialchars($settings['minimum_required_subjects'] ?? '0') ?>"
                                           min="0" max="20">
                                    <small class="text-muted">Minimum subjects for student registration (0 = optional)</small>
                                </div>
                                
                                <div class="col-12">
                                    <div class="alert alert-success">
                                        <i class="bi bi-calculator me-2"></i>
                                        <strong>Pricing Preview (Parent View):</strong><br>
                                        <div class="row mt-2">
                                            <div class="col-md-6">
                                                <strong>Monthly (3 subjects):</strong> 
                                                <span class="fs-5"><?= $settings['currency_symbol'] ?? 'E' ?><?= number_format(($settings['subject_fee'] ?? 350) * 3, 2) ?></span>
                                            </div>
                                            <div class="col-md-6">
                                                <strong>Yearly (3 subjects):</strong> 
                                                <span class="fs-5"><?= $settings['currency_symbol'] ?? 'E' ?><?= number_format((($settings['subject_fee'] ?? 350) * 3 * 12) * (1 - (($settings['yearly_discount'] ?? 10) / 100)), 2) ?></span>
                                                <span class="badge bg-success ms-2">Save <?= number_format((($settings['subject_fee'] ?? 350) * 3 * 12) * (($settings['yearly_discount'] ?? 10) / 100), 2) ?>!</span>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Card Payment Configuration -->
                            <div class="row mb-4">
                                <div class="col-12">
                                    <h5 class="border-bottom pb-2 mb-3">
                                        <i class="bi bi-credit-card me-2"></i>Card Payment Gateway
                                    </h5>
                                </div>
                                
                                <div class="col-md-12 mb-3">
                                    <div class="form-check form-switch">
                                        <input class="form-check-input" type="checkbox" id="card_payments_enabled" 
                                               name="card_payments_enabled" <?= ($settings['card_payments_enabled'] ?? 0) ? 'checked' : '' ?>>
                                        <label class="form-check-label" for="card_payments_enabled">
                                            <strong>Enable Card Payments for Parents</strong>
                                        </label>
                                    </div>
                                    <small class="text-muted">Allow parents to pay using credit/debit cards</small>
                                </div>
                                
                                <div id="card-payment-settings" style="display: <?= ($settings['card_payments_enabled'] ?? 0) ? 'block' : 'none' ?>;">
                                    <div class="col-md-6 mb-3">
                                        <label for="payment_gateway" class="form-label">
                                            <i class="bi bi-building me-1"></i>Payment Gateway
                                        </label>
                                        <select class="form-select" id="payment_gateway" name="payment_gateway">
                                            <?php
                                            $gateways = [
                                                'paygate' => 'PayGate (South Africa)',
                                                'payfast' => 'PayFast (South Africa)',
                                                'stripe' => 'Stripe (International)',
                                                'flutterwave' => 'Flutterwave (Africa)'
                                            ];
                                            $current_gateway = $settings['payment_gateway'] ?? 'paygate';
                                            foreach ($gateways as $value => $label):
                                            ?>
                                                <option value="<?= $value ?>" <?= $current_gateway === $value ? 'selected' : '' ?>>
                                                    <?= htmlspecialchars($label) ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>
                                    
                                    <div class="col-md-6 mb-3">
                                        <label for="paygate_id" class="form-label">
                                            <i class="bi bi-key me-1"></i>PayGate ID / Merchant ID
                                        </label>
                                        <input type="text" class="form-control" id="paygate_id" name="paygate_id" 
                                               value="<?= htmlspecialchars($settings['paygate_id'] ?? '') ?>"
                                               placeholder="10011072130">
                                        <small class="text-muted">Your PayGate ID from the merchant portal</small>
                                    </div>
                                    
                                    <div class="col-md-6 mb-3">
                                        <label for="paygate_secret" class="form-label">
                                            <i class="bi bi-shield-lock me-1"></i>Secret Key / Encryption Key
                                        </label>
                                        <input type="password" class="form-control" id="paygate_secret" name="paygate_secret" 
                                               value="<?= htmlspecialchars($settings['paygate_secret'] ?? '') ?>"
                                               placeholder="Enter secret key">
                                        <small class="text-muted">Keep this secret! Never share publicly</small>
                                    </div>
                                    
                                    <div class="col-12">
                                        <div class="alert alert-warning">
                                            <i class="bi bi-exclamation-triangle me-2"></i>
                                            <strong>Important:</strong> To enable card payments, you need:
                                            <ol class="mb-0 mt-2">
                                                <li>Register with a payment gateway (PayGate, PayFast, etc.)</li>
                                                <li>Complete merchant verification</li>
                                                <li>Get your Merchant ID and Secret Key</li>
                                                <li>Enter credentials above</li>
                                                <li>Enable the toggle</li>
                                            </ol>
                                        </div>
                                    </div>
                                    
                                    <div class="col-12">
                                        <div class="card bg-light">
                                            <div class="card-body">
                                                <h6><i class="bi bi-info-circle me-2"></i>Supported Payment Gateways</h6>
                                                <ul class="mb-0">
                                                    <li><strong>PayGate:</strong> https://www.paygate.co.za (Recommended for South Africa)</li>
                                                    <li><strong>PayFast:</strong> https://www.payfast.co.za (South Africa)</li>
                                                    <li><strong>Stripe:</strong> https://stripe.com (International)</li>
                                                    <li><strong>Flutterwave:</strong> https://flutterwave.com (African markets)</li>
                                                </ul>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Moodle API Integration (Super Admin Only) -->
                            <div class="row mb-4">
                                <div class="col-12">
                                    <h5 class="border-bottom pb-2 mb-3">
                                        <i class="bi bi-mortarboard me-2"></i>Moodle API Integration
                                        <?php if (isSuperAdmin()): ?>
                                            <span class="badge bg-warning text-dark">Super Admin Only</span>
                                        <?php endif; ?>
                                    </h5>
                                </div>

                                <?php if (!isSuperAdmin()): ?>
                                    <div class="col-md-12">
                                        <div class="alert alert-warning">
                                            <i class="bi bi-shield-lock me-2"></i>
                                            <strong>Restricted Access:</strong> Moodle API integration can only be configured by super administrators due to its critical nature.
                                            <?php if ($settings['moodle_enabled'] ?? 0): ?>
                                                <hr class="my-2">
                                                <small class="d-block mt-2">
                                                    <i class="bi bi-info-circle me-1"></i>
                                                    <strong>Current Status:</strong> Moodle integration is <span class="badge bg-success">ENABLED</span> for this school.
                                                    <br>
                                                    Moodle URL: <code><?= htmlspecialchars($settings['moodle_url'] ?? 'Not configured') ?></code>
                                                </small>
                                            <?php else: ?>
                                                <hr class="my-2">
                                                <small class="d-block mt-2">
                                                    <i class="bi bi-info-circle me-1"></i>
                                                    <strong>Current Status:</strong> Moodle integration is <span class="badge bg-secondary">DISABLED</span>.
                                                </small>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                <?php else: ?>
                                    <div class="col-md-12 mb-3">
                                        <div class="form-check form-switch">
                                            <input class="form-check-input" type="checkbox" id="moodle_enabled" name="moodle_enabled" 
                                                   <?= ($settings['moodle_enabled'] ?? 0) ? 'checked' : '' ?>>
                                            <label class="form-check-label" for="moodle_enabled">
                                                <strong>Enable Moodle Integration</strong>
                                                <small class="d-block text-muted">Sync students with your Moodle LMS</small>
                                            </label>
                                        </div>
                                    </div>

                                <div id="moodle-settings" style="display: <?= ($settings['moodle_enabled'] ?? 0) ? 'block' : 'none' ?>;">
                                    <div class="col-md-12 mb-3">
                                        <label for="moodle_url" class="form-label">
                                            <i class="bi bi-link-45deg me-1"></i>Moodle URL
                                        </label>
                                        <input type="url" class="form-control" id="moodle_url" name="moodle_url" 
                                               value="<?= htmlspecialchars($settings['moodle_url'] ?? '') ?>"
                                               placeholder="https://your-moodle-site.com">
                                        <small class="text-muted">Your Moodle site URL (e.g., https://lms.yourschool.com)</small>
                                    </div>

                                    <div class="col-md-12 mb-3">
                                        <label for="moodle_token" class="form-label">
                                            <i class="bi bi-key me-1"></i>Moodle API Token
                                        </label>
                                        <input type="password" class="form-control" id="moodle_token" name="moodle_token" 
                                               value="<?= htmlspecialchars($settings['moodle_token'] ?? '') ?>"
                                               placeholder="Enter your Moodle web service token">
                                        <small class="text-muted">Web service token from Moodle Site Administration</small>
                                    </div>

                                    <div class="col-md-12 mb-3">
                                        <label for="moodle_service_name" class="form-label">
                                            Service Name
                                        </label>
                                        <input type="text" class="form-control" id="moodle_service_name" name="moodle_service_name" 
                                               value="<?= htmlspecialchars($settings['moodle_service_name'] ?? 'moodle_mobile_app') ?>">
                                        <small class="text-muted">Usually "moodle_mobile_app" (default)</small>
                                    </div>

                                    <div class="col-md-12">
                                        <div class="alert alert-info">
                                            <i class="bi bi-info-circle me-2"></i>
                                            <strong>How to get your Moodle API token:</strong><br>
                                            <small>
                                                1. Login to Moodle as admin<br>
                                                2. Go to Site Administration → Plugins → Web services → Manage tokens<br>
                                                3. Create a new token for your user<br>
                                                4. Copy and paste the token here
                                            </small>
                                        </div>
                                    </div>

                                    <div class="col-md-12">
                                        <button type="button" class="btn btn-outline-primary" id="test-moodle-btn">
                                            <i class="bi bi-arrow-repeat me-1"></i>
                                            Test Moodle Connection
                                        </button>
                                        <div id="moodle-test-result" class="mt-3" style="display: none;"></div>
                                    </div>
                                </div>
                                <?php endif; ?>
                            </div>

                            <!-- Email Settings -->
                            <div class="row mb-4">
                                <div class="col-12">
                                    <h5 class="border-bottom pb-2 mb-3">
                                        <i class="bi bi-envelope me-2"></i>Email Settings
                                    </h5>
                                </div>

                                <div class="col-md-12 mb-3">
                                    <div class="form-check form-switch">
                                        <input class="form-check-input" type="checkbox" id="use_custom_email" name="use_custom_email" 
                                               <?= ($settings['use_custom_email'] ?? 0) ? 'checked' : '' ?>>
                                        <label class="form-check-label" for="use_custom_email">
                                            <strong>Use Custom Email Server</strong>
                                            <small class="d-block text-muted">By default, system emails use info@melanegroup.com</small>
                                        </label>
                                    </div>
                                </div>

                                <div class="col-md-12 mb-3">
                                    <label class="form-label">
                                        Default Email Address <span class="badge bg-secondary">Read-only</span>
                                    </label>
                                    <input type="text" class="form-control" value="info@melanegroup.com" disabled>
                                    <small class="text-muted">System default - cannot be modified by school admins</small>
                                </div>

                                <div id="custom-email-settings" style="display: <?= ($settings['use_custom_email'] ?? 0) ? 'block' : 'none' ?>;">
                                    <div class="col-md-12 mb-3">
                                        <div class="alert alert-warning">
                                            <i class="bi bi-exclamation-triangle me-2"></i>
                                            <strong>Custom Email Configuration:</strong> Configure your own SMTP server to send emails from your domain.
                                        </div>
                                    </div>

                                    <div class="col-md-6 mb-3">
                                        <label for="smtp_host" class="form-label">
                                            <i class="bi bi-hdd-network me-1"></i>SMTP Host
                                        </label>
                                        <input type="text" class="form-control" id="smtp_host" name="smtp_host" 
                                               value="<?= htmlspecialchars($settings['smtp_host'] ?? '') ?>"
                                               placeholder="smtp.gmail.com">
                                        <small class="text-muted">Your email server hostname</small>
                                    </div>

                                    <div class="col-md-6 mb-3">
                                        <label for="smtp_port" class="form-label">
                                            <i class="bi bi-arrow-left-right me-1"></i>SMTP Port
                                        </label>
                                        <input type="number" class="form-control" id="smtp_port" name="smtp_port" 
                                               value="<?= htmlspecialchars($settings['smtp_port'] ?? '587') ?>"
                                               placeholder="587">
                                        <small class="text-muted">Usually 587 for TLS or 465 for SSL</small>
                                    </div>

                                    <div class="col-md-6 mb-3">
                                        <label for="smtp_username" class="form-label">
                                            <i class="bi bi-person me-1"></i>SMTP Username
                                        </label>
                                        <input type="email" class="form-control" id="smtp_username" name="smtp_username" 
                                               value="<?= htmlspecialchars($settings['smtp_username'] ?? '') ?>"
                                               placeholder="your-email@yourdomain.com">
                                        <small class="text-muted">Email address for authentication</small>
                                    </div>

                                    <div class="col-md-6 mb-3">
                                        <label for="smtp_password" class="form-label">
                                            <i class="bi bi-lock me-1"></i>SMTP Password
                                        </label>
                                        <input type="password" class="form-control" id="smtp_password" name="smtp_password" 
                                               placeholder="<?= !empty($settings['smtp_password']) ? '••••••••' : 'Enter password' ?>">
                                        <small class="text-muted">Email account password or app password</small>
                                    </div>

                                    <div class="col-md-6 mb-3">
                                        <label for="smtp_encryption" class="form-label">
                                            <i class="bi bi-shield-check me-1"></i>Encryption
                                        </label>
                                        <select class="form-select" id="smtp_encryption" name="smtp_encryption">
                                            <option value="tls" <?= ($settings['smtp_encryption'] ?? 'tls') === 'tls' ? 'selected' : '' ?>>TLS (Recommended)</option>
                                            <option value="ssl" <?= ($settings['smtp_encryption'] ?? 'tls') === 'ssl' ? 'selected' : '' ?>>SSL</option>
                                            <option value="none" <?= ($settings['smtp_encryption'] ?? 'tls') === 'none' ? 'selected' : '' ?>>None</option>
                                        </select>
                                    </div>

                                    <div class="col-md-12">
                                        <hr class="my-3">
                                        <label for="test_email_address" class="form-label">
                                            <i class="bi bi-envelope-check me-1"></i>Test Email Address
                                        </label>
                                        <div class="input-group mb-3">
                                            <input type="email" class="form-control" id="test_email_address" 
                                                   placeholder="your-email@example.com" 
                                                   value="<?= htmlspecialchars($_SESSION['admin_email'] ?? '') ?>">
                                            <button type="button" class="btn btn-outline-primary" id="test-email-btn">
                                                <i class="bi bi-send me-1"></i>
                                                Send Test Email
                                            </button>
                                        </div>
                                        <small class="text-muted">Enter an email address to receive a test email</small>
                                        <div id="email-test-result" class="mt-3" style="display: none;"></div>
                                    </div>
                                </div>

                                <div class="col-md-6 mb-3">
                                    <label for="email_from_name" class="form-label">
                                        Email From Name
                                    </label>
                                    <input type="text" class="form-control" id="email_from_name" name="email_from_name" 
                                           value="<?= htmlspecialchars($settings['email_from_name'] ?? $schoolName) ?>"
                                           placeholder="School Name">
                                    <small class="text-muted">Name that appears in sent emails</small>
                                </div>

                                <div class="col-md-12 mb-3">
                                    <label for="email_signature" class="form-label">
                                        Email Signature
                                    </label>
                                    <textarea class="form-control" id="email_signature" name="email_signature" rows="3" 
                                              placeholder="Best regards,&#10;Your School Team"><?= htmlspecialchars($settings['email_signature'] ?? '') ?></textarea>
                                    <small class="text-muted">Signature added to all automated emails</small>
                                </div>
                            </div>

                            <!-- SMS Gateway Settings -->
                            <div class="row mt-4">
                                <div class="col-12">
                                    <h4 class="mb-3">
                                        <i class="bi bi-phone text-primary me-2"></i>
                                        SMS Gateway Configuration
                                    </h4>
                                </div>

                                <!-- Use Custom SMS Toggle -->
                                <div class="col-md-12 mb-3">
                                    <div class="form-check form-switch">
                                        <input class="form-check-input" type="checkbox" id="use_custom_sms" name="use_custom_sms" 
                                               <?= ($settings['use_custom_sms'] ?? 0) ? 'checked' : '' ?>
                                               onchange="toggleSMSSettings()">
                                        <label class="form-check-label" for="use_custom_sms">
                                            <strong>Use Custom SMS Gateway</strong>
                                            <br><small class="text-muted">Configure your own SMS gateway. If disabled, the super admin's default gateway will be used.</small>
                                        </label>
                                    </div>
                                </div>

                                <!-- Custom SMS Settings (Hidden by default) -->
                                <div id="custom-sms-settings" style="display: <?= ($settings['use_custom_sms'] ?? 0) ? 'block' : 'none' ?>;">
                                    
                                    <!-- Enable SMS -->
                                    <div class="col-md-12 mb-3">
                                        <div class="form-check form-switch">
                                            <input class="form-check-input" type="checkbox" id="sms_enabled" name="sms_enabled" 
                                                   <?= ($settings['sms_enabled'] ?? 0) ? 'checked' : '' ?>>
                                            <label class="form-check-label" for="sms_enabled">
                                                <strong>Enable SMS for this School</strong>
                                                <br><small class="text-muted">Send verification codes and credentials via SMS</small>
                                            </label>
                                        </div>
                                    </div>

                                    <div class="alert alert-info">
                                        <strong><i class="bi bi-info-circle"></i> SMS Gateway Options:</strong>
                                        <ul class="mb-0 mt-2">
                                            <li><strong>Africa's Talking:</strong> Recommended for African markets (~E0.20-0.30 per SMS)</li>
                                            <li><strong>Twilio:</strong> Global provider with support for multiple countries</li>
                                        </ul>
                                    </div>

                                    <!-- Provider Selection -->
                                    <div class="col-md-12 mb-3">
                                        <label class="form-label">SMS Provider</label>
                                        <select class="form-select" id="sms_provider" name="sms_provider" onchange="toggleSMSProvider()">
                                            <option value="africastalking" <?= ($settings['sms_provider'] ?? 'africastalking') == 'africastalking' ? 'selected' : '' ?>>Africa's Talking</option>
                                            <option value="twilio" <?= ($settings['sms_provider'] ?? '') == 'twilio' ? 'selected' : '' ?>>Twilio</option>
                                        </select>
                                    </div>

                                    <!-- Africa's Talking Settings -->
                                    <div id="africastalking-settings" style="display: <?= ($settings['sms_provider'] ?? 'africastalking') == 'africastalking' ? 'block' : 'none' ?>;">
                                        <div class="row">
                                            <div class="col-md-6 mb-3">
                                                <label for="sms_username" class="form-label">Username</label>
                                                <input type="text" class="form-control" id="sms_username" name="sms_username" 
                                                       value="<?= htmlspecialchars($settings['sms_username'] ?? '') ?>"
                                                       placeholder="sandbox or your username">
                                                <small class="text-muted">From Africa's Talking dashboard</small>
                                            </div>
                                            <div class="col-md-6 mb-3">
                                                <label for="sms_sender_id" class="form-label">Sender ID (Optional)</label>
                                                <input type="text" class="form-control" id="sms_sender_id" name="sms_sender_id" 
                                                       value="<?= htmlspecialchars($settings['sms_sender_id'] ?? '') ?>"
                                                       placeholder="SCHOOL" maxlength="11">
                                                <small class="text-muted">Max 11 characters</small>
                                            </div>
                                        </div>
                                        <div class="row">
                                            <div class="col-md-12 mb-3">
                                                <label for="sms_api_key" class="form-label">API Key</label>
                                                <input type="password" class="form-control" id="sms_api_key" name="sms_api_key" 
                                                       value="<?= htmlspecialchars($settings['sms_api_key'] ?? '') ?>"
                                                       placeholder="atsk_...">
                                                <small class="text-muted">Keep this secret! Leave blank to keep current value.</small>
                                            </div>
                                        </div>
                                    </div>

                                    <!-- Twilio Settings -->
                                    <div id="twilio-settings" style="display: <?= ($settings['sms_provider'] ?? '') == 'twilio' ? 'block' : 'none' ?>;">
                                        <div class="row">
                                            <div class="col-md-6 mb-3">
                                                <label for="sms_twilio_sid" class="form-label">Account SID</label>
                                                <input type="text" class="form-control" id="sms_twilio_sid" name="sms_twilio_sid" 
                                                       value="<?= htmlspecialchars($settings['sms_twilio_sid'] ?? '') ?>"
                                                       placeholder="ACxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx">
                                            </div>
                                            <div class="col-md-6 mb-3">
                                                <label for="sms_twilio_from" class="form-label">From Number</label>
                                                <input type="text" class="form-control" id="sms_twilio_from" name="sms_twilio_from" 
                                                       value="<?= htmlspecialchars($settings['sms_twilio_from'] ?? '') ?>"
                                                       placeholder="+268XXXXXXXX">
                                            </div>
                                        </div>
                                        <div class="row">
                                            <div class="col-md-12 mb-3">
                                                <label for="sms_twilio_token" class="form-label">Auth Token</label>
                                                <input type="password" class="form-control" id="sms_twilio_token" name="sms_twilio_token" 
                                                       value="<?= htmlspecialchars($settings['sms_twilio_token'] ?? '') ?>"
                                                       placeholder="Your auth token">
                                                <small class="text-muted">Keep this secret! Leave blank to keep current value.</small>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Save Button -->
                            <div class="row">
                                <div class="col-12">
                                    <hr>
                                    <div class="d-flex gap-2">
                                        <button type="submit" class="btn btn-primary btn-lg">
                                            <i class="bi bi-save me-1"></i>
                                            Save Settings
                                        </button>
                                        <a href="dashboard.php" class="btn btn-secondary btn-lg">
                                            <i class="bi bi-x-circle me-1"></i>
                                            Cancel
                                        </a>
                                    </div>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Quick Info Card -->
                <div class="card shadow mt-4">
                    <div class="card-header bg-light">
                        <h5 class="mb-0">
                            <i class="bi bi-info-circle me-2"></i>Settings Information
                        </h5>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-6">
                                <h6>School Reference Code</h6>
                                <p class="text-muted mb-3">
                                    <span class="badge bg-primary fs-6"><?= htmlspecialchars($currentSchool) ?></span>
                                </p>
                                <small class="text-muted">This code cannot be changed and is used throughout the system.</small>
                            </div>
                            <div class="col-md-6">
                                <h6>Branding Guidelines</h6>
                                <ul class="small text-muted">
                                    <li>Logo should be square or landscape format</li>
                                    <li>Recommended size: 200x200px minimum</li>
                                    <li>Colors will apply to navigation and buttons</li>
                                    <li>Changes take effect immediately</li>
                                </ul>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Sync color picker with text input
        document.getElementById('primary_color').addEventListener('input', function(e) {
            document.getElementById('primary_color_text').value = e.target.value;
        });
        document.getElementById('secondary_color').addEventListener('input', function(e) {
            document.getElementById('secondary_color_text').value = e.target.value;
        });

        // Toggle Moodle settings visibility (Super Admin Only)
        const moodleEnabledToggle = document.getElementById('moodle_enabled');
        if (moodleEnabledToggle) {
            moodleEnabledToggle.addEventListener('change', function() {
                const moodleSettings = document.getElementById('moodle-settings');
                if (this.checked) {
                    moodleSettings.style.display = 'block';
                } else {
                    moodleSettings.style.display = 'none';
                }
            });
        }

        // Toggle Card Payment settings visibility
        const cardPaymentsToggle = document.getElementById('card_payments_enabled');
        if (cardPaymentsToggle) {
            cardPaymentsToggle.addEventListener('change', function() {
                const cardSettings = document.getElementById('card-payment-settings');
                if (this.checked) {
                    cardSettings.style.display = 'block';
                } else {
                    cardSettings.style.display = 'none';
                }
            });
        }

        // Toggle Custom Email settings visibility
        document.getElementById('use_custom_email').addEventListener('change', function() {
            const emailSettings = document.getElementById('custom-email-settings');
            if (this.checked) {
                emailSettings.style.display = 'block';
            } else {
                emailSettings.style.display = 'none';
            }
        });

        // Test Moodle Connection (Super Admin Only)
        const testMoodleBtn = document.getElementById('test-moodle-btn');
        if (testMoodleBtn) {
            testMoodleBtn.addEventListener('click', function() {
                const btn = this;
                const resultDiv = document.getElementById('moodle-test-result');
                const moodleUrl = document.getElementById('moodle_url').value;
                const moodleToken = document.getElementById('moodle_token').value;
            
            if (!moodleUrl || !moodleToken) {
                showResult(resultDiv, false, 'Please enter both Moodle URL and API token');
                return;
            }
            
            // Show loading state
            btn.disabled = true;
            btn.innerHTML = '<span class="spinner-border spinner-border-sm me-1"></span>Testing...';
            resultDiv.style.display = 'none';
            
            // Send AJAX request
            fetch('settings.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: new URLSearchParams({
                    action: 'test_moodle',
                    moodle_url: moodleUrl,
                    moodle_token: moodleToken
                })
            })
            .then(response => response.json())
            .then(data => {
                let message = data.message;
                if (data.success && data.site_name) {
                    message += '<br><small>Site: ' + data.site_name + ' (Version: ' + data.version + ')</small>';
                }
                showResult(resultDiv, data.success, message);
            })
            .catch(error => {
                showResult(resultDiv, false, 'Error testing connection: ' + error);
            })
            .finally(() => {
                btn.disabled = false;
                btn.innerHTML = '<i class="bi bi-arrow-repeat me-1"></i>Test Moodle Connection';
            });
            });
        }

        // Test Email Connection
        document.getElementById('test-email-btn').addEventListener('click', function() {
            const btn = this;
            const resultDiv = document.getElementById('email-test-result');
            const smtpHost = document.getElementById('smtp_host').value;
            const smtpPort = document.getElementById('smtp_port').value;
            const smtpUsername = document.getElementById('smtp_username').value;
            const smtpPassword = document.getElementById('smtp_password').value;
            const smtpEncryption = document.getElementById('smtp_encryption').value;
            const testEmail = document.getElementById('test_email_address').value;
            
            if (!smtpHost || !smtpUsername || !smtpPassword) {
                showResult(resultDiv, false, 'Please configure all SMTP settings first');
                return;
            }
            
            if (!testEmail) {
                showResult(resultDiv, false, 'Please enter a test email address');
                return;
            }
            
            // Show loading state
            btn.disabled = true;
            btn.innerHTML = '<span class="spinner-border spinner-border-sm me-1"></span>Sending...';
            resultDiv.style.display = 'none';
            
            // Send AJAX request
            fetch('settings.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: new URLSearchParams({
                    action: 'test_email',
                    smtp_host: smtpHost,
                    smtp_port: smtpPort,
                    smtp_username: smtpUsername,
                    smtp_password: smtpPassword,
                    smtp_encryption: smtpEncryption,
                    test_email_address: testEmail
                })
            })
            .then(response => response.json())
            .then(data => {
                showResult(resultDiv, data.success, data.message);
            })
            .catch(error => {
                showResult(resultDiv, false, 'Error sending test email: ' + error);
            })
            .finally(() => {
                btn.disabled = false;
                btn.innerHTML = '<i class="bi bi-send me-1"></i>Send Test Email';
            });
        });

        // Helper function to show test results
        function showResult(element, success, message) {
            element.className = 'alert ' + (success ? 'alert-success' : 'alert-danger');
            element.innerHTML = '<i class="bi bi-' + (success ? 'check-circle' : 'exclamation-triangle') + ' me-2"></i>' + message;
            element.style.display = 'block';
            
            // Auto-hide after 10 seconds
            setTimeout(() => {
                element.style.display = 'none';
            }, 10000);
        }

        // Toggle SMS settings visibility
        function toggleSMSSettings() {
            const checkbox = document.getElementById('use_custom_sms');
            const settings = document.getElementById('custom-sms-settings');
            settings.style.display = checkbox.checked ? 'block' : 'none';
        }

        // Toggle SMS provider settings
        function toggleSMSProvider() {
            const provider = document.getElementById('sms_provider').value;
            document.getElementById('africastalking-settings').style.display = provider === 'africastalking' ? 'block' : 'none';
            document.getElementById('twilio-settings').style.display = provider === 'twilio' ? 'block' : 'none';
        }
    </script>
</body>
</html>

