<?php
/**
 * Admin interface for restoring suspended students with selective subject enrollment
 */

require_once '../config.php';
require_once '../includes/functions.php';

// Check admin authentication
requireAdminLogin();

$doExport = isset($_GET['export']) && $_GET['export'] === 'csv';

$message = '';
$messageType = '';

// Filters and pagination
$q = $_GET['q'] ?? '';
$gradeIdFilter = isset($_GET['grade_id']) ? (int)$_GET['grade_id'] : 0;
$fromDate = $_GET['from'] ?? '';
$toDate = $_GET['to'] ?? '';
$page = max(1, (int)($_GET['page'] ?? 1));
$pageSize = 12;

// Handle restore request
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['restore_student'])) {
    $studentId = (int)$_POST['student_id'];
    $selectedSubjects = isset($_POST['subjects']) ? array_map('intval', $_POST['subjects']) : [];
    
    if (empty($selectedSubjects)) {
        $message = 'Please select at least one subject to enroll the student in.';
        $messageType = 'danger';
    } else {
        $result = restoreStudent($studentId, $selectedSubjects);
        
        if ($result['success']) {
            $message = "Student restored successfully! ";
            $message .= "Moodle: " . ($result['details']['moodle_restore'] ? 'Restored' : 'Failed') . " | ";
            $message .= "Subjects Enrolled: {$result['details']['subjects_enrolled']} | ";
            $message .= "Subjects Failed: {$result['details']['subjects_failed']}";
            $messageType = 'success';
        } else {
            $message = $result['message'];
            $messageType = 'danger';
        }
    }
}

// Handle permanent deletion request
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_permanently'])) {
    $studentId = (int)$_POST['student_id'];
    
    $result = permanentlyDeleteStudent($studentId);
    
    if ($result) {
        $message = "Student permanently deleted from the system successfully!";
        $messageType = 'success';
    } else {
        $message = "Failed to permanently delete student. Please check the error logs for details.";
        $messageType = 'danger';
    }
}

$grades = getGrades();
$subjects = getAllSubjects();

// Get suspended students (filtered + paginated)
$totalRows = 0;
$suspendedStudents = getSuspendedStudentsFiltered([
    'q' => $q,
    'grade_id' => $gradeIdFilter ?: null,
    'from' => $fromDate ?: null,
    'to' => $toDate ?: null,
], $page, $pageSize, $totalRows);
$totalPages = max(1, (int)ceil($totalRows / $pageSize));

// CSV export if requested
if ($doExport) {
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="suspended_students.csv"');
    $out = fopen('php://output', 'w');
    fputcsv($out, ['id','first_name','last_name','email','username','grade','deleted_at','days_remaining']);
    foreach ($suspendedStudents as $s) {
        fputcsv($out, [
            $s['id'], $s['first_name'], $s['last_name'], $s['email'], $s['username'], $s['grade_name'], $s['deleted_at'], $s['days_remaining']
        ]);
    }
    fclose($out);
    exit;
}

// Handle bulk restore request
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['bulk_restore'])) {
    $selectedIds = isset($_POST['selected']) && is_array($_POST['selected']) ? array_map('intval', $_POST['selected']) : [];
    $bulkSubjects = isset($_POST['bulk_subjects']) && is_array($_POST['bulk_subjects']) ? array_map('intval', $_POST['bulk_subjects']) : [];
    $useOriginals = isset($_POST['use_originals']) && $_POST['use_originals'] == '1';
    if (empty($selectedIds)) {
        $message = 'Please select at least one student for bulk restore.';
        $messageType = 'warning';
    } elseif (!$useOriginals && empty($bulkSubjects)) {
        $message = 'Please select at least one subject in the bulk subject picker.';
        $messageType = 'warning';
    } else {
        $successCount = 0;
        $failCount = 0;
        $skippedNoOriginals = 0;
        foreach ($selectedIds as $sid) {
            $subjectsToUse = $bulkSubjects;
            if ($useOriginals) {
                $orig = getOriginalSubjectIds($sid);
                if (!empty($orig)) {
                    $subjectsToUse = $orig;
                } else {
                    $skippedNoOriginals++;
                    $failCount++;
                    continue;
                }
            }
            $res = restoreStudent($sid, $subjectsToUse);
            if (is_array($res) && !empty($res['success'])) {
                $successCount++;
            } else {
                $failCount++;
            }
        }
        $extra = $useOriginals && $skippedNoOriginals > 0 ? " ({$skippedNoOriginals} had no original subjects)" : '';
        $message = "Bulk restore completed: {$successCount} succeeded, {$failCount} failed{$extra}.";
        $messageType = $failCount > 0 ? 'warning' : 'success';
    }
}
$subjects = getAllSubjects();
?>

<?php $pageTitle = 'Restore Suspended Students - '.APP_NAME.' Admin'; $hideRestoreLink = true; include __DIR__.'/partials/header.php'; ?>
    <style>
        /* Modern top navbar */
        .top-navbar {
            background: linear-gradient(135deg, #1a1a2e 0%, #16213e 50%, #0f3460 100%);
            border-bottom: 3px solid #007bff;
        }
        .top-navbar .navbar-brand {
            font-weight: 700;
            color: #ffffff;
            letter-spacing: .5px;
        }
        .top-navbar .navbar-brand img {
            height: 28px;
            width: 28px;
            margin-right: .5rem;
        }
        .top-navbar .nav-link { color: #b8c5d1; }
        .top-navbar .nav-link:hover { color: #ffffff; }
        body {
            background: radial-gradient(1200px 600px at 10% -10%, rgba(0, 123, 255, 0.08), transparent 60%),
                        radial-gradient(1200px 600px at 110% 10%, rgba(32, 201, 151, 0.08), transparent 60%),
                        #f6f8fb;
        }
        .container-max {
            max-width: 1320px;
        }
        .card-modern {
            border: none;
            border-radius: 15px;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
            transition: transform .15s ease, box-shadow .15s ease;
        }
        .card-modern:hover { transform: translateY(-2px); box-shadow: 0 8px 16px rgba(0,0,0,.12); }
        .card-header-modern {
            background: linear-gradient(90deg, rgba(220,53,69,.12), rgba(220,53,69,.05));
            border-bottom: 1px solid rgba(220,53,69,.2);
        }
        .brand-underline {
            border-bottom: 3px solid #007bff;
            display: inline-block;
        }
        
        /* Modern Professional Sidebar Styles */
        .sidebar-modern {
            background: linear-gradient(180deg, #1a1a2e 0%, #16213e 50%, #0f3460 100%);
            min-height: 100vh;
            border-right: 3px solid #007bff;
            position: relative;
            overflow: hidden;
            display: flex;
            flex-direction: column;
        }
        
        .sidebar-modern::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 3px;
            background: linear-gradient(90deg, #007bff, #28a745, #ffc107, #dc3545, #6f42c1);
            animation: gradientShift 3s ease-in-out infinite;
        }
        
        @keyframes gradientShift {
            0%, 100% { opacity: 0.8; }
            50% { opacity: 1; }
        }
        
        .sidebar-brand {
            padding: 1rem;
            border-bottom: 1px solid rgba(255, 255, 255, 0.1);
            margin-bottom: 1.5rem;
        }
        
        .brand-logo {
            width: 50px;
            height: 50px;
            background: linear-gradient(135deg, #007bff, #0056b3);
            border-radius: 15px;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 1rem;
            box-shadow: 0 4px 15px rgba(0, 123, 255, 0.3);
            transition: all 0.3s ease;
        }
        
        .brand-logo:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(0, 123, 255, 0.4);
        }
        
        .brand-logo i {
            font-size: 24px;
            color: #ffffff;
        }
        
        .brand-title {
            color: #ffffff;
            font-size: 18px;
            font-weight: 700;
            margin: 0;
            letter-spacing: 0.5px;
        }
        
        .brand-subtitle {
            color: #b8c5d1;
            font-size: 12px;
            font-weight: 500;
            text-transform: uppercase;
            letter-spacing: 1px;
        }
        
        .sidebar-nav .nav-item {
            margin-bottom: 0.5rem;
        }
        
        .sidebar-nav .nav-link {
            display: flex;
            align-items: center;
            padding: 0.75rem 1rem;
            color: #b8c5d1;
            text-decoration: none;
            border-radius: 10px;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }
        
        .sidebar-nav .nav-link::before {
            content: '';
            position: absolute;
            top: 0;
            left: -100%;
            width: 100%;
            height: 100%;
            background: linear-gradient(90deg, transparent, rgba(255, 255, 255, 0.1), transparent);
            transition: left 0.5s ease;
        }
        
        .sidebar-nav .nav-link:hover::before {
            left: 100%;
        }
        
        .sidebar-nav .nav-link:hover {
            color: #ffffff;
            background: linear-gradient(135deg, rgba(0, 123, 255, 0.1), rgba(0, 86, 179, 0.1));
            transform: translateX(5px);
        }
        
        .sidebar-nav .nav-link.active {
            color: #ffffff;
            background: linear-gradient(135deg, #007bff, #0056b3);
            box-shadow: 0 4px 15px rgba(0, 123, 255, 0.3);
        }
        
        .nav-icon {
            width: 20px;
            height: 20px;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-right: 12px;
            transition: all 0.3s ease;
        }
        
        .nav-icon i {
            font-size: 16px;
        }
        
        .nav-text {
            font-weight: 500;
            font-size: 14px;
        }
        
        .sidebar-nav .nav-link:hover .nav-icon {
            transform: scale(1.1);
        }
        
        .sidebar-nav .nav-link.active .nav-icon {
            transform: scale(1.1);
        }
        
        .sidebar-footer {
            margin-top: auto;
            padding: 1rem;
            border-top: 1px solid rgba(255, 255, 255, 0.1);
            background: rgba(0, 0, 0, 0.2);
        }
        
        .footer-info {
            display: flex;
            align-items: center;
            color: #b8c5d1;
            font-size: 12px;
            font-weight: 500;
        }
        
        .footer-info i {
            color: #28a745;
            margin-right: 8px;
            font-size: 14px;
        }
        
        @media (max-width: 768px) {
            .sidebar-modern {
                min-height: auto;
            }
            
            .brand-logo {
                width: 40px;
                height: 40px;
            }
            
            .brand-logo i {
                font-size: 20px;
            }
            
            .brand-title {
                font-size: 16px;
            }
            
            .sidebar-nav .nav-link {
                padding: 0.6rem 0.8rem;
            }
            
            .nav-text {
                font-size: 13px;
            }
        }
        .subject-checkbox {
            margin-bottom: 8px;
        }
        .student-card {
            border-left: 4px solid #dc3545;
        }
        .restore-btn {
            background: linear-gradient(45deg, #28a745, #20c997);
            border: none;
            color: white;
            padding: 8px 16px;
            border-radius: 8px;
            transition: all 0.3s ease;
        }
        .restore-btn:hover {
            background: linear-gradient(45deg, #20c997, #28a745);
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.2);
        }
        .delete-btn {
            background: linear-gradient(45deg, #dc3545, #c82333);
            border: none;
            color: white;
            padding: 8px 16px;
            border-radius: 8px;
            transition: all 0.3s ease;
        }
        .delete-btn:hover {
            background: linear-gradient(45deg, #c82333, #dc3545);
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.2);
        }
        .badge-soft {
            background-color: rgba(13,110,253,.12);
            color: #0d6efd;
            border: 1px solid rgba(13,110,253,.2);
            padding: .35rem .55rem;
            border-radius: .75rem;
            font-weight: 600;
        }
        .section-title {
            display: flex;
            align-items: center;
            gap: .5rem;
            color: #0f3460;
            font-weight: 700;
            letter-spacing: .2px;
        }
        .pill {
            display: inline-flex; align-items: center; gap: .4rem;
            background: #eef3ff; color: #0d6efd; border: 1px solid rgba(13,110,253,.2);
            padding: .25rem .6rem; border-radius: 999px; font-size: .8rem; font-weight: 600;
        }
        .stat {
            display: inline-flex; align-items: center; gap: .4rem; font-size: .9rem; color: #445;
        }
        .divider-soft { height: 1px; background: linear-gradient(90deg, transparent, #e3e7ef, transparent); margin: 1rem 0; }
    </style>
<body>
    <div class="container-fluid py-3">
        <div class="row">
            <!-- Main content (full-width) -->
            <main class="col-12 px-md-4 container-max mx-auto">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">
                        <i class="bi bi-arrow-clockwise text-success me-2"></i>
                        Restore Suspended Students
                    </h1>
                    <div>
                        <form class="row g-2" method="GET">
                            <div class="col-auto">
                                <input type="text" class="form-control" name="q" placeholder="Search name/email/username" value="<?php echo htmlspecialchars($q); ?>" aria-label="Search suspended students">
                        </div>
                            <div class="col-auto">
                                <select name="grade_id" class="form-select">
                                    <option value="">All Grades</option>
                                    <?php foreach ($grades as $g): ?>
                                        <option value="<?php echo (int)$g['id']; ?>" <?php echo $gradeIdFilter===(int)$g['id']?'selected':''; ?>>
                                            <?php echo htmlspecialchars($g['name']); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                    </div>
                            <div class="col-auto">
                                <input type="date" class="form-control" name="from" value="<?php echo htmlspecialchars($fromDate); ?>">
                                    </div>
                            <div class="col-auto">
                                <input type="date" class="form-control" name="to" value="<?php echo htmlspecialchars($toDate); ?>">
                                    </div>
                            <div class="col-auto">
                                <button class="btn btn-outline-primary" type="submit"><i class="bi bi-funnel me-1"></i>Filter</button>
                                    </div>
                            <div class="col-auto">
                                <a class="btn btn-outline-secondary" href="restore_student.php"><i class="bi bi-x-circle me-1"></i>Clear</a>
                                    </div>
                            <div class="col-auto">
                                <a class="btn btn-outline-success" href="restore_student.php?export=csv&q=<?php echo urlencode($q); ?>&grade_id=<?php echo (int)$gradeIdFilter; ?>&from=<?php echo urlencode($fromDate); ?>&to=<?php echo urlencode($toDate); ?>">
                                    <i class="bi bi-file-earmark-spreadsheet me-1"></i>Export CSV
                                </a>
                                    </div>
                        </form>
                    </div>
                </div>

                <!-- Bulk restore toolbar -->
                <form method="POST" class="card card-modern mb-3" id="bulkSelectForm">
                    <div class="card-body">
                        <div class="row g-3 align-items-end">
                            <div class="col-md-6">
                                <label class="form-label fw-bold section-title"><i class="bi bi-collection"></i> Bulk Subjects</label>
                                <div class="border rounded p-2" style="max-height: 160px; overflow-y:auto;">
                                    <?php foreach ($subjects as $subject): ?>
                                        <div class="form-check">
                                            <input class="form-check-input" type="checkbox" name="bulk_subjects[]" value="<?php echo (int)$subject['id']; ?>" id="bulk_subject_<?php echo (int)$subject['id']; ?>">
                                            <label class="form-check-label" for="bulk_subject_<?php echo (int)$subject['id']; ?>"><?php echo htmlspecialchars($subject['name']); ?></label>
                        </div>
                                    <?php endforeach; ?>
                    </div>
                </div>
                            <div class="col-md-3">
                                <div class="form-check mb-2">
                                    <input class="form-check-input" type="checkbox" name="use_originals" value="1" id="use_originals_cb" onclick="toggleBulkSubjects(this)">
                                    <label class="form-check-label" for="use_originals_cb">
                                        Ignore shared picker; use each student's original subjects
                                    </label>
            </div>
                                <button type="submit" name="bulk_restore" value="1" class="btn btn-success w-100">
                                    <i class="bi bi-arrow-clockwise me-2"></i>Bulk Restore Selected
                                </button>
                </div>
                            <div class="col-md-3 text-md-end">
                                <span class="pill"><i class="bi bi-lightbulb"></i>Tip: select students on their cards</span>
                            </div>
                        </div>
                    </div>
                </form>

                <?php if ($message): ?>
                    <div class="alert alert-<?php echo $messageType; ?> alert-dismissible fade show" role="alert">
                        <i class="bi bi-<?php echo $messageType === 'success' ? 'check-circle' : 'exclamation-triangle'; ?> me-2"></i>
                        <?php echo htmlspecialchars($message); ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <?php if (empty($suspendedStudents)): ?>
                    <div class="card card-modern">
                        <div class="card-body text-center py-5">
                            <i class="bi bi-check-circle text-success" style="font-size: 4rem;"></i>
                            <h3 class="mt-3">No Suspended Students</h3>
                            <p class="text-muted">All students are currently active. No restoration needed.</p>
                            <a href="dashboard.php" class="btn btn-primary">
                                <i class="bi bi-house-door me-2"></i>Back to Dashboard
                            </a>
                        </div>
                    </div>
                <?php else: ?>
                    <div class="row">
                        <?php foreach ($suspendedStudents as $student): ?>
                            <div class="col-md-6 col-lg-4 mb-4">
                                <div class="card card-modern student-card">
                                    <div class="card-header card-header-modern text-white">
                                        <h5 class="mb-0">
                                            <span class="badge badge-soft me-2"><i class="bi bi-person-x"></i></span>
                                            <?php echo htmlspecialchars(($student['first_name'] ?? '') . ' ' . ($student['last_name'] ?? '')); ?>
                                        </h5>
                                    </div>
                                    <div class="card-body">
                                    <div class="form-check mb-2">
                                            <input class="form-check-input" type="checkbox" name="selected[]" form="bulkSelectForm" value="<?php echo (int)$student['id']; ?>" id="select_<?php echo (int)$student['id']; ?>">
                                            <label class="form-check-label" for="select_<?php echo (int)$student['id']; ?>">Select for bulk restore</label>
                                        </div>
                                        <div class="mb-3">
                                            <strong>Email:</strong> <?php echo htmlspecialchars($student['email'] ?? 'N/A'); ?><br>
                                            <strong>Username:</strong> <?php echo htmlspecialchars(formatUsernameForMoodle($student['username'] ?? '')); ?><br>
                                            <strong>Grade:</strong> <?php echo htmlspecialchars($student['grade_name'] ?? 'Not assigned'); ?><br>
                                            <strong>Deleted:</strong> <?php echo date('M j, Y', strtotime($student['deleted_at'])); ?><br>
                                            <?php $moodle = getMoodleStatusForStudent($student); ?>
                                            <strong>Moodle:</strong> 
                                            <span class="badge bg-<?php echo $moodle['found']?'success':'secondary'; ?>">
                                                <?php echo $moodle['found']?'Found':'Not Found'; ?>
                                            </span>
                                            <span class="badge bg-info text-dark">Enrollments: <?php echo (int)$moodle['enrollments']; ?></span><br>
                                            <strong>Days Remaining:</strong> 
                                            <span class="badge bg-<?php echo $student['days_remaining'] <= 7 ? 'danger' : ($student['days_remaining'] <= 30 ? 'warning' : 'success'); ?>">
                                                <?php echo $student['days_remaining']; ?> days
                                            </span>
                                        </div>

                                        <!-- Restore Form -->
                                        <form method="POST" class="restore-form">
                                            <input type="hidden" name="student_id" value="<?php echo $student['id']; ?>">
                                            
                                            <div class="mb-3">
                                                <label class="form-label fw-bold">
                                                    <i class="bi bi-book me-2"></i>Select Subjects to Enroll:
                                                </label>
                                                <div class="border rounded p-3" style="max-height: 200px; overflow-y: auto;">
                                                    <?php $origIds = getOriginalSubjectIds($student['id']); ?>
                                                    <?php foreach ($subjects as $subject): ?>
                                                        <div class="form-check subject-checkbox">
                                                            <input class="form-check-input" type="checkbox" 
                                                                   name="subjects[]" 
                                                                   value="<?php echo $subject['id']; ?>"
                                                                   id="subject_<?php echo $student['id']; ?>_<?php echo $subject['id']; ?>"
                                                                   <?php echo in_array((int)$subject['id'], $origIds, true)?'checked':''; ?>>
                                                            <label class="form-check-label" for="subject_<?php echo $student['id']; ?>_<?php echo $subject['id']; ?>">
                                                                <?php echo htmlspecialchars($subject['name']); ?>
                                                            </label>
                                                        </div>
                                                    <?php endforeach; ?>
                                                </div>
                                            </div>

                                            <div class="d-grid gap-2">
                                                <button type="submit" name="restore_student" class="btn restore-btn">
                                                    <i class="bi bi-arrow-clockwise me-2"></i>
                                                    Restore Student
                                                </button>
                                                <button type="button" class="btn btn-outline-success" onclick="restoreWithOriginals(<?php echo (int)$student['id']; ?>)">
                                                    <i class="bi bi-magic me-2"></i>Restore with Original Subjects
                                                </button>
                                            </div>
                                            
                                            <!-- Permanent Delete Section -->
                                            <hr class="my-3">
                                            <div class="alert alert-danger">
                                                <h6 class="alert-heading">
                                                    <i class="bi bi-exclamation-triangle me-2"></i>
                                                    Danger Zone
                                                </h6>
                                                <p class="mb-2">Permanently delete this student from the system. This action cannot be undone.</p>
                                                <form method="POST" class="d-inline" id="deleteForm_<?php echo $student['id']; ?>">
                                                    <input type="hidden" name="student_id" value="<?php echo $student['id']; ?>">
                                                    <input type="hidden" name="delete_permanently" value="1">
                                                    <button type="button" name="delete_permanently" class="btn delete-btn" onclick="handleDeleteClick(event)">
                                                        <i class="bi bi-trash3 me-2"></i>
                                                        Delete Permanently
                                                    </button>
                                                </form>
                                            </div>
                                        </form>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>

                <!-- Pagination -->
                <?php if ($totalPages > 1): ?>
                <nav class="mt-3">
                    <ul class="pagination">
                        <?php for ($p=1; $p <= $totalPages; $p++): ?>
                            <li class="page-item <?php echo $p===$page?'active':''; ?>">
                                <a class="page-link" href="?q=<?php echo urlencode($q); ?>&grade_id=<?php echo (int)$gradeIdFilter; ?>&from=<?php echo urlencode($fromDate); ?>&to=<?php echo urlencode($toDate); ?>&page=<?php echo $p; ?>"><?php echo $p; ?></a>
                            </li>
                        <?php endfor; ?>
                    </ul>
                </nav>
                <?php endif; ?>
            </main>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Add some interactivity
        document.addEventListener('DOMContentLoaded', function() {
            // Add confirmation for restore action
            const restoreForms = document.querySelectorAll('.restore-form');
            restoreForms.forEach(form => {
                form.addEventListener('submit', function(e) {
                    const checkedSubjects = form.querySelectorAll('input[name="subjects[]"]:checked');
                    if (checkedSubjects.length === 0) {
                        e.preventDefault();
                        // Show a more user-friendly message instead of alert
                        const messageDiv = document.createElement('div');
                        messageDiv.className = 'alert alert-warning alert-dismissible fade show';
                        messageDiv.innerHTML = `
                            <i class="bi bi-exclamation-triangle me-2"></i>
                            Please select at least one subject to enroll the student in.
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        `;
                        form.parentNode.insertBefore(messageDiv, form);
                        return false;
                    }
                    
                    const studentName = form.closest('.card').querySelector('.card-header h5').textContent.trim();
                    if (!confirm(`Are you sure you want to restore ${studentName} with the selected subjects?`)) {
                        e.preventDefault();
                        return false;
                    }
                });
            });
        });
        
        function toggleBulkSubjects(cb) {
            const container = document.querySelector('#bulkSelectForm .border.rounded');
            if (!container) return;
            const inputs = container.querySelectorAll('input[type="checkbox"]');
            inputs.forEach(i => { i.disabled = cb.checked; });
            container.style.opacity = cb.checked ? 0.5 : 1;
        }
        
        // Confirmation function for permanent deletion
        function confirmPermanentDelete() {
            const studentName = document.querySelector('.card-header h5').textContent.trim();
            const confirmMessage = `⚠️ WARNING: You are about to PERMANENTLY DELETE ${studentName} from the system.\n\n` +
                                 `This action will:\n` +
                                 `• Remove the student from the database\n` +
                                 `• Unenroll them from all Moodle courses\n` +
                                 `• Suspend their Moodle account\n` +
                                 `• This action CANNOT be undone\n\n` +
                                 `Are you absolutely sure you want to continue?`;
            
            return confirm(confirmMessage);
        }
        
        // Enhanced delete button functionality
        function handleDeleteClick(event) {
            event.preventDefault();
            const form = event.target.closest('form');
            const studentId = form.querySelector('input[name="student_id"]').value;
            const studentName = form.closest('.card').querySelector('.card-header h5').textContent.trim();
            
            // Custom confirmation modal instead of browser confirm
            if (confirm(`⚠️ WARNING: You are about to PERMANENTLY DELETE ${studentName} from the system.\n\nThis action will:\n• Remove the student from the database\n• Unenroll them from all Moodle courses\n• Suspend their Moodle account\n• This action CANNOT be undone\n\nAre you absolutely sure you want to continue?`)) {
                // Submit the form with the hidden delete_permanently flag
                form.submit();
            }
        }

        // Quick restore with original subjects
        function restoreWithOriginals(studentId) {
            const card = document.querySelector(`input[name="student_id"][value="${studentId}"]`).closest('.card');
            const form = card.querySelector('form.restore-form');
            // Ensure only original subjects remain checked
            const checkboxes = form.querySelectorAll('input[name="subjects[]"]');
            checkboxes.forEach(cb => {
                // Original subjects are prechecked server-side; uncheck any user-changed boxes
                const isPrechecked = cb.outerHTML.includes('checked');
                cb.checked = isPrechecked;
            });
            if (confirm('Restore student with original subjects?')) {
                form.submit();
            }
        }
    </script>

    <!-- Professional Modern Footer -->
    <footer class="footer-modern">
        <div class="container-fluid">
            <div class="row align-items-center">
                <div class="col-md-6">
                    <div class="d-flex align-items-center">
                        <div class="footer-logo me-3">
                            <div class="logo-icon">
                                <i class="bi bi-mortarboard-fill"></i>
                            </div>
                        </div>
                        <div class="footer-brand">
                            <h6 class="brand-title"><?php echo APP_NAME; ?> Learning Platform</h6>
                            <p class="brand-subtitle">Advanced Student Management System</p>
                        </div>
                    </div>
                </div>
                <div class="col-md-6 text-md-end">
                    <div class="d-flex align-items-center justify-content-md-end">
                        <div class="developer-info">
                            <div class="d-flex align-items-center">
                                <div class="developer-logo me-2">
                                    <i class="bi bi-code-slash"></i>
                                </div>
                                <div class="developer-text">
                                    <span class="powered-by">Powered by</span>
                                    <a href="https://binarygroup.com" target="_blank" class="developer-link">
                                        BinaryGroup.com
                                    </a>
                                    <span class="developer-tagline">Professional Web Development</span>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="footer-divider"></div>
            <div class="row align-items-center">
                <div class="col-md-6">
                    <div class="footer-info">
                        <i class="bi bi-shield-check me-2"></i>
                        <span>Secure & Reliable Student Management</span>
                    </div>
                </div>
                <div class="col-md-6 text-md-end">
                    <div class="footer-copyright">
                        <i class="bi bi-calendar3 me-2"></i>
                        <span>© <?php echo date('Y'); ?> All rights reserved</span>
                    </div>
                </div>
            </div>
        </div>
    </footer>

    <style>
        .footer-modern {
            background: linear-gradient(135deg, #1a1a2e 0%, #16213e 50%, #0f3460 100%);
            color: #ffffff;
            padding: 1.5rem 0;
            margin-top: 2rem;
            border-top: 3px solid #007bff;
            position: relative;
            overflow: hidden;
        }
        
        .footer-modern::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 2px;
            background: linear-gradient(90deg, #007bff, #28a745, #ffc107, #dc3545, #6f42c1);
            animation: gradientShift 3s ease-in-out infinite;
        }
        
        @keyframes gradientShift {
            0%, 100% { opacity: 0.8; }
            50% { opacity: 1; }
        }
        
        .footer-logo .logo-icon {
            width: 40px;
            height: 40px;
            background: linear-gradient(135deg, #007bff, #0056b3);
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            box-shadow: 0 4px 15px rgba(0, 123, 255, 0.3);
            transition: all 0.3s ease;
        }
        
        .footer-logo .logo-icon:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(0, 123, 255, 0.4);
        }
        
        .footer-logo .logo-icon i {
            font-size: 18px;
            color: #ffffff;
        }
        
        .footer-brand .brand-title {
            color: #ffffff;
            font-size: 16px;
            font-weight: 600;
            margin: 0;
            letter-spacing: 0.5px;
        }
        
        .footer-brand .brand-subtitle {
            color: #b8c5d1;
            font-size: 12px;
            margin: 0;
            font-weight: 400;
        }
        
        .developer-logo {
            width: 32px;
            height: 32px;
            background: linear-gradient(135deg, #28a745, #20c997);
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            box-shadow: 0 3px 10px rgba(40, 167, 69, 0.3);
            transition: all 0.3s ease;
        }
        
        .developer-logo:hover {
            transform: translateY(-1px);
            box-shadow: 0 5px 15px rgba(40, 167, 69, 0.4);
        }
        
        .developer-logo i {
            font-size: 14px;
            color: #ffffff;
        }
        
        .developer-text {
            display: flex;
            flex-direction: column;
            align-items: flex-end;
        }
        
        .powered-by {
            color: #b8c5d1;
            font-size: 10px;
            font-weight: 500;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .developer-link {
            color: #ffffff;
            text-decoration: none;
            font-size: 14px;
            font-weight: 600;
            transition: all 0.3s ease;
            position: relative;
        }
        
        .developer-link:hover {
            color: #28a745;
            transform: translateX(2px);
        }
        
        .developer-link::after {
            content: '';
            position: absolute;
            bottom: -2px;
            left: 0;
            width: 0;
            height: 2px;
            background: linear-gradient(90deg, #28a745, #20c997);
            transition: width 0.3s ease;
        }
        
        .developer-link:hover::after {
            width: 100%;
        }
        
        .developer-tagline {
            color: #b8c5d1;
            font-size: 10px;
            font-weight: 400;
        }
        
        .footer-divider {
            height: 1px;
            background: linear-gradient(90deg, transparent, #007bff, transparent);
            margin: 1rem 0;
            opacity: 0.6;
        }
        
        .footer-info, .footer-copyright {
            color: #b8c5d1;
            font-size: 11px;
            font-weight: 500;
            display: flex;
            align-items: center;
        }
        
        .footer-info i, .footer-copyright i {
            color: #007bff;
            font-size: 12px;
        }
        
        .footer-modern .row {
            margin: 0;
        }
        
        .footer-modern .col-md-6 {
            padding: 0.5rem 0;
        }
        
        @media (max-width: 768px) {
            .footer-modern {
                padding: 1rem 0;
            }
            
            .footer-brand .brand-title {
                font-size: 14px;
            }
            
            .footer-brand .brand-subtitle {
                font-size: 11px;
            }
            
            .developer-text {
                align-items: flex-start;
            }
            
            .footer-info, .footer-copyright {
                font-size: 10px;
                margin-top: 0.5rem;
            }
        }
    </style>
<?php include __DIR__.'/partials/footer.php'; ?>
