<?php
/**
 * Map Subjects to Moodle Courses
 * Admin interface to link internal subjects with Moodle LMS courses
 */

require_once '../includes/functions.php';
require_once '../includes/moodle_api.php';
requireAdminLogin();

$pdo = getDB();
$academy_reference = getCurrentAcademyReference();

// Initialize Moodle API
$moodle = new MoodleAPI($academy_reference, $pdo);

// Handle form submissions
$message = '';
$message_type = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'save_mapping':
                $subject_id = intval($_POST['subject_id']);
                $moodle_course_id = intval($_POST['moodle_course_id']) ?: null;
                $moodle_course_name = $_POST['moodle_course_name'] ?? '';
                $moodle_shortname = $_POST['moodle_shortname'] ?? '';
                $auto_enroll = isset($_POST['auto_enroll']) ? 1 : 0;
                $moodle_sync_enabled = isset($_POST['moodle_sync_enabled']) ? 1 : 0;
                
                try {
                    $stmt = $pdo->prepare("
                        UPDATE subjects 
                        SET moodle_course_id = ?, 
                            moodle_course_name = ?, 
                            moodle_shortname = ?,
                            auto_enroll = ?,
                            moodle_sync_enabled = ?
                        WHERE id = ?
                    ");
                    $stmt->execute([
                        $moodle_course_id, $moodle_course_name, $moodle_shortname,
                        $auto_enroll, $moodle_sync_enabled, $subject_id
                    ]);
                    
                    $message = "Mapping saved successfully!";
                    $message_type = "success";
                } catch (PDOException $e) {
                    $message = "Error saving mapping: " . $e->getMessage();
                    $message_type = "danger";
                }
                break;
                
            case 'sync_courses':
                $result = $moodle->getCourses();
                if ($result['success']) {
                    $message = "Synced " . count($result['data']) . " courses from Moodle!";
                    $message_type = "success";
                } else {
                    $message = "Error syncing courses: " . $result['error'];
                    $message_type = "danger";
                }
                break;
                
            case 'test_connection':
                $result = $moodle->testConnection();
                $message = $result['message'];
                $message_type = $result['success'] ? 'success' : 'danger';
                if ($result['success']) {
                    $message .= " (Site: " . $result['site_name'] . ", Version: " . $result['moodle_version'] . ")";
                }
                break;
        }
    }
}

// Get all subjects
$subjects = getAllSubjects();

// Get cached Moodle courses
$stmt = $pdo->prepare("
    SELECT * FROM moodle_course_cache 
    WHERE academy_reference = ? OR academy_reference IS NULL
    ORDER BY course_name
");
$stmt->execute([$academy_reference]);
$moodle_courses = $stmt->fetchAll(PDO::FETCH_ASSOC);

include 'partials/header.php';
?>

<div class="container-fluid py-4">
    <div class="row">
        <div class="col-12">
            <!-- Header -->
            <div class="d-flex justify-content-between align-items-center mb-4">
                <div>
                    <h2 class="mb-1">
                        <i class="bi bi-link-45deg me-2"></i>
                        Map Subjects to Moodle Courses
                    </h2>
                    <p class="text-muted mb-0">Link your subjects with Moodle LMS courses for automatic enrollment</p>
                </div>
                <div>
                    <a href="settings.php" class="btn btn-outline-secondary me-2">
                        <i class="bi bi-gear me-2"></i>Moodle Settings
                    </a>
                    <a href="dashboard.php" class="btn btn-outline-secondary">
                        <i class="bi bi-arrow-left me-2"></i>Back
                    </a>
                </div>
            </div>

            <?php if ($message): ?>
            <div class="alert alert-<?= $message_type ?> alert-dismissible fade show">
                <i class="bi bi-<?= $message_type === 'success' ? 'check-circle' : 'exclamation-triangle' ?>-fill me-2"></i>
                <?= htmlspecialchars($message) ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
            <?php endif; ?>

            <!-- Moodle Status Card -->
            <div class="card mb-4">
                <div class="card-body">
                    <div class="row align-items-center">
                        <div class="col-md-8">
                            <h5 class="card-title mb-2">
                                <i class="bi bi-server me-2"></i>Moodle Connection Status
                            </h5>
                            <?php if ($moodle->isEnabled()): ?>
                                <span class="badge bg-success fs-6">
                                    <i class="bi bi-check-circle me-1"></i>Enabled & Configured
                                </span>
                                <p class="text-muted mt-2 mb-0">
                                    <small>Last course sync: <?= !empty($moodle_courses) ? 'Recently' : 'Never' ?></small>
                                </p>
                            <?php else: ?>
                                <span class="badge bg-danger fs-6">
                                    <i class="bi bi-x-circle me-1"></i>Not Configured
                                </span>
                                <p class="text-muted mt-2 mb-0">
                                    <small>Please configure Moodle settings first</small>
                                </p>
                            <?php endif; ?>
                        </div>
                        <div class="col-md-4 text-end">
                            <form method="POST" class="d-inline">
                                <input type="hidden" name="action" value="test_connection">
                                <button type="submit" class="btn btn-outline-primary me-2" <?= !$moodle->isEnabled() ? 'disabled' : '' ?>>
                                    <i class="bi bi-arrow-repeat me-2"></i>Test Connection
                                </button>
                            </form>
                            <form method="POST" class="d-inline">
                                <input type="hidden" name="action" value="sync_courses">
                                <button type="submit" class="btn btn-primary" <?= !$moodle->isEnabled() ? 'disabled' : '' ?>>
                                    <i class="bi bi-cloud-download me-2"></i>Sync Courses
                                </button>
                            </form>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Mapping Table -->
            <div class="card">
                <div class="card-header bg-primary text-white">
                    <h5 class="mb-0">
                        <i class="bi bi-table me-2"></i>Subject to Moodle Course Mapping
                    </h5>
                </div>
                <div class="card-body">
                    <?php if (empty($subjects)): ?>
                        <div class="alert alert-info">
                            <i class="bi bi-info-circle me-2"></i>
                            No subjects found. Please add subjects first.
                        </div>
                    <?php else: ?>
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead class="table-light">
                                    <tr>
                                        <th>Subject Name</th>
                                        <th>Subject Code</th>
                                        <th>Moodle Course</th>
                                        <th>Auto-Enroll</th>
                                        <th>Sync Enabled</th>
                                        <th>Status</th>
                                        <th>Action</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($subjects as $subject): ?>
                                    <tr>
                                        <td><strong><?= htmlspecialchars($subject['name']) ?></strong></td>
                                        <td><code><?= htmlspecialchars($subject['code']) ?></code></td>
                                        <td>
                                            <?php if (!empty($subject['moodle_course_name'])): ?>
                                                <span class="text-success">
                                                    <i class="bi bi-link me-1"></i>
                                                    <?= htmlspecialchars($subject['moodle_course_name']) ?>
                                                </span>
                                                <br>
                                                <small class="text-muted">
                                                    <?= htmlspecialchars($subject['moodle_shortname'] ?? '') ?>
                                                </small>
                                            <?php else: ?>
                                                <span class="text-muted">
                                                    <i class="bi bi-dash-circle me-1"></i>Not mapped
                                                </span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if ($subject['auto_enroll']): ?>
                                                <span class="badge bg-success">Yes</span>
                                            <?php else: ?>
                                                <span class="badge bg-secondary">No</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if ($subject['moodle_sync_enabled']): ?>
                                                <span class="badge bg-primary">Enabled</span>
                                            <?php else: ?>
                                                <span class="badge bg-secondary">Disabled</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if (!empty($subject['moodle_course_id'])): ?>
                                                <span class="badge bg-success">
                                                    <i class="bi bi-check-circle me-1"></i>Mapped
                                                </span>
                                            <?php else: ?>
                                                <span class="badge bg-warning text-dark">
                                                    <i class="bi bi-exclamation-triangle me-1"></i>Unmapped
                                                </span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <button type="button" class="btn btn-sm btn-primary" 
                                                    data-bs-toggle="modal" 
                                                    data-bs-target="#mapModal<?= $subject['id'] ?>">
                                                <i class="bi bi-pencil me-1"></i>Edit Mapping
                                            </button>
                                        </td>
                                    </tr>
                                    
                                    <!-- Mapping Modal -->
                                    <div class="modal fade" id="mapModal<?= $subject['id'] ?>" tabindex="-1">
                                        <div class="modal-dialog">
                                            <div class="modal-content">
                                                <form method="POST">
                                                    <div class="modal-header">
                                                        <h5 class="modal-title">
                                                            Map: <?= htmlspecialchars($subject['name']) ?>
                                                        </h5>
                                                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                                                    </div>
                                                    <div class="modal-body">
                                                        <input type="hidden" name="action" value="save_mapping">
                                                        <input type="hidden" name="subject_id" value="<?= $subject['id'] ?>">
                                                        
                                                        <div class="mb-3">
                                                            <label class="form-label">Select Moodle Course</label>
                                                            <select class="form-select" name="moodle_course_id" id="course_select_<?= $subject['id'] ?>">
                                                                <option value="">-- No mapping --</option>
                                                                <?php foreach ($moodle_courses as $course): ?>
                                                                    <option value="<?= $course['moodle_course_id'] ?>"
                                                                            data-name="<?= htmlspecialchars($course['course_name']) ?>"
                                                                            data-short="<?= htmlspecialchars($course['shortname']) ?>"
                                                                            <?= ($subject['moodle_course_id'] == $course['moodle_course_id']) ? 'selected' : '' ?>>
                                                                        <?= htmlspecialchars($course['course_name']) ?>
                                                                        (<?= htmlspecialchars($course['shortname']) ?>)
                                                                    </option>
                                                                <?php endforeach; ?>
                                                            </select>
                                                            <input type="hidden" name="moodle_course_name" id="course_name_<?= $subject['id'] ?>" value="<?= htmlspecialchars($subject['moodle_course_name'] ?? '') ?>">
                                                            <input type="hidden" name="moodle_shortname" id="course_short_<?= $subject['id'] ?>" value="<?= htmlspecialchars($subject['moodle_shortname'] ?? '') ?>">
                                                        </div>
                                                        
                                                        <div class="mb-3">
                                                            <div class="form-check form-switch">
                                                                <input class="form-check-input" type="checkbox" 
                                                                       name="auto_enroll" id="auto_enroll_<?= $subject['id'] ?>"
                                                                       <?= $subject['auto_enroll'] ? 'checked' : '' ?>>
                                                                <label class="form-check-label" for="auto_enroll_<?= $subject['id'] ?>">
                                                                    Auto-enroll students when they pay/register
                                                                </label>
                                                            </div>
                                                        </div>
                                                        
                                                        <div class="mb-3">
                                                            <div class="form-check form-switch">
                                                                <input class="form-check-input" type="checkbox" 
                                                                       name="moodle_sync_enabled" id="sync_enabled_<?= $subject['id'] ?>"
                                                                       <?= $subject['moodle_sync_enabled'] ? 'checked' : '' ?>>
                                                                <label class="form-check-label" for="sync_enabled_<?= $subject['id'] ?>">
                                                                    Enable Moodle sync for this subject
                                                                </label>
                                                            </div>
                                                        </div>
                                                    </div>
                                                    <div class="modal-footer">
                                                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                                                        <button type="submit" class="btn btn-primary">
                                                            <i class="bi bi-save me-2"></i>Save Mapping
                                                        </button>
                                                    </div>
                                                </form>
                                            </div>
                                        </div>
                                    </div>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
            
            <!-- Help Card -->
            <div class="card mt-4">
                <div class="card-body">
                    <h5 class="card-title">
                        <i class="bi bi-question-circle me-2"></i>How to Use
                    </h5>
                    <ol class="mb-0">
                        <li><strong>Configure Moodle Settings:</strong> Go to Settings and add your Moodle URL and API token</li>
                        <li><strong>Sync Courses:</strong> Click "Sync Courses" to fetch all courses from your Moodle site</li>
                        <li><strong>Map Subjects:</strong> Click "Edit Mapping" for each subject and select the corresponding Moodle course</li>
                        <li><strong>Enable Auto-Enroll:</strong> Turn on auto-enrollment if you want students automatically enrolled when they pay</li>
                        <li><strong>Done!</strong> Students will be enrolled in Moodle courses automatically</li>
                    </ol>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
// Update hidden fields when course is selected
document.querySelectorAll('[id^="course_select_"]').forEach(select => {
    select.addEventListener('change', function() {
        const selectedOption = this.options[this.selectedIndex];
        const subjectId = this.id.split('_').pop();
        
        if (selectedOption.value) {
            document.getElementById('course_name_' + subjectId).value = selectedOption.dataset.name;
            document.getElementById('course_short_' + subjectId).value = selectedOption.dataset.short;
        } else {
            document.getElementById('course_name_' + subjectId).value = '';
            document.getElementById('course_short_' + subjectId).value = '';
        }
    });
});
</script>

<?php include 'partials/footer.php'; ?>

