<?php
/**
 * Test Subject-Course Mappings
 * Verify that all mappings are correctly configured
 */

session_start();
require_once '../../includes/functions.php';
require_once '../../includes/school_context.php';
require_once '../../includes/moodle_api.php';
require_once '../../includes/favicon_helper.php';
requireAdminLogin();

$pdo = getDB();
$academy_reference = getCurrentSchool();
$schoolName = getSchoolName();
$moodleAPI = new MoodleAPI($academy_reference, $pdo);

$results = [];

// Get all mappings
$stmt = $pdo->prepare("
    SELECT 
        smm.*,
        s.name as subject_name,
        s.code as subject_code,
        GROUP_CONCAT(DISTINCT g.name ORDER BY g.name SEPARATOR ', ') as grades
    FROM subject_moodle_mapping smm
    INNER JOIN subjects s ON smm.subject_id = s.id
    LEFT JOIN grade_subject gs ON s.id = gs.subject_id
    LEFT JOIN grades g ON gs.grade_id = g.id
    WHERE smm.academy_reference = ?
    GROUP BY smm.id, smm.academy_reference, smm.subject_id, smm.moodle_course_id,
             smm.moodle_course_shortname, smm.moodle_course_idnumber,
             smm.auto_enroll, smm.sync_enabled, smm.created_by,
             smm.created_at, smm.updated_at,
             s.name, s.code
    ORDER BY s.name
");
$stmt->execute([$academy_reference]);
$mappings = $stmt->fetchAll();

// Test each mapping
foreach ($mappings as $mapping) {
    $test_result = [
        'mapping' => $mapping,
        'tests' => []
    ];
    
    // Test 1: Check if course exists in Moodle
    try {
        $courses_result = $moodleAPI->getCourses();
        $course_found = false;
        $moodle_course = null;
        
        // getCourses() returns ['success' => bool, 'data' => array] or ['success' => false, 'error' => string]
        $courses = [];
        if (isset($courses_result['success']) && $courses_result['success'] && isset($courses_result['data'])) {
            $courses = is_array($courses_result['data']) ? $courses_result['data'] : [];
        }
        
        foreach ($courses as $course) {
            // Ensure $course is an array with the expected keys
            if (!is_array($course) || !isset($course['id']) || !isset($course['shortname'])) {
                continue;
            }
            
            if ($course['id'] == $mapping['moodle_course_id'] || 
                $course['shortname'] == $mapping['moodle_course_shortname']) {
                $course_found = true;
                $moodle_course = $course;
                break;
            }
        }
        
        // Check if API call failed
        if (isset($courses_result['success']) && !$courses_result['success']) {
            $error_msg = $courses_result['error'] ?? 'Unknown error';
            $test_result['tests']['course_exists'] = [
                'status' => 'error',
                'message' => 'Failed to fetch courses from Moodle',
                'details' => $error_msg
            ];
        } else {
            $test_result['tests']['course_exists'] = [
                'status' => $course_found ? 'success' : 'error',
                'message' => $course_found ? 'Course exists in Moodle' : 'Course not found in Moodle',
                'details' => $moodle_course
            ];
        }
    } catch (Exception $e) {
        $test_result['tests']['course_exists'] = [
            'status' => 'error',
            'message' => 'Failed to verify course',
            'details' => $e->getMessage()
        ];
    }
    
    // Test 2: Check if students are assigned to this subject
    $stmt = $pdo->prepare("
        SELECT COUNT(*) as student_count
        FROM student_subject
        WHERE subject_id = ?
    ");
    $stmt->execute([$mapping['subject_id']]);
    $student_count = $stmt->fetchColumn();
    
    $test_result['tests']['students'] = [
        'status' => $student_count > 0 ? 'success' : 'info',
        'message' => $student_count . ' students assigned to this subject',
        'details' => $student_count
    ];
    
    // Test 3: Check sync status
    $test_result['tests']['sync_status'] = [
        'status' => $mapping['sync_enabled'] ? 'success' : 'warning',
        'message' => $mapping['sync_enabled'] ? 'Sync enabled' : 'Sync disabled',
        'details' => [
            'Auto-enroll' => $mapping['auto_enroll'] ? 'Yes' : 'No',
            'Sync enabled' => $mapping['sync_enabled'] ? 'Yes' : 'No'
        ]
    ];
    
    $results[] = $test_result;
}

$total_tests = count($mappings);
$passed_tests = 0;
foreach ($results as $result) {
    if ($result['tests']['course_exists']['status'] === 'success') {
        $passed_tests++;
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Test Mappings - <?= htmlspecialchars($schoolName) ?></title>
    <?php echoFavicon($academy_reference); ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.min.css" rel="stylesheet">
    <style>
        .mapping-card {
            transition: all 0.3s;
            margin-bottom: 20px;
        }
        .test-badge { margin: 2px; }
    </style>
</head>
<body class="bg-light">
    <div class="container mt-4 mb-5">
        <!-- Header -->
        <div class="d-flex justify-content-between align-items-center mb-4">
            <div>
                <h2><i class="bi bi-check2-square text-primary"></i> Test Subject Mappings</h2>
                <p class="text-muted mb-0">Verification results for <?= htmlspecialchars($schoolName) ?></p>
            </div>
            <div>
                <a href="configure_mapping.php" class="btn btn-outline-secondary me-2">
                    <i class="bi bi-arrow-left"></i> Back to Mappings
                </a>
                <a href="javascript:location.reload()" class="btn btn-primary">
                    <i class="bi bi-arrow-repeat"></i> Retest
                </a>
            </div>
        </div>

        <!-- Summary -->
        <div class="row mb-4">
            <div class="col-md-4">
                <div class="card text-center">
                    <div class="card-body">
                        <h1 class="display-4"><?= $total_tests ?></h1>
                        <p class="text-muted mb-0">Total Mappings</p>
                    </div>
                </div>
            </div>
            <div class="col-md-4">
                <div class="card text-center">
                    <div class="card-body">
                        <h1 class="display-4 text-success"><?= $passed_tests ?></h1>
                        <p class="text-muted mb-0">Verified</p>
                    </div>
                </div>
            </div>
            <div class="col-md-4">
                <div class="card text-center">
                    <div class="card-body">
                        <h1 class="display-4 text-danger"><?= $total_tests - $passed_tests ?></h1>
                        <p class="text-muted mb-0">Issues</p>
                    </div>
                </div>
            </div>
        </div>

        <?php if (empty($mappings)): ?>
        <div class="alert alert-info">
            <h4><i class="bi bi-info-circle"></i> No Mappings Found</h4>
            <p>You haven't configured any subject-course mappings yet.</p>
            <a href="configure_mapping.php" class="btn btn-primary">
                <i class="bi bi-plus-circle"></i> Configure Mappings
            </a>
        </div>
        <?php else: ?>
        
        <!-- Results -->
        <?php foreach ($results as $result): ?>
        <div class="card mapping-card">
            <div class="card-header bg-white">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <h5 class="mb-0">
                            <?= htmlspecialchars($result['mapping']['subject_name']) ?>
                            <small class="text-muted">(<?= htmlspecialchars($result['mapping']['subject_code']) ?>)</small>
                        </h5>
                        <small class="text-muted">Grades: <?= htmlspecialchars($result['mapping']['grades']) ?></small>
                    </div>
                    <div>
                        <?php if ($result['tests']['course_exists']['status'] === 'success'): ?>
                        <span class="badge bg-success">✓ Verified</span>
                        <?php else: ?>
                        <span class="badge bg-danger">✗ Failed</span>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-6">
                        <h6>Mapped Course:</h6>
                        <p class="mb-2">
                            <strong>Shortname:</strong> <?= htmlspecialchars($result['mapping']['moodle_course_shortname']) ?><br>
                            <?php if ($result['mapping']['moodle_course_id']): ?>
                            <strong>Course ID:</strong> <?= htmlspecialchars($result['mapping']['moodle_course_id']) ?><br>
                            <?php endif; ?>
                            <?php if ($result['mapping']['moodle_course_idnumber']): ?>
                            <strong>ID Number:</strong> <?= htmlspecialchars($result['mapping']['moodle_course_idnumber']) ?>
                            <?php endif; ?>
                        </p>
                    </div>
                    <div class="col-md-6">
                        <h6>Test Results:</h6>
                        <?php foreach ($result['tests'] as $test_name => $test): ?>
                        <div class="mb-2">
                            <?php if ($test['status'] === 'success'): ?>
                            <span class="badge bg-success test-badge">
                                <i class="bi bi-check-circle"></i> <?= htmlspecialchars($test['message']) ?>
                            </span>
                            <?php elseif ($test['status'] === 'warning'): ?>
                            <span class="badge bg-warning text-dark test-badge">
                                <i class="bi bi-exclamation-triangle"></i> <?= htmlspecialchars($test['message']) ?>
                            </span>
                            <?php elseif ($test['status'] === 'info'): ?>
                            <span class="badge bg-info test-badge">
                                <i class="bi bi-info-circle"></i> <?= htmlspecialchars($test['message']) ?>
                            </span>
                            <?php else: ?>
                            <span class="badge bg-danger test-badge">
                                <i class="bi bi-x-circle"></i> <?= htmlspecialchars($test['message']) ?>
                            </span>
                            <?php endif; ?>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>
                
                <?php if ($result['tests']['course_exists']['status'] === 'success' && 
                          isset($result['tests']['course_exists']['details'])): ?>
                <div class="mt-3 p-3 bg-light rounded">
                    <small>
                        <strong>Moodle Course Details:</strong><br>
                        Full Name: <?= htmlspecialchars($result['tests']['course_exists']['details']['fullname']) ?><br>
                        Shortname: <?= htmlspecialchars($result['tests']['course_exists']['details']['shortname']) ?><br>
                        ID: <?= htmlspecialchars($result['tests']['course_exists']['details']['id']) ?>
                    </small>
                </div>
                <?php elseif ($result['tests']['course_exists']['status'] === 'error'): ?>
                <div class="alert alert-danger mt-3 mb-0">
                    <i class="bi bi-exclamation-triangle"></i> 
                    <strong>Action Required:</strong> 
                    This course was not found in Moodle. 
                    <a href="configure_mapping.php" class="alert-link">Update mapping</a> or 
                    <a href="test_connection.php" class="alert-link">verify Moodle connection</a>.
                </div>
                <?php endif; ?>
            </div>
        </div>
        <?php endforeach; ?>
        
        <?php endif; ?>

        <!-- Actions -->
        <div class="card mt-4">
            <div class="card-body">
                <h5>Next Steps</h5>
                <div class="btn-group" role="group">
                    <a href="configure_mapping.php" class="btn btn-primary">
                        <i class="bi bi-pencil"></i> Edit Mappings
                    </a>
                    <a href="sync_students.php" class="btn btn-success">
                        <i class="bi bi-cloud-upload"></i> Sync Students
                    </a>
                    <a href="test_connection.php" class="btn btn-info">
                        <i class="bi bi-plug"></i> Test Connection
                    </a>
                    <a href="index.php" class="btn btn-secondary">
                        <i class="bi bi-house"></i> Moodle Hub
                    </a>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
