<?php
/**
 * Manual Student Sync to Moodle
 * Syncs unsynced students to Moodle and enrolls them in courses
 */

session_start();
require_once '../../includes/functions.php';
require_once '../../includes/school_context.php';
require_once '../../includes/moodle_enrollment_handler.php';
require_once '../../includes/moodle_api.php';
require_once '../../includes/favicon_helper.php';
requireAdminLogin();

$pdo = getDB();
$academy_reference = getCurrentSchool();
$schoolName = getSchoolName();

$successful = 0;
$partial = 0;
$failed = 0;
$sync_log = [];

$flashMessage = $_SESSION['sync_flash'] ?? null;
unset($_SESSION['sync_flash']);

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['enroll_missing'])) {
        $student_id = (int)($_POST['student_id'] ?? 0);
        $subjectIdsRaw = trim($_POST['subject_ids'] ?? '');
        $subject_ids = array_filter(array_unique(array_map('intval', array_filter(explode(',', $subjectIdsRaw)))));
        
        if ($student_id <= 0) {
            $_SESSION['sync_flash'] = ['type' => 'danger', 'message' => 'Invalid student selection.'];
        } elseif (empty($subject_ids)) {
            $_SESSION['sync_flash'] = ['type' => 'warning', 'message' => 'No pending subjects selected for enrollment.'];
        } else {
            $stmt = $pdo->prepare("SELECT id, full_name FROM students WHERE id = ? AND academy_reference = ?");
            $stmt->execute([$student_id, $academy_reference]);
            $student = $stmt->fetch();
            
            if (!$student) {
                $_SESSION['sync_flash'] = ['type' => 'danger', 'message' => 'Student not found for this school.'];
            } else {
                $placeholders = implode(',', array_fill(0, count($subject_ids), '?'));
                $nameStmt = $pdo->prepare("SELECT id, name FROM subjects WHERE id IN ($placeholders)");
                $nameStmt->execute($subject_ids);
                $subjectNameMap = [];
                foreach ($nameStmt->fetchAll(PDO::FETCH_ASSOC) as $row) {
                    $subjectNameMap[(int)$row['id']] = $row['name'];
                }
                
                try {
                    $enrollmentHandler = new MoodleEnrollmentHandler($academy_reference, $pdo);
                    $result = $enrollmentHandler->enrollStudentInMultipleSubjects($student_id, $subject_ids);
                    $enrolled = $result['enrolled'] ?? 0;
                    $total = $result['total'] ?? count($subject_ids);
                    $failedCount = $total - $enrolled;
                    
                    if ($enrolled > 0 && $failedCount === 0) {
                        $_SESSION['sync_flash'] = [
                            'type' => 'success',
                            'message' => "{$student['full_name']} enrolled in {$enrolled} subject(s) successfully."
                        ];
                    } elseif ($enrolled > 0) {
                        $errors = [];
                        foreach (($result['results'] ?? []) as $sid => $details) {
                            if (empty($details['success'])) {
                                $errors[] = ($subjectNameMap[$sid] ?? "Subject #{$sid}") . ' - ' . ($details['error'] ?? 'Enrollment failed');
                            }
                        }
                        $_SESSION['sync_flash'] = [
                            'type' => 'warning',
                            'message' => "{$student['full_name']} enrolled in {$enrolled} of {$total} subject(s). Issues: " . implode('; ', $errors)
                        ];
                    } else {
                        $errors = [];
                        foreach (($result['results'] ?? []) as $sid => $details) {
                            if (empty($details['success'])) {
                                $errors[] = ($subjectNameMap[$sid] ?? "Subject #{$sid}") . ' - ' . ($details['error'] ?? 'Enrollment failed');
                            }
                        }
                        if (empty($errors) && isset($result['error'])) {
                            $errors[] = $result['error'];
                        }
                        $_SESSION['sync_flash'] = [
                            'type' => 'danger',
                            'message' => "Failed to enroll {$student['full_name']}: " . implode('; ', $errors ?: ['Unknown error'])
                        ];
                    }
                } catch (Exception $e) {
                    $_SESSION['sync_flash'] = [
                        'type' => 'danger',
                        'message' => "Enrollment error: " . $e->getMessage()
                    ];
                }
            }
        }
        
        header('Location: sync_students.php#pending-enrollments');
        exit;
    } elseif (isset($_POST['sync_students'])) {
        $student_ids = $_POST['student_ids'] ?? [];
        
        if (!empty($student_ids)) {
            $moodleAPI = new MoodleAPI($academy_reference, $pdo);
            $enrollmentHandler = new MoodleEnrollmentHandler($academy_reference, $pdo);
            
            foreach ($student_ids as $student_id) {
            // Get student details
            $stmt = $pdo->prepare("
                SELECT * FROM students WHERE id = ? AND academy_reference = ?
            ");
            $stmt->execute([$student_id, $academy_reference]);
            $student = $stmt->fetch();
            
            if (!$student) continue;
            
            try {
                // Create user in Moodle
                $moodle_result = $moodleAPI->createUser($student);
                
                if ($moodle_result && isset($moodle_result['moodle_user_id'])) {
                    $moodle_user_id = $moodle_result['moodle_user_id'];
                    
                    // Update student record
                    $stmt = $pdo->prepare("
                        UPDATE students 
                        SET moodle_user_id = ?, moodle_synced = 1
                        WHERE id = ?
                    ");
                    $stmt->execute([$moodle_user_id, $student_id]);
                    
                    // Fetch subject assignments
                    $stmt = $pdo->prepare("SELECT subject_id FROM student_subject WHERE student_id = ?");
                    $stmt->execute([$student_id]);
                    $subject_ids = array_unique(array_map('intval', array_column($stmt->fetchAll(), 'subject_id')));
                    
                    $subjectNameMap = [];
                    if (!empty($subject_ids)) {
                        $placeholders = implode(',', array_fill(0, count($subject_ids), '?'));
                        $nameStmt = $pdo->prepare("SELECT id, name FROM subjects WHERE id IN ($placeholders)");
                        $nameStmt->execute($subject_ids);
                        foreach ($nameStmt->fetchAll(PDO::FETCH_ASSOC) as $row) {
                            $subjectNameMap[(int)$row['id']] = $row['name'];
                        }
                    }
                    
                    $enrollmentSummary = [
                        'total' => count($subject_ids),
                        'enrolled' => 0,
                        'failed' => 0,
                        'results' => [],
                        'failed_subjects' => []
                    ];
                    
                    if (!empty($subject_ids)) {
                        $enrollmentSummary = $enrollmentHandler->enrollStudentInMultipleSubjects($student_id, $subject_ids);
                        $totalSubjects = $enrollmentSummary['total'] ?? count($subject_ids);
                        $enrolledSubjects = $enrollmentSummary['enrolled'] ?? 0;
                        $failedSubjects = max(0, $totalSubjects - $enrolledSubjects);
                        $enrollmentSummary['failed'] = $failedSubjects;
                        
                        $failedSubjectNames = [];
                        if (!empty($enrollmentSummary['results'])) {
                            foreach ($enrollmentSummary['results'] as $subjectId => $result) {
                                if (empty($result['success'])) {
                                    $failedSubjectNames[] = $subjectNameMap[(int)$subjectId] ?? ('Subject #' . $subjectId);
                                }
                            }
                        }
                        $enrollmentSummary['failed_subjects'] = $failedSubjectNames;
                    } else {
                        $enrollmentSummary['total'] = 0;
                    }
                    
                    $totalSubjects = $enrollmentSummary['total'];
                    $enrolledSubjects = $enrollmentSummary['enrolled'] ?? 0;
                    $failedSubjects = $enrollmentSummary['failed'] ?? max(0, $totalSubjects - $enrolledSubjects);
                    
                    $messageParts = [];
                    $messageParts[] = isset($moodle_result['existing']) && $moodle_result['existing']
                        ? "Linked to existing Moodle user (ID: {$moodle_user_id})"
                        : "Synced successfully (Moodle ID: {$moodle_user_id})";
                    
                    if ($totalSubjects > 0) {
                        $messageParts[] = "Enrolled in {$enrolledSubjects}/{$totalSubjects} subjects";
                        if ($failedSubjects > 0) {
                            $messageParts[] = "Needs attention: " . implode(', ', $enrollmentSummary['failed_subjects']);
                        }
                    } else {
                        $messageParts[] = 'No subjects assigned yet – Moodle user created only';
                    }
                    
                    if ($failedSubjects > 0) {
                        $partial++;
                        $logStatus = 'warning';
                    } else {
                        $successful++;
                        $logStatus = 'success';
                    }
                    
                    $sync_log[] = [
                        'student' => $student['full_name'],
                        'status' => $logStatus,
                        'message' => implode(' • ', array_filter($messageParts)),
                        'enrollment' => $enrollmentSummary
                    ];
                } else {
                    $failed++;
                    $error_message = 'Failed to create user in Moodle';
                    if (is_array($moodle_result)) {
                        if (!empty($moodle_result['message'])) {
                            $error_message = $moodle_result['message'];
                        } elseif (!empty($moodle_result['error'])) {
                            $error_message = $moodle_result['error'];
                        } elseif (!empty($moodle_result['response']['message'])) {
                            $error_message = $moodle_result['response']['message'];
                        }
                    }
                    error_log("Moodle sync failed for student {$student['id']}: {$error_message}");
                    $sync_log[] = [
                        'student' => $student['full_name'],
                        'status' => 'failed',
                        'message' => $error_message
                    ];
                }
                
            } catch (Exception $e) {
                $failed++;
                $sync_log[] = [
                    'student' => $student['full_name'],
                    'status' => 'failed',
                    'message' => $e->getMessage()
                ];
            }
            }
        }
    }
}

// Get unsynced students
$stmt = $pdo->prepare("
    SELECT s.*, g.name as grade_name,
           GROUP_CONCAT(sub.name SEPARATOR ', ') as subjects
    FROM students s
    LEFT JOIN grades g ON s.grade_id = g.id
    LEFT JOIN student_subject ss ON s.id = ss.student_id
    LEFT JOIN subjects sub ON ss.subject_id = sub.id
    WHERE s.academy_reference = ?
    AND (s.moodle_user_id IS NULL OR s.moodle_synced = 0)
    GROUP BY s.id
    ORDER BY s.created_at DESC
");
$stmt->execute([$academy_reference]);
$unsynced_students = $stmt->fetchAll();

// Students needing enrollment reconciliation (Moodle user exists but courses missing)
$students_pending_enrollment = [];
try {
    $subjectSelectionsStmt = $pdo->prepare("
        SELECT 
            s.id AS student_id,
            s.full_name,
            s.username,
            s.email,
            s.moodle_user_id,
            g.name AS grade_name,
            subj.id AS subject_id,
            subj.name AS subject_name
        FROM students s
        INNER JOIN student_subject ss ON ss.student_id = s.id
        INNER JOIN subjects subj ON subj.id = ss.subject_id
        LEFT JOIN grades g ON g.id = s.grade_id
        WHERE s.academy_reference = ?
          AND s.moodle_user_id IS NOT NULL
        ORDER BY s.full_name
    ");
    $subjectSelectionsStmt->execute([$academy_reference]);
    $subjectSelections = $subjectSelectionsStmt->fetchAll(PDO::FETCH_ASSOC);

    if (!empty($subjectSelections)) {
        $latestStatusStmt = $pdo->prepare("
            SELECT mel.student_id, mel.subject_id, mel.status, mel.attempted_at
            FROM moodle_enrollment_log mel
            INNER JOIN (
                SELECT student_id, subject_id, MAX(attempted_at) AS latest_attempt
                FROM moodle_enrollment_log
                WHERE academy_reference = ? AND action = 'enroll'
                GROUP BY student_id, subject_id
            ) latest ON latest.student_id = mel.student_id
                   AND latest.subject_id = mel.subject_id
                   AND latest.latest_attempt = mel.attempted_at
            WHERE mel.academy_reference = ?
              AND mel.action = 'enroll'
        ");
        $latestStatusStmt->execute([$academy_reference, $academy_reference]);
        $statusRows = $latestStatusStmt->fetchAll(PDO::FETCH_ASSOC);

        $statusMap = [];
        foreach ($statusRows as $row) {
            $sid = (int)$row['student_id'];
            $statusMap[$sid][(int)$row['subject_id']] = [
                'status' => $row['status'],
                'attempted_at' => $row['attempted_at']
            ];
        }

        foreach ($subjectSelections as $row) {
            $studentId = (int)$row['student_id'];
            if (!isset($students_pending_enrollment[$studentId])) {
                $students_pending_enrollment[$studentId] = [
                    'student_id' => $studentId,
                    'full_name' => $row['full_name'],
                    'username' => $row['username'],
                    'email' => $row['email'],
                    'grade_name' => $row['grade_name'],
                    'subjects_selected' => 0,
                    'subjects_enrolled' => 0,
                'missing_subjects' => [],
                'missing_subject_ids' => [],
                    'last_attempt' => null
                ];
            }

            $students_pending_enrollment[$studentId]['subjects_selected']++;

            $statusData = $statusMap[$studentId][$row['subject_id']] ?? null;
            if ($statusData && strtolower($statusData['status']) === 'success') {
                $students_pending_enrollment[$studentId]['subjects_enrolled']++;
            } else {
                $students_pending_enrollment[$studentId]['missing_subjects'][] = $row['subject_name'];
            $students_pending_enrollment[$studentId]['missing_subject_ids'][] = (int)$row['subject_id'];
            }

            if ($statusData && $statusData['attempted_at']) {
                $currentAttempt = $statusData['attempted_at'];
                if ($students_pending_enrollment[$studentId]['last_attempt'] === null ||
                    $currentAttempt > $students_pending_enrollment[$studentId]['last_attempt']) {
                    $students_pending_enrollment[$studentId]['last_attempt'] = $currentAttempt;
                }
            }
        }

        // Filter to only students with missing enrollments
        $students_pending_enrollment = array_values(array_filter(
            $students_pending_enrollment,
            function ($data) {
                return $data['subjects_selected'] > 0
                    && $data['subjects_selected'] > $data['subjects_enrolled'];
            }
        ));

    foreach ($students_pending_enrollment as &$pendingStudent) {
        $pendingStudent['missing_subject_ids'] = array_values(array_unique($pendingStudent['missing_subject_ids']));
        $pendingStudent['missing_subjects'] = array_values(array_unique($pendingStudent['missing_subjects']));
    }
    unset($pendingStudent);
    }
} catch (PDOException $e) {
    $students_pending_enrollment = [];
}

// Get sync statistics
$stmt = $pdo->prepare("
    SELECT 
        COUNT(*) as total,
        SUM(CASE WHEN moodle_user_id IS NOT NULL THEN 1 ELSE 0 END) as synced,
        SUM(CASE WHEN moodle_user_id IS NULL THEN 1 ELSE 0 END) as not_synced
    FROM students
    WHERE academy_reference = ?
");
$stmt->execute([$academy_reference]);
$stats = $stmt->fetch();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Sync Students to Moodle - <?= htmlspecialchars($schoolName) ?></title>
    <?php echoFavicon($academy_reference); ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.min.css" rel="stylesheet">
    <style>
        .stat-card {
            border-radius: 10px;
            padding: 20px;
            text-align: center;
            background: white;
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
        }
        .stat-number {
            font-size: 36px;
            font-weight: bold;
        }
        .pending-card-header {
            background: linear-gradient(135deg, #2b2d6e, #5f43dd);
            border-radius: 1rem 1rem 0 0;
            padding: 1.75rem;
            color: #ffffff;
        }
        .pending-card-body {
            background: #f5f7ff;
            border-radius: 0 0 1rem 1rem;
        }
        .modern-table table {
            border-collapse: separate;
            border-spacing: 0 14px;
        }
        .modern-table thead tr {
            background: rgba(43, 45, 110, 0.05);
            letter-spacing: 0.05em;
        }
        .modern-table tbody tr.pending-row {
            background: #ffffff;
            border-radius: 16px;
            box-shadow: 0 10px 22px rgba(15, 23, 42, 0.08);
            transition: transform 0.2s ease, box-shadow 0.2s ease;
        }
        .modern-table tbody tr.pending-row:hover {
            transform: translateY(-3px);
            box-shadow: 0 18px 32px rgba(27, 35, 83, 0.15);
        }
        .modern-table tbody tr td:first-child {
            border-top-left-radius: 16px;
            border-bottom-left-radius: 16px;
        }
        .modern-table tbody tr td:last-child {
            border-top-right-radius: 16px;
            border-bottom-right-radius: 16px;
        }
        .avatar {
            width: 48px;
            height: 48px;
            border-radius: 50%;
            display: grid;
            place-items: center;
            background: rgba(95, 67, 221, 0.12);
            color: #5f43dd;
            font-size: 1.25rem;
        }
        .info-banner {
            background: #ffffff;
            border-radius: 18px;
            box-shadow: 0 12px 24px rgba(15, 23, 42, 0.1);
        }
        .subject-stack {
            display: flex;
            flex-wrap: wrap;
            gap: 0.35rem;
            align-items: center;
        }
        .subject-pill {
            display: inline-flex;
            align-items: center;
            gap: 0.35rem;
            padding: 0.35rem 0.6rem;
            border-radius: 999px;
            background: rgba(220, 38, 38, 0.12);
            color: #b91c1c;
            font-size: 0.8rem;
            box-shadow: 0 4px 10px rgba(220, 38, 38, 0.15);
            border: 1px solid rgba(220, 38, 38, 0.18);
        }
        .subject-summary {
            font-size: 0.8rem;
            color: #475569;
            cursor: pointer;
            font-weight: 600;
            display: inline-flex;
            align-items: center;
            padding: 0.2rem 0.5rem;
            background: rgba(59, 130, 246, 0.12);
            border-radius: 999px;
            text-decoration: none;
        }
        .subject-summary:hover {
            color: #1d4ed8;
            background: rgba(59, 130, 246, 0.18);
        }
        .action-group {
            display: flex;
            flex-wrap: wrap;
            gap: 0.5rem;
            justify-content: flex-end;
            align-items: center;
        }
        .action-group form {
            margin: 0;
        }
        .action-group .btn {
            border-radius: 999px;
            padding: 0.45rem 1.1rem;
            box-shadow: 0 8px 18px rgba(15, 23, 42, 0.12);
            transition: transform 0.2s ease, box-shadow 0.2s ease;
        }
        .action-group .btn:hover {
            transform: translateY(-1px);
            box-shadow: 0 12px 24px rgba(15, 23, 42, 0.14);
        }
    </style>
</head>
<body class="bg-light">
    <div class="container mt-4 mb-5">
        <!-- Header -->
        <div class="d-flex justify-content-between align-items-center mb-4">
            <div>
                <h2><i class="bi bi-cloud-upload text-primary"></i> Sync Students to Moodle</h2>
                <p class="text-muted mb-0">Manually sync students to your Moodle LMS</p>
            </div>
            <div>
                <a href="index.php" class="btn btn-outline-secondary me-2">
                    <i class="bi bi-arrow-left"></i> Moodle Hub
                </a>
                <a href="../dashboard.php" class="btn btn-outline-secondary">
                    <i class="bi bi-house"></i> Dashboard
                </a>
            </div>
        </div>

        <?php if ($flashMessage): ?>
        <div class="alert alert-<?= $flashMessage['type'] ?? 'info' ?> alert-dismissible fade show">
            <i class="bi bi-info-circle"></i>
            <?= htmlspecialchars($flashMessage['message'] ?? '') ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php endif; ?>

        <!-- Sync Results -->
        <?php if (!empty($sync_log)): ?>
        <div class="alert alert-<?= $failed === 0 && $partial === 0 ? 'success' : 'warning' ?> alert-dismissible fade show">
            <h5>
                <i class="bi bi-<?= $failed === 0 && $partial === 0 ? 'check-circle' : 'exclamation-triangle' ?>-fill"></i> 
                Sync Complete
            </h5>
            <p class="mb-2">
                <strong>Full success:</strong> <?= $successful ?> |
                <strong>With enrollment issues:</strong> <?= $partial ?> |
                <strong>Failed:</strong> <?= $failed ?>
            </p>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            
            <details class="mt-2">
                <summary class="btn btn-sm btn-outline-dark">View Details</summary>
                <div class="mt-2">
                    <?php foreach ($sync_log as $log): ?>
                    <?php
                        $statusClass = 'text-danger';
                        $statusIcon = 'x-circle';
                        if ($log['status'] === 'success') {
                            $statusClass = 'text-success';
                            $statusIcon = 'check-circle';
                        } elseif ($log['status'] === 'warning') {
                            $statusClass = 'text-warning';
                            $statusIcon = 'exclamation-triangle';
                        }
                    ?>
                    <div class="<?= $statusClass ?>">
                        <i class="bi bi-<?= $statusIcon ?>"></i>
                        <?= htmlspecialchars($log['student']) ?>: <?= htmlspecialchars($log['message']) ?>
                    </div>
                    <?php endforeach; ?>
                </div>
            </details>
        </div>
        <?php endif; ?>

        <!-- Statistics -->
        <div class="row mb-4">
            <div class="col-md-4">
                <div class="stat-card">
                    <i class="bi bi-people text-primary" style="font-size: 32px;"></i>
                    <div class="stat-number text-primary"><?= $stats['total'] ?></div>
                    <small class="text-muted">Total Students</small>
                </div>
            </div>
            <div class="col-md-4">
                <div class="stat-card">
                    <i class="bi bi-check-circle text-success" style="font-size: 32px;"></i>
                    <div class="stat-number text-success"><?= $stats['synced'] ?></div>
                    <small class="text-muted">Synced to Moodle</small>
                </div>
            </div>
            <div class="col-md-4">
                <div class="stat-card">
                    <i class="bi bi-exclamation-circle text-warning" style="font-size: 32px;"></i>
                    <div class="stat-number text-warning"><?= $stats['not_synced'] ?></div>
                    <small class="text-muted">Not Synced</small>
                </div>
            </div>
        </div>

        <!-- Students to Sync -->
        <div class="card">
            <div class="card-header bg-white">
                <h5 class="mb-0">Unsynced Students (<?= count($unsynced_students) ?>)</h5>
            </div>
            <div class="card-body">
                <?php if (empty($unsynced_students)): ?>
                <div class="text-center py-4 text-muted">
                    <i class="bi bi-check-circle" style="font-size: 64px;"></i>
                    <h4 class="mt-3">All Students Synced!</h4>
                    <p>All students in your database are synced to Moodle.</p>
                </div>
                <?php else: ?>
                <form method="POST">
                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead class="table-light">
                                <tr>
                                    <th>
                                        <input type="checkbox" id="select-all" class="form-check-input">
                                    </th>
                                    <th>Student</th>
                                    <th>Grade</th>
                                    <th>Subjects</th>
                                    <th>Enrolled</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($unsynced_students as $student): ?>
                                <tr>
                                    <td>
                                        <input type="checkbox" name="student_ids[]" 
                                               value="<?= $student['id'] ?>" 
                                               class="form-check-input student-checkbox">
                                    </td>
                                    <td>
                                        <strong><?= htmlspecialchars($student['full_name']) ?></strong><br>
                                        <small class="text-muted">
                                            <?= htmlspecialchars($student['username']) ?> | 
                                            <?= htmlspecialchars($student['email']) ?>
                                        </small>
                                    </td>
                                    <td><?= htmlspecialchars($student['grade_name']) ?></td>
                                    <td>
                                        <small><?= htmlspecialchars($student['subjects'] ?? 'No subjects') ?></small>
                                    </td>
                                    <td>
                                        <?= date('M j, Y', strtotime($student['created_at'])) ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    
                    <div class="d-flex justify-content-between align-items-center mt-3">
                        <div>
                            <span id="selected-count">0</span> students selected
                        </div>
                        <div>
                            <button type="submit" name="sync_students" class="btn btn-primary btn-lg">
                                <i class="bi bi-cloud-upload"></i> Sync Selected Students to Moodle
                            </button>
                        </div>
                    </div>
                </form>
                <?php endif; ?>
            </div>
        </div>

        <?php if (!empty($students_pending_enrollment)): ?>
        <div class="card mt-4 border-0 shadow-lg" id="pending-enrollments">
            <div class="card-header pending-card-header border-0">
                <div class="d-flex flex-column flex-md-row align-items-start align-items-md-center justify-content-between gap-3">
                    <div>
                        <span class="badge bg-white text-primary text-uppercase fw-semibold small rounded-pill px-3 py-2 mb-2 shadow-sm">
                            Enrollment Watchlist
                        </span>
                        <h4 class="mb-1 fw-semibold">Students Needing Moodle Enrollment Attention</h4>
                        <p class="mb-0 text-white-50">Continue enrollment for Moodle users who still have pending subjects.</p>
                    </div>
                    <span class="badge bg-light text-primary fs-6 px-3 py-2 shadow-sm">
                        <i class="bi bi-people-fill me-1"></i><?= count($students_pending_enrollment) ?> Pending
                    </span>
                </div>
            </div>
            <div class="card-body pending-card-body p-4">
                <div class="table-responsive modern-table">
                    <table class="table table-borderless align-middle mb-0">
                        <thead>
                            <tr class="text-uppercase small text-muted">
                                <th style="width:22%">Student</th>
                                <th style="width:12%">Grade</th>
                                <th class="text-center" style="width:12%">Selected</th>
                                <th class="text-center" style="width:12%">Enrolled</th>
                                <th>Missing Subjects</th>
                                <th class="text-end" style="width:14%">Last Attempt</th>
                                <th class="text-end" style="width:16%">Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($students_pending_enrollment as $student): ?>
                            <tr class="pending-row">
                                <td>
                                    <div class="d-flex align-items-center">
                                        <div class="avatar me-3">
                                            <i class="bi bi-person-fill"></i>
                                        </div>
                                        <div>
                                            <div class="fw-semibold text-dark"><?= htmlspecialchars($student['full_name']) ?></div>
                                            <div class="text-muted small">
                                                <?= htmlspecialchars($student['username']) ?> · <?= htmlspecialchars($student['email']) ?>
                                            </div>
                                        </div>
                                    </div>
                                </td>
                                <td class="fw-semibold text-muted"><?= htmlspecialchars($student['grade_name'] ?? '—') ?></td>
                                <td class="text-center">
                                    <span class="badge bg-secondary-subtle text-secondary px-3 py-2 rounded-pill shadow-sm">
                                        <?= (int)$student['subjects_selected'] ?>
                                    </span>
                                </td>
                                <td class="text-center">
                                    <span class="badge bg-success-subtle text-success px-3 py-2 rounded-pill shadow-sm">
                                        <?= (int)$student['subjects_enrolled'] ?>
                                    </span>
                                </td>
                                <td>
                                    <?php
                                        $missingList = array_values(array_unique($student['missing_subjects']));
                                        $displaySubjects = array_slice($missingList, 0, 2);
                                        $remainingSubjects = array_slice($missingList, 2);
                                    ?>
                                    <div class="subject-stack">
                                        <?php foreach ($displaySubjects as $subjectName): ?>
                                            <span class="subject-pill">
                                                <i class="bi bi-exclamation-triangle"></i><?= htmlspecialchars($subjectName) ?>
                                            </span>
                                        <?php endforeach; ?>
                                        <?php if (!empty($remainingSubjects)): ?>
                                            <span class="subject-summary" data-bs-toggle="tooltip" data-bs-placement="top"
                                                  title="<?= htmlspecialchars(implode(', ', $remainingSubjects)) ?>">
                                                +<?= count($remainingSubjects) ?> more
                                            </span>
                                        <?php endif; ?>
                                    </div>
                                </td>
                                <td class="text-end">
                                    <?php if (!empty($student['last_attempt'])): ?>
                                        <span class="badge bg-dark-subtle text-dark px-3 py-2 shadow-sm">
                                            <i class="bi bi-clock-history me-1"></i><?= date('M j, Y H:i', strtotime($student['last_attempt'])) ?>
                                        </span>
                                    <?php else: ?>
                                        <span class="text-muted fst-italic">No attempts yet</span>
                                    <?php endif; ?>
                                </td>
                                <td class="text-end">
                                    <?php
                                        $enrollSubjectIds = implode(',', $student['missing_subject_ids'] ?? []);
                                    ?>
                                    <div class="action-group">
                                        <form method="post">
                                            <input type="hidden" name="student_id" value="<?= (int)$student['student_id'] ?>">
                                            <input type="hidden" name="subject_ids" value="<?= htmlspecialchars($enrollSubjectIds) ?>">
                                            <button type="submit" name="enroll_missing" class="btn btn-primary btn-sm">
                                                <i class="bi bi-arrow-repeat me-1"></i> Enroll Now
                                            </button>
                                        </form>
                                        <a href="../edit.php?id=<?= (int)$student['student_id'] ?>" class="btn btn-outline-secondary btn-sm" target="_blank">
                                            <i class="bi bi-pencil-square me-1"></i> Edit Subjects
                                        </a>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                <div class="info-banner d-flex flex-column flex-lg-row align-items-lg-center justify-content-lg-between gap-3 mt-4 p-3">
                    <div class="d-flex align-items-start text-muted">
                        <span class="badge bg-primary-subtle text-primary me-3 mt-1">
                            <i class="bi bi-info-circle"></i>
                        </span>
                        <div>
                            <strong>Tip:</strong> These students already have Moodle accounts. Retry enrollment instantly or adjust their subject list before trying again.
                        </div>
                    </div>
                    <a href="configure_mapping.php" class="btn btn-light border rounded-pill shadow-sm">
                        <i class="bi bi-gear-wide-connected me-1"></i> Review Subject-Course Mapping
                    </a>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <!-- Help -->
        <div class="card mt-4">
            <div class="card-body">
                <h5><i class="bi bi-info-circle"></i> How This Works</h5>
                <ol>
                    <li>Select students you want to sync to Moodle</li>
                    <li>Click "Sync Selected Students"</li>
                    <li>System creates Moodle user accounts for each student</li>
                    <li>Automatically enrolls them in their assigned courses</li>
                    <li>Students can then login to Moodle with their credentials</li>
                </ol>
                <div class="alert alert-warning mt-3 mb-0">
                    <i class="bi bi-exclamation-triangle"></i> <strong>Note:</strong> 
                    Make sure subject-course mappings are configured before syncing.
                    <a href="configure_mapping.php" class="alert-link">Configure mappings</a>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Select all checkbox
        document.getElementById('select-all').addEventListener('change', function() {
            const checkboxes = document.querySelectorAll('.student-checkbox');
            checkboxes.forEach(cb => cb.checked = this.checked);
            updateSelectedCount();
        });
        
        // Update selected count
        document.querySelectorAll('.student-checkbox').forEach(cb => {
            cb.addEventListener('change', updateSelectedCount);
        });
        
        function updateSelectedCount() {
            const checked = document.querySelectorAll('.student-checkbox:checked').length;
            document.getElementById('selected-count').textContent = checked;
        }

        const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle=\"tooltip\"]'));
        tooltipTriggerList.forEach(function (tooltipTriggerEl) {
            new bootstrap.Tooltip(tooltipTriggerEl);
        });
    </script>
</body>
</html>
