<?php
/**
 * Moodle Management Hub
 * Central page for all Moodle-related configuration and testing
 */

session_start();
require_once '../../includes/functions.php';
require_once '../../includes/school_context.php';
require_once '../../includes/favicon_helper.php';
requireAdminLogin();

$pdo = getDB();
$academy_reference = getCurrentSchool();
$schoolName = getSchoolName();

// Get Moodle statistics
$stmt = $pdo->prepare("
    SELECT 
        COUNT(*) as total,
        SUM(CASE WHEN moodle_user_id IS NOT NULL THEN 1 ELSE 0 END) as synced,
        SUM(CASE WHEN moodle_user_id IS NULL THEN 1 ELSE 0 END) as not_synced,
        SUM(CASE WHEN suspended = 1 THEN 1 ELSE 0 END) as suspended
    FROM students
    WHERE academy_reference = ?
");
$stmt->execute([$academy_reference]);
$student_stats = $stmt->fetch();

// Get subject-course mappings
$stmt = $pdo->prepare("
    SELECT COUNT(*) as count FROM subject_moodle_mapping 
    WHERE academy_reference = ?
");
$stmt->execute([$academy_reference]);
$mapping_count = $stmt->fetchColumn();

// Get recent sync logs
try {
    $stmt = $pdo->prepare("
        SELECT COUNT(*) as count FROM moodle_sync_log 
        WHERE academy_reference = ?
    ");
    $stmt->execute([$academy_reference]);
    $sync_log_count = $stmt->fetchColumn();
} catch (Exception $e) {
    $sync_log_count = 0;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Moodle Management - <?= htmlspecialchars($schoolName) ?></title>
    <?php echoFavicon($academy_reference); ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.min.css" rel="stylesheet">
    <style>
        .moodle-card {
            border-radius: 15px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            transition: transform 0.3s;
            cursor: pointer;
            border: 2px solid transparent;
        }
        .moodle-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 5px 20px rgba(0,0,0,0.15);
            border-color: #667eea;
        }
        .stat-box {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 20px;
            border-radius: 10px;
            text-align: center;
        }
        .stat-number {
            font-size: 48px;
            font-weight: bold;
        }
    </style>
</head>
<body class="bg-light">
    <div class="container mt-4 mb-5">
        <!-- Header -->
        <div class="d-flex justify-content-between align-items-center mb-4">
            <div>
                <h2><i class="bi bi-mortarboard-fill text-primary"></i> Moodle Management</h2>
                <p class="text-muted mb-0">Configure and manage Moodle integration for <?= htmlspecialchars($schoolName) ?></p>
            </div>
            <a href="../dashboard.php" class="btn btn-outline-secondary">
                <i class="bi bi-arrow-left"></i> Back to Dashboard
            </a>
        </div>

        <!-- Statistics -->
        <div class="row mb-4">
            <div class="col-md-3">
                <div class="stat-box">
                    <i class="bi bi-people" style="font-size: 32px;"></i>
                    <div class="stat-number"><?= $student_stats['total'] ?></div>
                    <small>Total Students</small>
                </div>
            </div>
            <div class="col-md-3">
                <div class="stat-box" style="background: linear-gradient(135deg, #28a745 0%, #20c997 100%);">
                    <i class="bi bi-check-circle" style="font-size: 32px;"></i>
                    <div class="stat-number"><?= $student_stats['synced'] ?></div>
                    <small>Synced to Moodle</small>
                </div>
            </div>
            <div class="col-md-3">
                <div class="stat-box" style="background: linear-gradient(135deg, #ffc107 0%, #ff9800 100%);">
                    <i class="bi bi-exclamation-triangle" style="font-size: 32px;"></i>
                    <div class="stat-number"><?= $student_stats['not_synced'] ?></div>
                    <small>Not Synced</small>
                </div>
            </div>
            <div class="col-md-3">
                <div class="stat-box" style="background: linear-gradient(135deg, #dc3545 0%, #bd2130 100%);">
                    <i class="bi bi-ban" style="font-size: 32px;"></i>
                    <div class="stat-number"><?= $student_stats['suspended'] ?></div>
                    <small>Suspended</small>
                </div>
            </div>
        </div>

        <!-- Quick Actions -->
        <div class="row mb-4">
            <div class="col-md-3">
                <div class="card moodle-card" onclick="location.href='settings.php'">
                    <div class="card-body text-center p-4">
                        <i class="bi bi-gear" style="font-size: 64px; color: #667eea;"></i>
                        <h4 class="mt-3">API Settings</h4>
                        <p class="text-muted">Configure Moodle API</p>
                        <span class="badge bg-primary">Configure</span>
                    </div>
                </div>
            </div>
            
            <div class="col-md-3">
                <div class="card moodle-card" onclick="location.href='configure_mapping.php'">
                    <div class="card-body text-center p-4">
                        <i class="bi bi-diagram-3" style="font-size: 64px; color: #28a745;"></i>
                        <h4 class="mt-3">Mappings</h4>
                        <p class="text-muted">Map subjects to courses</p>
                        <span class="badge bg-success"><?= $mapping_count ?> configured</span>
                    </div>
                </div>
            </div>
            
            <div class="col-md-3">
                <div class="card moodle-card" onclick="location.href='sync_students.php'">
                    <div class="card-body text-center p-4">
                        <i class="bi bi-cloud-upload" style="font-size: 64px; color: #17a2b8;"></i>
                        <h4 class="mt-3">Sync Students</h4>
                        <p class="text-muted">Sync to Moodle</p>
                        <span class="badge bg-<?= $student_stats['not_synced'] > 0 ? 'warning' : 'success' ?>">
                            <?= $student_stats['not_synced'] ?> to sync
                        </span>
                    </div>
                </div>
            </div>
            
            <div class="col-md-3">
                <div class="card moodle-card" onclick="location.href='test_connection.php'">
                    <div class="card-body text-center p-4">
                        <i class="bi bi-plug" style="font-size: 64px; color: #ffc107;"></i>
                        <h4 class="mt-3">Test Connection</h4>
                        <p class="text-muted">Verify API access</p>
                        <span class="badge bg-warning text-dark">Diagnostic</span>
                    </div>
                </div>
            </div>
        </div>

        <div class="row mb-4">
            <div class="col-md-4">
                <div class="card moodle-card" onclick="location.href='test_mappings.php'">
                    <div class="card-body text-center p-4">
                        <i class="bi bi-list-check" style="font-size: 64px; color: #6610f2;"></i>
                        <h4 class="mt-3">Test Mappings</h4>
                        <p class="text-muted">Verify mappings</p>
                        <span class="badge bg-secondary">Validation</span>
                    </div>
                </div>
            </div>
            
            <div class="col-md-4">
                <div class="card moodle-card" onclick="window.open('/Multi-Tanent/test_moodle_functions.php?key=test123', '_blank')">
                    <div class="card-body text-center p-4">
                        <i class="bi bi-clipboard-check" style="font-size: 64px; color: #dc3545;"></i>
                        <h4 class="mt-3">Test API</h4>
                        <p class="text-muted">Test API functions</p>
                        <span class="badge bg-danger">10 tests</span>
                    </div>
                </div>
            </div>
            
            <div class="col-md-4">
                <div class="card moodle-card" onclick="window.open('/Multi-Tanent/test_moodle_complete_workflow.php?key=test123', '_blank')">
                    <div class="card-body text-center p-4">
                        <i class="bi bi-gear-wide-connected" style="font-size: 64px; color: #0dcaf0;"></i>
                        <h4 class="mt-3">Workflow Test</h4>
                        <p class="text-muted">End-to-end test</p>
                        <span class="badge bg-info">Full workflow</span>
                    </div>
                </div>
            </div>
        </div>

        <!-- Configuration Info -->
        <div class="card">
            <div class="card-header bg-primary text-white">
                <h5 class="mb-0"><i class="bi bi-info-circle"></i> Moodle Integration Status</h5>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-6">
                        <h6>Configuration</h6>
                        <p><strong>Moodle URL:</strong> <?= htmlspecialchars(MOODLE_URL) ?></p>
                        <p><strong>API Token:</strong> <?= substr(MOODLE_API_TOKEN, 0, 20) ?>...</p>
                    </div>
                    <div class="col-md-6">
                        <h6>Status</h6>
                        <p><strong>Mappings:</strong> <?= $mapping_count ?> configured</p>
                        <p><strong>Sync Logs:</strong> <?= $sync_log_count ?> records</p>
                    </div>
                </div>
                
                <?php if ($student_stats['not_synced'] > 0): ?>
                <div class="alert alert-warning mt-3 mb-0">
                    <i class="bi bi-exclamation-triangle"></i> 
                    <strong><?= $student_stats['not_synced'] ?> students</strong> are not synced to Moodle yet.
                    <a href="sync_students.php" class="alert-link">Sync them now</a>
                </div>
                <?php endif; ?>
                
                <?php if ($mapping_count == 0): ?>
                <div class="alert alert-info mt-3 mb-0">
                    <i class="bi bi-info-circle"></i> 
                    <strong>Setup Required:</strong> Configure subject-course mappings first.
                    <a href="configure_mapping.php" class="alert-link">Configure now</a>
                </div>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Help Section -->
        <div class="card mt-4">
            <div class="card-body">
                <h5><i class="bi bi-question-circle"></i> Quick Guide</h5>
                <ol>
                    <li><strong>Configure Mappings:</strong> Map your subjects to Moodle courses</li>
                    <li><strong>Test API:</strong> Verify Moodle connection is working</li>
                    <li><strong>Test Workflow:</strong> Test the complete sync and suspension process</li>
                    <li><strong>Sync Students:</strong> Students will auto-sync when created, or run manual sync</li>
                </ol>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

