<?php
/**
 * Diagnose Moodle Sync Errors
 * Detailed diagnostics for troubleshooting student sync failures
 */

session_start();
require_once '../../includes/functions.php';
require_once '../../includes/school_context.php';
require_once '../../includes/moodle_api.php';
requireAdminLogin();

$pdo = getDB();
$academy_reference = getCurrentSchool();
$schoolName = getSchoolName();

$student_id = $_GET['student_id'] ?? null;

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Moodle Sync Diagnostics - <?= htmlspecialchars($schoolName) ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.min.css" rel="stylesheet">
    <style>
        .diagnostic-section {
            background: white;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 20px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
        }
        .status-good { color: #28a745; }
        .status-bad { color: #dc3545; }
        .status-warning { color: #ffc107; }
        pre {
            background: #f8f9fa;
            padding: 15px;
            border-radius: 5px;
            overflow-x: auto;
        }
    </style>
</head>
<body class="bg-light">
    <div class="container mt-4 mb-5">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <div>
                <h2><i class="bi bi-bug text-danger"></i> Moodle Sync Diagnostics</h2>
                <p class="text-muted mb-0">Detailed troubleshooting for Moodle integration</p>
            </div>
            <div>
                <a href="sync_students.php" class="btn btn-outline-secondary">
                    <i class="bi bi-arrow-left"></i> Back to Sync
                </a>
            </div>
        </div>

        <!-- 1. Moodle Configuration Check -->
        <div class="diagnostic-section">
            <h4><i class="bi bi-gear"></i> 1. Moodle Configuration</h4>
            <?php
            $moodle = new MoodleAPI($academy_reference, $pdo);
            $is_enabled = $moodle->isEnabled();
            ?>
            <table class="table table-sm">
                <tr>
                    <td><strong>Integration Enabled:</strong></td>
                    <td class="<?= $is_enabled ? 'status-good' : 'status-bad' ?>">
                        <?= $is_enabled ? '✓ YES' : '✗ NO' ?>
                    </td>
                </tr>
            </table>
            
            <?php if (!$is_enabled): ?>
            <div class="alert alert-danger">
                <i class="bi bi-exclamation-triangle"></i> 
                <strong>Moodle is not configured!</strong>
                <a href="settings.php" class="alert-link">Configure Moodle Settings</a>
            </div>
            <?php endif; ?>
        </div>

        <!-- 2. Connection Test -->
        <div class="diagnostic-section">
            <h4><i class="bi bi-wifi"></i> 2. Moodle Connection Test</h4>
            <?php if ($is_enabled): ?>
                <?php
                $test_result = $moodle->testConnection();
                ?>
                <table class="table table-sm">
                    <tr>
                        <td><strong>Connection Status:</strong></td>
                        <td class="<?= $test_result['success'] ? 'status-good' : 'status-bad' ?>">
                            <?= $test_result['success'] ? '✓ CONNECTED' : '✗ FAILED' ?>
                        </td>
                    </tr>
                    <?php if ($test_result['success']): ?>
                    <tr>
                        <td><strong>Site Name:</strong></td>
                        <td><?= htmlspecialchars($test_result['site_name'] ?? 'N/A') ?></td>
                    </tr>
                    <tr>
                        <td><strong>Moodle Version:</strong></td>
                        <td><?= htmlspecialchars($test_result['version'] ?? 'N/A') ?></td>
                    </tr>
                    <?php else: ?>
                    <tr>
                        <td><strong>Error:</strong></td>
                        <td class="status-bad"><?= htmlspecialchars($test_result['error'] ?? 'Unknown error') ?></td>
                    </tr>
                    <?php endif; ?>
                </table>
            <?php else: ?>
                <div class="alert alert-warning">Skipped - Moodle not configured</div>
            <?php endif; ?>
        </div>

        <!-- 3. API Permissions Check -->
        <div class="diagnostic-section">
            <h4><i class="bi bi-shield-check"></i> 3. API Permissions</h4>
            <?php if ($is_enabled): ?>
                <p class="small text-muted">Testing if your API token has required permissions...</p>
                <?php
                // Test core_user_create_users permission
                $test_user_data = [
                    'users[0][username]' => 'test_diagnostic_' . time(),
                    'users[0][password]' => 'TestPass123!@#',
                    'users[0][firstname]' => 'Test',
                    'users[0][lastname]' => 'Diagnostic',
                    'users[0][email]' => 'test_diagnostic_' . time() . '@example.com',
                    'users[0][auth]' => 'manual',
                    'users[0][createpassword]' => 0
                ];
                
                // We won't actually create, just test the API call
                echo "<div class='small'><strong>Required Functions:</strong></div>";
                echo "<ul class='small'>";
                echo "<li><code>core_user_create_users</code> - Create Moodle users</li>";
                echo "<li><code>enrol_manual_enrol_users</code> - Enroll students in courses</li>";
                echo "<li><code>core_course_get_courses</code> - Fetch courses</li>";
                echo "</ul>";
                
                echo "<div class='alert alert-info'>";
                echo "<i class='bi bi-info-circle'></i> ";
                echo "Make sure your Moodle API token has a service with these functions enabled.";
                echo "</div>";
                ?>
            <?php else: ?>
                <div class="alert alert-warning">Skipped - Moodle not configured</div>
            <?php endif; ?>
        </div>

        <!-- 4. Recent Sync Log -->
        <div class="diagnostic-section">
            <h4><i class="bi bi-clock-history"></i> 4. Recent Sync Attempts</h4>
            <?php
            try {
                $stmt = $pdo->prepare("
                    SELECT * FROM moodle_sync_log 
                    WHERE academy_reference = ? 
                    AND action_type = 'user_create'
                    ORDER BY created_at DESC 
                    LIMIT 5
                ");
                $stmt->execute([$academy_reference]);
                $logs = $stmt->fetchAll();
                
                if (empty($logs)): ?>
                    <p class="text-muted">No sync attempts logged yet.</p>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-sm table-hover">
                            <thead>
                                <tr>
                                    <th>Time</th>
                                    <th>Student ID</th>
                                    <th>Status</th>
                                    <th>Error</th>
                                    <th>Details</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($logs as $log): ?>
                                <tr>
                                    <td><?= date('M j, H:i:s', strtotime($log['created_at'])) ?></td>
                                    <td><?= $log['student_id'] ?></td>
                                    <td class="<?= $log['sync_status'] === 'success' ? 'status-good' : 'status-bad' ?>">
                                        <?= htmlspecialchars($log['sync_status']) ?>
                                    </td>
                                    <td><?= htmlspecialchars($log['error_message'] ?? '-') ?></td>
                                    <td>
                                        <button class="btn btn-sm btn-outline-primary" 
                                                onclick="showDetails(<?= htmlspecialchars(json_encode($log)) ?>)">
                                            View
                                        </button>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif;
            } catch (Exception $e) {
                echo "<div class='alert alert-warning'>Could not load sync logs: " . htmlspecialchars($e->getMessage()) . "</div>";
            }
            ?>
        </div>

        <!-- 5. Test with Specific Student -->
        <?php if ($student_id): ?>
        <div class="diagnostic-section">
            <h4><i class="bi bi-person-check"></i> 5. Student-Specific Test</h4>
            <?php
            $stmt = $pdo->prepare("SELECT * FROM students WHERE id = ? AND academy_reference = ?");
            $stmt->execute([$student_id, $academy_reference]);
            $student = $stmt->fetch();
            
            if ($student):
            ?>
                <h5><?= htmlspecialchars($student['full_name']) ?></h5>
                <table class="table table-sm">
                    <tr>
                        <td><strong>Username:</strong></td>
                        <td><?= htmlspecialchars($student['username']) ?></td>
                    </tr>
                    <tr>
                        <td><strong>Email:</strong></td>
                        <td><?= htmlspecialchars($student['email']) ?></td>
                    </tr>
                    <tr>
                        <td><strong>Already Synced:</strong></td>
                        <td class="<?= $student['moodle_user_id'] ? 'status-good' : 'status-warning' ?>">
                            <?= $student['moodle_user_id'] ? '✓ YES (ID: ' . $student['moodle_user_id'] . ')' : '✗ NO' ?>
                        </td>
                    </tr>
                </table>
                
                <button class="btn btn-primary" onclick="testSyncStudent(<?= $student_id ?>)">
                    <i class="bi bi-play-circle"></i> Test Sync This Student
                </button>
                <div id="test-result" class="mt-3"></div>
            <?php else: ?>
                <div class="alert alert-danger">Student not found</div>
            <?php endif; ?>
        </div>
        <?php endif; ?>
    </div>

    <!-- Details Modal -->
    <div class="modal fade" id="detailsModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Sync Log Details</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div id="modal-content"></div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function showDetails(log) {
            const content = `
                <h6>Request Data:</h6>
                <pre>${log.request_data || 'N/A'}</pre>
                <h6>Response Data:</h6>
                <pre>${log.response_data || 'N/A'}</pre>
                <h6>Error:</h6>
                <pre>${log.error_message || 'None'}</pre>
                <h6>Execution Time:</h6>
                <p>${log.execution_time || '0'}s</p>
            `;
            document.getElementById('modal-content').innerHTML = content;
            new bootstrap.Modal(document.getElementById('detailsModal')).show();
        }
    </script>
</body>
</html>

