<?php
/**
 * Moodle Subject Mapping Configuration
 * Admin interface to map internal subjects to Moodle courses
 */

require_once '../../includes/functions.php';
require_once '../../includes/school_context.php';
require_once '../../includes/moodle_api.php';
require_once '../../includes/favicon_helper.php';
requireAdminLogin();

$academy_reference = getCurrentSchool();
if (!$academy_reference) {
    die("Error: No school selected.");
}

$pdo = getDB();
$schoolName = getSchoolName();
$moodle = new MoodleAPI($academy_reference, $pdo);

$success = '';
$error = '';

// Initialize moodle_courses array early to prevent undefined variable errors
$moodle_courses = [];

// Fetch available Moodle courses if enabled
if ($moodle->isEnabled()) {
    try {
        $result = $moodle->getCourses();
        if (isset($result['success']) && $result['success']) {
            $moodle_courses = $result['data'] ?? [];
        } elseif (is_array($result) && !isset($result['success'])) {
            // If it returns courses directly (legacy behavior)
            $moodle_courses = $result;
        }
    } catch (Exception $e) {
        error_log("Initial course fetch error: " . $e->getMessage());
    }
}

// Handle fetch courses from Moodle
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['fetch_courses'])) {
    try {
        $result = $moodle->getCourses();
        if (isset($result['success']) && $result['success']) {
            $moodle_courses = $result['data'] ?? [];
            $success = "Successfully fetched " . count($moodle_courses) . " courses from Moodle!";
        } elseif (is_array($result) && !isset($result['success'])) {
            // Legacy behavior - courses returned directly
            $moodle_courses = $result;
            $success = "Successfully fetched " . count($moodle_courses) . " courses from Moodle!";
        } else {
            $error = "Failed to fetch courses: " . ($result['error'] ?? 'Unknown error');
        }
    } catch (Exception $e) {
        $error = "Failed to fetch courses: " . $e->getMessage();
    }
}

// Handle bulk mapping by grade
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['bulk_map_grade'])) {
    $grade_id = (int)($_POST['grade_id'] ?? 0);
    $course_prefix = trim($_POST['course_prefix'] ?? '');
    $auto_enroll = isset($_POST['bulk_auto_enroll']) ? 1 : 0;
    
    if ($grade_id > 0) {
        try {
            // Get all subjects for this grade
            $stmt = $pdo->prepare("
                SELECT s.id, s.name, s.code
                FROM subjects s
                INNER JOIN grade_subject gs ON s.id = gs.subject_id
                WHERE gs.grade_id = ?
            ");
            $stmt->execute([$grade_id]);
            $grade_subjects = $stmt->fetchAll();
            
            $mapped = 0;
            foreach ($grade_subjects as $subject) {
                // Try to find matching Moodle course by name pattern
                $subject_name_pattern = $subject['name'];
                $matching_course = null;
                
                foreach ($moodle_courses as $course) {
                    if (stripos($course['fullname'], $subject_name_pattern) !== false ||
                        stripos($course['fullname'], $subject['code']) !== false) {
                        $matching_course = $course;
                        break;
                    }
                }
                
                if ($matching_course) {
                    // Check if mapping exists
                    $check = $pdo->prepare("
                        SELECT id FROM subject_moodle_mapping 
                        WHERE academy_reference = ? AND subject_id = ?
                    ");
                    $check->execute([$academy_reference, $subject['id']]);
                    $existing = $check->fetch();
                    
                    if ($existing) {
                        // Update existing
                        $stmt = $pdo->prepare("
                            UPDATE subject_moodle_mapping SET
                                moodle_course_id = ?,
                                moodle_course_shortname = ?,
                                moodle_course_idnumber = ?,
                                auto_enroll = ?,
                                updated_at = NOW()
                            WHERE id = ?
                        ");
                        $stmt->execute([
                            $matching_course['id'],
                            $matching_course['shortname'],
                            $matching_course['idnumber'] ?? null,
                            $auto_enroll,
                            $existing['id']
                        ]);
                    } else {
                        // Insert new
                        $stmt = $pdo->prepare("
                            INSERT INTO subject_moodle_mapping 
                            (academy_reference, subject_id, moodle_course_id,
                             moodle_course_shortname, moodle_course_idnumber, auto_enroll, 
                             sync_enabled, created_by)
                            VALUES (?, ?, ?, ?, ?, ?, 1, ?)
                        ");
                        $stmt->execute([
                            $academy_reference,
                            $subject['id'],
                            $matching_course['id'],
                            $matching_course['shortname'],
                            $matching_course['idnumber'] ?? null,
                            $auto_enroll,
                            $_SESSION['user_id'] ?? null
                        ]);
                    }
                    $mapped++;
                }
            }
            
            $success = "Bulk mapping complete! Mapped {$mapped} out of " . count($grade_subjects) . " subjects.";
        } catch (Exception $e) {
            $error = "Bulk mapping error: " . $e->getMessage();
        }
    }
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['save_mapping'])) {
    $subject_id = (int)($_POST['subject_id'] ?? 0);
    $moodle_course_id = trim($_POST['moodle_course_id'] ?? '');
    $moodle_course_shortname = trim($_POST['moodle_course_shortname'] ?? '');
    $moodle_course_idnumber = trim($_POST['moodle_course_idnumber'] ?? '');
    $auto_enroll = isset($_POST['auto_enroll']) ? 1 : 0;
    $sync_enabled = isset($_POST['sync_enabled']) ? 1 : 0;
    
    if ($subject_id > 0) {
        try {
            // Check if mapping exists
            $stmt = $pdo->prepare("
                SELECT id FROM subject_moodle_mapping 
                WHERE academy_reference = ? AND subject_id = ?
            ");
            $stmt->execute([$academy_reference, $subject_id]);
            $existing = $stmt->fetch();
            
            if ($existing) {
                // Update
                $stmt = $pdo->prepare("
                    UPDATE subject_moodle_mapping SET
                        moodle_course_id = ?,
                        moodle_course_shortname = ?,
                        moodle_course_idnumber = ?,
                        auto_enroll = ?,
                        sync_enabled = ?,
                        updated_at = NOW(),
                        created_by = ?
                    WHERE id = ?
                ");
                $stmt->execute([
                    $moodle_course_id ?: null,
                    $moodle_course_shortname ?: null,
                    $moodle_course_idnumber ?: null,
                    $auto_enroll,
                    $sync_enabled,
                    $_SESSION['user_id'] ?? null,
                    $existing['id']
                ]);
                $success = "Mapping updated successfully!";
            } else {
                // Insert
                $stmt = $pdo->prepare("
                    INSERT INTO subject_moodle_mapping 
                    (academy_reference, subject_id, moodle_course_id, moodle_course_shortname, 
                     moodle_course_idnumber, auto_enroll, sync_enabled, created_by)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?)
                ");
                $stmt->execute([
                    $academy_reference,
                    $subject_id,
                    $moodle_course_id ?: null,
                    $moodle_course_shortname ?: null,
                    $moodle_course_idnumber ?: null,
                    $auto_enroll,
                    $sync_enabled,
                    $_SESSION['user_id'] ?? null
                ]);
                $success = "Mapping created successfully!";
            }
        } catch (PDOException $e) {
            $error = "Database error: " . $e->getMessage();
        }
    }
}

// Handle delete mapping
if (isset($_GET['delete']) && isset($_GET['confirm'])) {
    $mapping_id = (int)$_GET['delete'];
    try {
        $stmt = $pdo->prepare("
            DELETE FROM subject_moodle_mapping 
            WHERE id = ? AND academy_reference = ?
        ");
        $stmt->execute([$mapping_id, $academy_reference]);
        $success = "Mapping deleted successfully!";
    } catch (PDOException $e) {
        $error = "Error deleting mapping: " . $e->getMessage();
    }
}

// Get grades that have active students OR that admin has configured
// This shows only grades that are actually being used by the school
$stmt = $pdo->prepare("
    SELECT DISTINCT g.id, g.name 
    FROM grades g
    WHERE EXISTS (
        SELECT 1 FROM students s 
        WHERE s.grade_id = g.id 
        AND s.academy_reference = ?
    )
    ORDER BY g.name
");
$stmt->execute([$academy_reference]);
$grades = $stmt->fetchAll();

// If no students yet, show all grades
if (count($grades) === 0) {
    $stmt = $pdo->query("SELECT id, name FROM grades ORDER BY name");
    $grades = $stmt->fetchAll();
}

// Get subjects ONLY for grades that have students in this school
// Only include subject-grade combinations where that grade has students
$stmt = $pdo->prepare("
    SELECT DISTINCT s.id, s.name, s.code, 
           GROUP_CONCAT(DISTINCT g.name ORDER BY g.name SEPARATOR ', ') as grades,
           GROUP_CONCAT(DISTINCT g.id ORDER BY g.name SEPARATOR ',') as grade_ids
    FROM subjects s
    INNER JOIN grade_subject gs ON s.id = gs.subject_id
    INNER JOIN grades g ON gs.grade_id = g.id
    INNER JOIN students st ON st.grade_id = g.id
    WHERE st.academy_reference = ?
    GROUP BY s.id, s.name, s.code
    ORDER BY s.name
");
$stmt->execute([$academy_reference]);
$subjects = $stmt->fetchAll();

// If no students yet, show all subjects
if (count($subjects) === 0) {
    $stmt = $pdo->query("
        SELECT DISTINCT s.id, s.name, s.code, 
               GROUP_CONCAT(DISTINCT g.name ORDER BY g.name SEPARATOR ', ') as grades,
               GROUP_CONCAT(DISTINCT g.id ORDER BY g.name SEPARATOR ',') as grade_ids
        FROM subjects s
        INNER JOIN grade_subject gs ON s.id = gs.subject_id
        INNER JOIN grades g ON gs.grade_id = g.id
        GROUP BY s.id, s.name, s.code
        ORDER BY s.name
    ");
    $subjects = $stmt->fetchAll();
}

// Get existing mappings
$stmt = $pdo->prepare("
    SELECT smm.*, s.name as subject_name, s.code as subject_code
    FROM subject_moodle_mapping smm
    JOIN subjects s ON smm.subject_id = s.id
    WHERE smm.academy_reference = ?
    ORDER BY s.name
");
$stmt->execute([$academy_reference]);
$mappings = $stmt->fetchAll();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Moodle Subject Mapping - Admin Panel</title>
    <?php echoFavicon($academy_reference); ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.min.css" rel="stylesheet">
    <style>
        .mapping-card {
            transition: all 0.3s;
        }
        .mapping-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 15px rgba(0,0,0,0.1);
        }
        .status-badge {
            font-size: 0.75rem;
        }
    </style>
</head>
<body class="bg-light">
    
    <!-- Header -->
    <?php require_once '../partials/header.php'; ?>
    
    <div class="container-fluid py-4">
        <div class="row mb-4">
            <div class="col-12">
                <h2><i class="bi bi-diagram-3 me-2"></i>Moodle Subject Mapping</h2>
                <p class="text-muted">Configure how internal subjects map to Moodle courses for automatic enrollment</p>
            </div>
        </div>
        
        <!-- Alerts -->
        <?php if ($success): ?>
        <div class="alert alert-success alert-dismissible fade show">
            <i class="bi bi-check-circle me-2"></i><?= htmlspecialchars($success) ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php endif; ?>
        
        <?php if ($error): ?>
        <div class="alert alert-danger alert-dismissible fade show">
            <i class="bi bi-exclamation-triangle me-2"></i><?= htmlspecialchars($error) ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php endif; ?>
        
        <?php if (!$moodle->isEnabled()): ?>
        <div class="alert alert-warning">
            <i class="bi bi-exclamation-circle me-2"></i>
            <strong>Moodle Not Configured:</strong> 
            Please configure Moodle API settings first.
            <a href="settings.php" class="alert-link">Configure Moodle API</a>
        </div>
        <?php endif; ?>
        
        <!-- Bulk Actions Panel -->
        <div class="row mb-4">
            <div class="col-md-6">
                <div class="card shadow-sm">
                    <div class="card-header bg-success text-white">
                        <h5 class="mb-0"><i class="bi bi-cloud-arrow-down me-2"></i>Fetch Courses from Moodle</h5>
                    </div>
                    <div class="card-body">
                        <p class="mb-3">Refresh the list of available courses from your Moodle site.</p>
                        <form method="POST" class="d-flex align-items-center gap-2">
                            <button type="submit" name="fetch_courses" class="btn btn-success">
                                <i class="bi bi-arrow-repeat me-2"></i>Fetch <?= count($moodle_courses) > 0 ? 'Latest' : '' ?> Courses
                            </button>
                            <?php if (count($moodle_courses) > 0): ?>
                            <span class="badge bg-info"><?= count($moodle_courses) ?> courses loaded</span>
                            <?php endif; ?>
                        </form>
                    </div>
                </div>
            </div>
            
            <div class="col-md-6">
                <div class="card shadow-sm">
                    <div class="card-header bg-warning text-dark">
                        <h5 class="mb-0"><i class="bi bi-lightning-charge me-2"></i>Bulk Map by Grade</h5>
                    </div>
                    <div class="card-body">
                        <form method="POST">
                            <div class="row g-2">
                                <div class="col-md-6">
                                    <select name="grade_id" class="form-select" required>
                                        <option value="">-- Select Grade --</option>
                                        <?php foreach ($grades as $grade): ?>
                                        <option value="<?= $grade['id'] ?>">
                                            <?= htmlspecialchars($grade['name']) ?>
                                        </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                <div class="col-md-6">
                                    <div class="form-check mt-2">
                                        <input class="form-check-input" type="checkbox" name="bulk_auto_enroll" 
                                               id="bulk_auto_enroll" checked>
                                        <label class="form-check-label" for="bulk_auto_enroll">
                                            Auto-enroll
                                        </label>
                                    </div>
                                </div>
                                <div class="col-12">
                                    <button type="submit" name="bulk_map_grade" class="btn btn-warning w-100">
                                        <i class="bi bi-magic me-2"></i>Auto-Map All Subjects in Grade
                                    </button>
                                    <small class="text-muted">
                                        System will match subjects to Moodle courses by name
                                    </small>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="row">
            <!-- Left: Create/Edit Mapping -->
            <div class="col-lg-5">
                <div class="card shadow-sm">
                    <div class="card-header bg-primary text-white">
                        <h5 class="mb-0"><i class="bi bi-plus-circle me-2"></i>Add/Update Mapping</h5>
                    </div>
                    <div class="card-body">
                        <?php if (count($subjects) === 0): ?>
                        <div class="alert alert-warning">
                            <i class="bi bi-exclamation-triangle"></i> 
                            <strong>No subjects available!</strong> 
                            All subjects are from disabled grades. Enable grades in your database to see subjects.
                        </div>
                        <?php endif; ?>
                        
                        <div class="alert alert-info alert-sm mb-3">
                            <small>
                                <i class="bi bi-info-circle"></i> 
                                Showing subjects from <strong><?= count($grades) ?> active grades</strong> 
                                (<?= count($subjects) ?> subjects available)
                                <br>Only grades with enrolled students are shown.
                            </small>
                        </div>
                        
                        <form method="POST" id="mappingForm">
                            <div class="mb-3">
                                <label class="form-label">Internal Subject *</label>
                                <select name="subject_id" class="form-select" required onchange="loadMappingData(this.value)">
                                    <option value="">-- Select Subject --</option>
                                    <?php foreach ($subjects as $subject): ?>
                                    <option value="<?= $subject['id'] ?>" 
                                            data-code="<?= htmlspecialchars($subject['code']) ?>">
                                        <?= htmlspecialchars($subject['name']) ?> 
                                        (<?= htmlspecialchars($subject['code']) ?>)
                                        - <?= htmlspecialchars($subject['grades']) ?>
                                    </option>
                                    <?php endforeach; ?>
                                </select>
                                <small class="text-muted">
                                    Showing subjects from grades with enrolled students only
                                </small>
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label">
                                    Moodle Course 
                                    <span class="badge bg-info"><?= count($moodle_courses) ?> available</span>
                                </label>
                                <input type="text" id="courseSearch" class="form-control mb-2" 
                                       placeholder="🔍 Search courses...">
                                <select name="moodle_course_id" id="moodleCourseSelect" 
                                        class="form-select" onchange="fillCourseDetails(this)" size="5">
                                    <option value="">-- Select Moodle Course --</option>
                                    <?php foreach ($moodle_courses as $course): ?>
                                    <option value="<?= $course['id'] ?>" 
                                            data-shortname="<?= htmlspecialchars($course['shortname']) ?>"
                                            data-idnumber="<?= htmlspecialchars($course['idnumber'] ?? '') ?>"
                                            data-fullname="<?= htmlspecialchars($course['fullname']) ?>">
                                        <?= htmlspecialchars($course['fullname']) ?> 
                                        (<?= htmlspecialchars($course['shortname']) ?>)
                                        <?php if (!empty($course['idnumber'])): ?>
                                        [ID: <?= htmlspecialchars($course['idnumber']) ?>]
                                        <?php endif; ?>
                                    </option>
                                    <?php endforeach; ?>
                                </select>
                                <small class="text-muted">Search and select, or enter details manually below</small>
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label">Course Shortname</label>
                                <input type="text" name="moodle_course_shortname" class="form-control" 
                                       placeholder="e.g., G7-MATH">
                                <small class="text-muted">Moodle course shortname (used for enrollment)</small>
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label">Course ID Number</label>
                                <input type="text" name="moodle_course_idnumber" class="form-control" 
                                       placeholder="e.g., G7-MAT-003">
                                <small class="text-muted">Moodle course ID number (optional)</small>
                            </div>
                            
                            <div class="mb-3">
                                <div class="form-check form-switch">
                                    <input class="form-check-input" type="checkbox" name="auto_enroll" 
                                           id="auto_enroll" checked>
                                    <label class="form-check-label" for="auto_enroll">
                                        Auto-enroll students when subject is assigned
                                    </label>
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <div class="form-check form-switch">
                                    <input class="form-check-input" type="checkbox" name="sync_enabled" 
                                           id="sync_enabled" checked>
                                    <label class="form-check-label" for="sync_enabled">
                                        Enable sync for this mapping
                                    </label>
                                </div>
                            </div>
                            
                            <div class="d-grid gap-2">
                                <button type="submit" name="save_mapping" class="btn btn-primary">
                                    <i class="bi bi-save me-2"></i>Save Mapping
                                </button>
                                <button type="reset" class="btn btn-secondary">
                                    <i class="bi bi-x-circle me-2"></i>Reset Form
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
                
                <!-- Quick Help -->
                <div class="card shadow-sm mt-3">
                    <div class="card-header bg-info text-white">
                        <h6 class="mb-0"><i class="bi bi-question-circle me-2"></i>How It Works</h6>
                    </div>
                    <div class="card-body">
                        <ol class="mb-0 small">
                            <li>Map each internal subject to its corresponding Moodle course</li>
                            <li>When admin assigns subject to student, system auto-enrolls to Moodle</li>
                            <li>When parent pays for subject, student auto-enrolls</li>
                            <li>Use Course Shortname or ID Number for enrollment</li>
                            <li>Enable auto-enroll to activate automatic enrollment</li>
                        </ol>
                    </div>
                </div>
            </div>
            
            <!-- Right: Existing Mappings -->
            <div class="col-lg-7">
                <div class="card shadow-sm">
                    <div class="card-header bg-secondary text-white">
                        <h5 class="mb-0"><i class="bi bi-list-ul me-2"></i>Current Mappings</h5>
                    </div>
                    <div class="card-body">
                        <?php if (empty($mappings)): ?>
                        <div class="text-center py-5 text-muted">
                            <i class="bi bi-inbox display-4 d-block mb-3"></i>
                            <p>No mappings configured yet.</p>
                            <p>Create your first mapping using the form on the left.</p>
                        </div>
                        <?php else: ?>
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>Subject</th>
                                        <th>Moodle Course</th>
                                        <th>Status</th>
                                        <th>Last Sync</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($mappings as $mapping): ?>
                                    <tr>
                                        <td>
                                            <strong><?= htmlspecialchars($mapping['subject_name']) ?></strong><br>
                                            <small class="text-muted"><?= htmlspecialchars($mapping['subject_code']) ?></small>
                                        </td>
                                        <td>
                                            <?php if ($mapping['moodle_course_shortname']): ?>
                                                <strong><?= htmlspecialchars($mapping['moodle_course_shortname']) ?></strong><br>
                                                <small class="text-muted">
                                                    ID: <?= htmlspecialchars($mapping['moodle_course_id'] ?: 'N/A') ?>
                                                </small>
                                            <?php else: ?>
                                                <span class="text-muted">Not configured</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if ($mapping['auto_enroll']): ?>
                                                <span class="badge bg-success status-badge">Auto-enroll ON</span>
                                            <?php else: ?>
                                                <span class="badge bg-secondary status-badge">Auto-enroll OFF</span>
                                            <?php endif; ?>
                                            <br>
                                            <?php if ($mapping['sync_enabled']): ?>
                                                <span class="badge bg-primary status-badge mt-1">Sync ON</span>
                                            <?php else: ?>
                                                <span class="badge bg-secondary status-badge mt-1">Sync OFF</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if ($mapping['last_sync_at']): ?>
                                                <small><?= date('Y-m-d H:i', strtotime($mapping['last_sync_at'])) ?></small><br>
                                                <span class="badge bg-<?= $mapping['sync_status'] === 'success' ? 'success' : 'danger' ?> status-badge">
                                                    <?= htmlspecialchars($mapping['sync_status']) ?>
                                                </span>
                                            <?php else: ?>
                                                <span class="text-muted">Never</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <button class="btn btn-sm btn-outline-primary" 
                                                    onclick="editMapping(<?= $mapping['id'] ?>, <?= $mapping['subject_id'] ?>, '<?= htmlspecialchars($mapping['moodle_course_shortname'], ENT_QUOTES) ?>', '<?= htmlspecialchars($mapping['moodle_course_idnumber'], ENT_QUOTES) ?>', <?= $mapping['auto_enroll'] ?>, <?= $mapping['sync_enabled'] ?>)">
                                                <i class="bi bi-pencil"></i>
                                            </button>
                                            <button class="btn btn-sm btn-outline-danger" 
                                                    onclick="deleteMapping(<?= $mapping['id'] ?>)">
                                                <i class="bi bi-trash"></i>
                                            </button>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
                
                <!-- Actions -->
                <div class="card shadow-sm mt-3">
                    <div class="card-body">
                        <h6 class="card-title">Bulk Actions</h6>
                        <div class="d-flex gap-2 flex-wrap">
                            <a href="test_mappings.php" class="btn btn-success">
                                <i class="bi bi-check2-circle me-2"></i>Test Mappings
                            </a>
                            <a href="sync_students.php" class="btn btn-primary">
                                <i class="bi bi-cloud-upload me-2"></i>Sync Students
                            </a>
                            <a href="test_connection.php" class="btn btn-info">
                                <i class="bi bi-plug me-2"></i>Test Connection
                            </a>
                            <a href="index.php" class="btn btn-secondary">
                                <i class="bi bi-house me-2"></i>Moodle Hub
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Course search functionality
        document.getElementById('courseSearch')?.addEventListener('input', function(e) {
            const searchTerm = e.target.value.toLowerCase();
            const select = document.getElementById('moodleCourseSelect');
            const options = select.querySelectorAll('option');
            
            options.forEach(option => {
                if (option.value === '') return; // Keep the first "Select" option
                
                const text = option.textContent.toLowerCase();
                const fullname = option.dataset.fullname?.toLowerCase() || '';
                const shortname = option.dataset.shortname?.toLowerCase() || '';
                
                if (text.includes(searchTerm) || fullname.includes(searchTerm) || shortname.includes(searchTerm)) {
                    option.style.display = '';
                } else {
                    option.style.display = 'none';
                }
            });
        });
        
        // Fill course details when selected from dropdown
        function fillCourseDetails(select) {
            const option = select.options[select.selectedIndex];
            if (option.value) {
                document.querySelector('[name="moodle_course_shortname"]').value = option.dataset.shortname || '';
                document.querySelector('[name="moodle_course_idnumber"]').value = option.dataset.idnumber || '';
            }
        }
        
        // Edit existing mapping
        function editMapping(id, subjectId, shortname, idnumber, autoEnroll, syncEnabled) {
            document.querySelector('[name="subject_id"]').value = subjectId;
            document.querySelector('[name="moodle_course_shortname"]').value = shortname || '';
            document.querySelector('[name="moodle_course_idnumber"]').value = idnumber || '';
            document.querySelector('[name="auto_enroll"]').checked = autoEnroll == 1;
            document.querySelector('[name="sync_enabled"]').checked = syncEnabled == 1;
            
            // Scroll to form
            document.getElementById('mappingForm').scrollIntoView({behavior: 'smooth'});
        }
        
        // Delete mapping with confirmation
        function deleteMapping(id) {
            if (confirm('Are you sure you want to delete this mapping? Students will no longer auto-enroll for this subject.')) {
                window.location.href = '?delete=' + id + '&confirm=1';
            }
        }
        
        // Load existing mapping data when subject selected
        function loadMappingData(subjectId) {
            // This could be enhanced with AJAX to fetch existing mapping
            console.log('Loading mapping for subject:', subjectId);
        }
    </script>
</body>
</html>

