<?php
require_once '../includes/functions.php';
require_once '../includes/school_context.php';
requireAdminLogin();
requireSchool();

// Get current school context
$currentSchool = getCurrentSchool();

$studentId = (int)($_GET['id'] ?? 0);
$student = null;
$errors = [];
$success = false;

if ($studentId > 0) {
    $student = getStudentById($studentId);
    
    // Verify student belongs to current school
    if ($student && $student['academy_reference'] !== $currentSchool && !isSuperAdmin()) {
        // Access denied - student belongs to different school
        header('Location: dashboard.php?error=access_denied');
        exit;
    }
}

if (!$student) {
    header('Location: dashboard.php');
    exit;
}

// Handle students with full_name but no first_name/last_name
if (!isset($student['first_name']) && isset($student['full_name'])) {
    $nameParts = explode(' ', $student['full_name'], 2);
    $student['first_name'] = $nameParts[0] ?? '';
    $student['last_name'] = $nameParts[1] ?? '';
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $firstName = trim($_POST['first_name'] ?? '');
    $lastName = trim($_POST['last_name'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $gradeId = (int)($_POST['grade_id'] ?? 0);
    $selectedSubjects = $_POST['subjects'] ?? [];
    
    if (empty($firstName)) $errors[] = 'First name is required.';
    if (empty($lastName)) $errors[] = 'Last name is required.';
    if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $errors[] = 'Valid email address is required.';
    }
    if ($gradeId <= 0) $errors[] = 'Please select a grade.';
    if (empty($selectedSubjects)) $errors[] = 'Please select at least one subject.';
    
    // Check if email already exists (excluding current student)
    if (empty($errors) && strcasecmp($email, (string)$student['email']) !== 0) {
        $pdo = getDB();
        $stmt = $pdo->prepare("SELECT id FROM students WHERE email = ? AND id != ? AND academy_reference = ?");
        $stmt->execute([$email, $studentId, $currentSchool]);
        if ($stmt->fetch()) {
            $errors[] = 'Email address is already registered.';
        }
    }
    
    if (empty($errors)) {
        $updateData = [
            'first_name' => $firstName,
            'last_name' => $lastName,
            'full_name' => $firstName . ' ' . $lastName,  // Update full_name as well
            'email' => $email,
            'grade_id' => $gradeId
        ];
        
        if (updateStudent($studentId, $updateData)) {
            // Update student subjects
            if (updateStudentSubjects($studentId, $selectedSubjects)) {
                $success = true;
                $student = getStudentById($studentId); // Refresh data
                
                // Re-parse name if needed after refresh
                if (!isset($student['first_name']) && isset($student['full_name'])) {
                    $nameParts = explode(' ', $student['full_name'], 2);
                    $student['first_name'] = $nameParts[0] ?? '';
                    $student['last_name'] = $nameParts[1] ?? '';
                }
            } else {
                $errors[] = 'Failed to update student subjects.';
            }
        } else {
            $errors[] = 'Failed to update student.';
        }
    }
}

$grades = getGrades();
$studentSubjects = getSubjectsForStudent($studentId);
$gradeSubjects = getSubjectsForGrade($student['grade_id']);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Edit Student - <?php echo APP_NAME; ?></title>
    
    <!-- Favicon -->
    <link rel="icon" type="image/svg+xml" href="../images/img-melanesys-logo.svg">
    <link rel="alternate icon" href="../images/img-melanesys-logo.svg">
    <link rel="apple-touch-icon" href="../images/img-melanesys-logo.svg">
    
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
    <style>
        :root{ --brand-primary:#6f42c1; --brand-accent:#20c997; --brand-soft:#f3f0ff; }
        body{background:linear-gradient(135deg,var(--brand-soft),#e8fff9);} 
        .card{
            border: 0; 
            box-shadow: 0 4px 20px rgba(0,0,0,0.08);
            border-radius: 1rem;
            overflow: hidden;
            transition: all 0.3s ease;
        }
        .card:hover{
            transform: translateY(-2px);
            box-shadow: 0 8px 30px rgba(0,0,0,0.12);
        }
        .card-header{
            background: linear-gradient(135deg, var(--brand-primary), var(--brand-accent));
            border: none;
            padding: 1.5rem;
        }
        .card-body{
            padding: 2rem;
        }
        .form-control, .form-select{
            border: 2px solid rgba(111,66,193,0.1);
            border-radius: 0.75rem;
            padding: 0.875rem 1rem;
            font-size: 0.9rem;
            transition: all 0.3s ease;
            box-shadow: 0 2px 8px rgba(0,0,0,0.04);
        }
        .form-control:focus, .form-select:focus{
            border-color: var(--brand-primary);
            box-shadow: 0 0 0 0.2rem rgba(111,66,193,0.15), 0 4px 20px rgba(111,66,193,0.15);
            transform: translateY(-1px);
        }
        .form-control[readonly]{
            background: linear-gradient(135deg, #f8f9fa, #e9ecef);
            border-color: rgba(111,66,193,0.2);
            color: #6c757d;
            cursor: not-allowed;
        }
        .form-control[readonly]:focus{
            border-color: rgba(111,66,193,0.2);
            box-shadow: 0 2px 8px rgba(0,0,0,0.04);
            transform: none;
        }
        .form-label{
            font-weight: 600;
            color: #2c3e50;
            margin-bottom: 0.5rem;
        }
        .form-text{
            color: #6c757d;
            font-size: 0.8rem;
            font-weight: 500;
        }
        .btn-primary{
            background: linear-gradient(135deg, var(--brand-primary), var(--brand-accent));
            border: 0;
            border-radius: 0.75rem;
            padding: 0.875rem 2rem;
            font-weight: 600;
            transition: all 0.3s ease;
            box-shadow: 0 4px 15px rgba(111,66,193,0.3);
        }
        .btn-primary:hover{
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(111,66,193,0.4);
            filter: brightness(1.05);
        }
        .btn-secondary{
            background: #6c757d;
            border: 0;
            border-radius: 0.75rem;
            padding: 0.875rem 2rem;
            font-weight: 600;
            transition: all 0.3s ease;
            box-shadow: 0 4px 15px rgba(108,117,125,0.3);
        }
        .btn-secondary:hover{
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(108,117,125,0.4);
            filter: brightness(1.05);
        }
        .alert{
            border: 0;
            border-radius: 0.75rem;
            padding: 1rem 1.5rem;
        }
        .alert-success{
            background: linear-gradient(135deg, rgba(32,201,151,0.1), rgba(32,201,151,0.05));
            border-left: 4px solid var(--brand-accent);
            color: #155724;
        }
        .alert-danger{
            background: linear-gradient(135deg, rgba(220,53,69,0.1), rgba(220,53,69,0.05));
            border-left: 4px solid #dc3545;
            color: #721c24;
        }
        .input-group-text{
            background: linear-gradient(135deg, #f8f9fa, #e9ecef);
            border: 2px solid rgba(111,66,193,0.1);
            border-right: none;
            border-radius: 0.75rem 0 0 0.75rem;
            color: #6c757d;
            font-weight: 500;
        }
        .input-group .form-control{
            border-left: none;
            border-radius: 0 0.75rem 0.75rem 0;
        }
        .input-group:focus-within .input-group-text{
            border-color: rgba(111,66,193,0.2);
            background: linear-gradient(135deg, rgba(111,66,193,0.05), rgba(111,66,193,0.1));
        }
        .subjects-container{
            background: linear-gradient(135deg, rgba(111,66,193,0.03), rgba(32,201,151,0.03));
            border: 1px solid rgba(111,66,193,0.1);
            border-radius: 0.75rem;
            padding: 1.5rem;
            margin-bottom: 0.5rem;
        }
        .form-check{
            margin-bottom: 0.75rem;
            padding-left: 2rem;
        }
        .form-check-input{
            width: 1.25rem;
            height: 1.25rem;
            border: 2px solid rgba(111,66,193,0.3);
            border-radius: 0.375rem;
            background-color: #ffffff;
            transition: all 0.3s ease;
        }
        .form-check-input:checked{
            background-color: var(--brand-primary);
            border-color: var(--brand-primary);
            box-shadow: 0 2px 8px rgba(111,66,193,0.3);
        }
        .form-check-input:focus{
            border-color: var(--brand-primary);
            box-shadow: 0 0 0 0.2rem rgba(111,66,193,0.15);
        }
        .form-check-label{
            font-weight: 500;
            color: #2c3e50;
            cursor: pointer;
            transition: color 0.3s ease;
        }
        .form-check:hover .form-check-label{
            color: var(--brand-primary);
        }
    </style>
</head>
<body class="bg-light">
    <nav class="navbar navbar-expand-lg navbar-dark bg-primary">
        <div class="container-fluid">
            <a class="navbar-brand" href="dashboard.php">
                <i class="bi bi-speedometer2 me-2"></i>Admin Dashboard
            </a>
            <div class="navbar-nav ms-auto">
                <a class="nav-link" href="dashboard.php">
                    <i class="bi bi-arrow-left me-1"></i>Back to Dashboard
                </a>
                <a class="nav-link" href="logout.php">
                    <i class="bi bi-box-arrow-right me-1"></i>Logout
                </a>
            </div>
        </div>
    </nav>

    <div class="container my-4">
        <div class="row">
            <div class="col-12">
                <h2 class="mb-4">Edit Student</h2>
            </div>
        </div>

        <div class="row justify-content-center">
            <div class="col-md-8">
                <div class="card shadow">
                    <div class="card-header bg-primary text-white">
                        <h4 class="card-title mb-0">
                            <i class="bi bi-person-gear me-2"></i>Edit Student Information
                        </h4>
                    </div>
                    <div class="card-body">
                        <?php if ($success): ?>
                            <div class="alert alert-success">
                                <i class="bi bi-check-circle me-1"></i>Student updated successfully!
                            </div>
                        <?php endif; ?>

                        <?php if (!empty($errors)): ?>
                            <div class="alert alert-danger">
                                <h6><i class="bi bi-exclamation-triangle me-1"></i>Please correct the following errors:</h6>
                                <ul class="mb-0">
                                    <?php foreach ($errors as $error): ?>
                                        <li><?php echo htmlspecialchars($error); ?></li>
                                    <?php endforeach; ?>
                                </ul>
                            </div>
                        <?php endif; ?>

                        <form method="POST" action="edit.php?id=<?php echo $studentId; ?>">
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label for="first_name" class="form-label">First Name *</label>
                                    <input type="text" class="form-control" id="first_name" name="first_name" 
                                           value="<?php echo htmlspecialchars($student['first_name'] ?? ''); ?>" required>
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label for="last_name" class="form-label">Last Name *</label>
                                    <input type="text" class="form-control" id="last_name" name="last_name" 
                                           value="<?php echo htmlspecialchars($student['last_name'] ?? ''); ?>" required>
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="email" class="form-label">Email Address *</label>
                                <input type="email" class="form-control" id="email" name="email" 
                                       value="<?php echo htmlspecialchars($student['email'] ?? ''); ?>" required>
                            </div>
                            
                            <div class="mb-3">
                                <label for="grade_id" class="form-label">Grade *</label>
                                <select class="form-select" id="grade_id" name="grade_id" required>
                                    <option value="">Select grade</option>
                                    <?php foreach ($grades as $grade): ?>
                                        <option value="<?php echo $grade['id']; ?>" 
                                                <?php echo ($student['grade_id'] == $grade['id']) ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($grade['name']); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label">
                                    <i class="bi bi-book me-1"></i>Subjects *
                                </label>
                                <div class="subjects-container">
                                    <?php if (!empty($gradeSubjects)): ?>
                                        <?php foreach ($gradeSubjects as $subject): ?>
                                            <div class="form-check form-check-inline">
                                                <input class="form-check-input" type="checkbox" 
                                                       id="subject_<?php echo $subject['id']; ?>" 
                                                       name="subjects[]" 
                                                       value="<?php echo $subject['id']; ?>"
                                                       <?php echo in_array($subject['id'], array_column($studentSubjects, 'id')) ? 'checked' : ''; ?>>
                                                <label class="form-check-label" for="subject_<?php echo $subject['id']; ?>">
                                                    <?php echo htmlspecialchars($subject['name']); ?>
                                                </label>
                                            </div>
                                        <?php endforeach; ?>
                                    <?php else: ?>
                                        <p class="text-muted">No subjects available for this grade.</p>
                                    <?php endif; ?>
                                </div>
                                <div class="form-text">
                                    <i class="bi bi-info-circle me-1"></i>Select all subjects the student will be taking.
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label">
                                    <i class="bi bi-person-badge me-1"></i>Username
                                </label>
                                <div class="input-group">
                                    <span class="input-group-text bg-light border-end-0">
                                        <i class="bi bi-lock text-muted"></i>
                                    </span>
                                    <input type="text" class="form-control border-start-0" value="<?php echo htmlspecialchars($student['username'] ?? ''); ?>" readonly>
                                </div>
                                <div class="form-text">
                                    <i class="bi bi-info-circle me-1"></i>Username cannot be changed after registration for security reasons.
                                </div>
                            </div>
                            
                            <div class="d-flex gap-2">
                                <button type="submit" class="btn btn-primary">
                                    <i class="bi bi-check-lg me-1"></i>Update Student
                                </button>
                                <a href="dashboard.php" class="btn btn-secondary">
                                    <i class="bi bi-x-lg me-1"></i>Cancel
                                </a>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const gradeSelect = document.getElementById('grade_id');
            const subjectsContainer = document.querySelector('.subjects-container');
            
            gradeSelect.addEventListener('change', function() {
                const gradeId = this.value;
                
                if (gradeId) {
                    // Show loading state
                    subjectsContainer.innerHTML = '<div class="text-center"><i class="bi bi-hourglass-split me-2"></i>Loading subjects...</div>';
                    
                    // Fetch subjects for the selected grade
                    fetch(`../register.php?action=subjects&grade_id=${gradeId}`)
                        .then(response => response.json())
                        .then(data => {
                            if (data.success && data.subjects.length > 0) {
                                let subjectsHtml = '';
                                data.subjects.forEach(subject => {
                                    subjectsHtml += `
                                        <div class="form-check form-check-inline">
                                            <input class="form-check-input" type="checkbox" 
                                                   id="subject_${subject.id}" 
                                                   name="subjects[]" 
                                                   value="${subject.id}">
                                            <label class="form-check-label" for="subject_${subject.id}">
                                                ${subject.name}
                                            </label>
                                        </div>
                                    `;
                                });
                                subjectsContainer.innerHTML = subjectsHtml;
                            } else {
                                subjectsContainer.innerHTML = '<p class="text-muted">No subjects available for this grade.</p>';
                            }
                        })
                        .catch(error => {
                            console.error('Error loading subjects:', error);
                            subjectsContainer.innerHTML = '<p class="text-danger">Error loading subjects. Please try again.</p>';
                        });
                } else {
                    subjectsContainer.innerHTML = '<p class="text-muted">Please select a grade first.</p>';
                }
            });
        });
    </script>
</body>
</html>

