<?php
require_once '../includes/functions.php';
require_once '../includes/school_context.php';
requireAdminLogin();
requireSchool();

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['ok' => false, 'message' => 'Method not allowed']);
    exit;
}

$currentSchool = getCurrentSchool();
$errors = [];

try {
    // Get form data
    $firstName = trim($_POST['first_name'] ?? '');
    $lastName = trim($_POST['last_name'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $phone = trim($_POST['phone'] ?? '');
    $gradeId = (int)($_POST['grade_id'] ?? 0);
    $password = (string)($_POST['password'] ?? '');
    $confirm = (string)($_POST['password_confirm'] ?? '');
    $subjectIds = array_filter(array_map('intval', (array)($_POST['subject_ids'] ?? [])));

    // Validation
    if ($firstName === '') $errors[] = 'First name is required.';
    if ($lastName === '') $errors[] = 'Last name is required.';
    if ($email === '' || !filter_var($email, FILTER_VALIDATE_EMAIL)) $errors[] = 'Valid email is required.';
    if ($phone === '') $errors[] = 'Phone number is required.';
    if ($phone !== '' && !preg_match('/^[0-9]{10,15}$/', $phone)) $errors[] = 'Please enter a valid phone number (10-15 digits).';
    if ($gradeId <= 0) $errors[] = 'Please select a grade.';
    if ($password === '' || strlen($password) < 8) $errors[] = 'Password must be at least 8 characters.';
    if ($password !== $confirm) $errors[] = 'Passwords do not match.';
    
    // Debug logging
    if (defined('APP_DEBUG') && APP_DEBUG) {
        error_log('Dashboard create student - First: ' . $firstName . ', Last: ' . $lastName . ', Email: ' . $email . ', Grade: ' . $gradeId . ', Password length: ' . strlen($password));
    }
    
    // Unique email check (per tenant)
    if (empty($errors)) {
        $pdo = getDB();
        $stmt = $pdo->prepare('SELECT id FROM students WHERE email = ? AND academy_reference = ?');
        $stmt->execute([$email, $currentSchool]);
        if ($stmt->fetch()) {
            $errors[] = 'Email address is already registered.';
        }
    }
    
    if (empty($errors)) {
        // Create student
        $studentId = createStudent([
            'first_name' => $firstName,
            'last_name' => $lastName,
            'email' => $email,
            'phone' => $phone,
            'grade_id' => $gradeId,
            'password_plain' => $password,
            'subject_ids' => $subjectIds,
            'academy_reference' => $currentSchool
        ]);
        
        echo json_encode([
            'ok' => true, 
            'message' => 'Student created successfully',
            'student_id' => $studentId
        ]);
    } else {
        echo json_encode([
            'ok' => false, 
            'errors' => $errors
        ]);
    }
    
} catch (Throwable $e) {
    error_log('Dashboard create student error: ' . $e->getMessage() . ' - Trace: ' . $e->getTraceAsString());
    echo json_encode([
        'ok' => false, 
        'message' => 'Error creating student: ' . $e->getMessage()
    ]);
}
?>
