<?php
require_once '../includes/functions.php';
require_once '../config.php';
requireAdminLogin();

header('Content-Type: application/json');

$action = $_GET['action'] ?? 'status';

switch ($action) {
    case 'status':
        getBotStatus();
        break;
    case 'logs':
        getBotLogs();
        break;
    case 'trigger':
        triggerBot();
        break;
    case 'metrics':
        getBotMetrics();
        break;
    default:
        http_response_code(400);
        echo json_encode(['error' => 'Invalid action']);
}

function getBotStatus() {
    $response = [
        'status' => 'unknown',
        'last_run' => null,
        'next_run' => null,
        'lock_file' => false,
        'log_size' => 0,
        'error_count' => 0,
        'bot_type' => 'unknown',
        'health_status' => 'unknown',
        'performance_metrics' => []
    ];
    
    // Check both PHP and Python bots, prioritize Python bot (enhanced version)
    $phpLockFile = '../bot/temp/sync.lock';
    $pythonLockFile = '../bot/temp/python_sync.lock';
    
    // Check Python bot first (enhanced version)
    if (file_exists($pythonLockFile)) {
        $response['lock_file'] = true;
        $response['status'] = 'running';
        $response['started_at'] = date('Y-m-d H:i:s', filemtime($pythonLockFile));
        $response['bot_type'] = 'python_enhanced';
        $response['health_status'] = getPythonBotHealth();
        $response['performance_metrics'] = getPythonBotMetrics();
    } 
    // Fallback to PHP bot
    elseif (file_exists($phpLockFile)) {
        $response['lock_file'] = true;
        $response['status'] = 'running';
        $response['started_at'] = date('Y-m-d H:i:s', filemtime($phpLockFile));
        $response['bot_type'] = 'php';
        $response['health_status'] = 'legacy';
    } 
    // No bot running
    else {
        $response['status'] = 'idle';
        $response['bot_type'] = 'python_enhanced'; // Default to enhanced Python bot
        $response['health_status'] = getPythonBotHealth();
    }
    
    // Get last run time from logs (check both Python and PHP bot logs)
    $pythonLogFile = '../bot/logs/python_sync.log';
    $phpLogFile = '../bot/logs/sync.log';
    
    // Prioritize Python bot logs
    $logFile = file_exists($pythonLogFile) ? $pythonLogFile : $phpLogFile;
    
    if (file_exists($logFile)) {
        $response['log_size'] = filesize($logFile);
        
        // Read last few lines to find last completion
        $lines = file($logFile, FILE_IGNORE_NEW_LINES);
        $lines = array_reverse($lines);
        
        foreach ($lines as $line) {
            // Check for Python bot completion messages (enhanced format)
            if (strpos($line, '🎉 Moodle sync completed successfully!') !== false || 
                strpos($line, '=== COMPREHENSIVE SYNC METRICS ===') !== false ||
                strpos($line, '=== Moodle Sync Bot Completed ===') !== false) {
                
                // Extract timestamp (Python bot format: [2025-10-17 14:40:54,184])
                preg_match('/\[(\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2})/', $line, $matches);
                if ($matches) {
                    $response['last_run'] = $matches[1];
                }
                break;
            }
        }
        
        // Count recent errors
        $errorCount = 0;
        $recentLines = array_slice($lines, 0, 50); // Check last 50 lines
        foreach ($recentLines as $line) {
            if (strpos($line, '[ERROR]') !== false || strpos($line, '❌') !== false) {
                $errorCount++;
            }
        }
        $response['error_count'] = $errorCount;
    }
    
    // Estimate next run (assuming cron runs every 15 minutes)
    if ($response['last_run']) {
        $lastRun = strtotime($response['last_run']);
        $nextRun = $lastRun + (15 * 60); // 15 minutes
        $response['next_run'] = date('Y-m-d H:i:s', $nextRun);
    }
    
    echo json_encode($response);
}


function triggerBot() {
    // Check if any bot is already running
    $phpLockFile = '../bot/temp/sync.lock';
    $pythonLockFile = '../bot/temp/python_sync.lock';
    
    if (file_exists($phpLockFile) || file_exists($pythonLockFile)) {
        echo json_encode(['success' => false, 'message' => 'Bot is already running']);
        return;
    }
    
    // Prioritize Python bot (enhanced version)
    $pythonBotScript = '../bot/moodle_sync_bot.py';
    $phpBotScript = '../bot/moodle_sync_bot.php';
    
    $botScript = null;
    $botType = null;
    
    if (file_exists($pythonBotScript)) {
        $botScript = $pythonBotScript;
        $botType = 'Python Enhanced';
    } elseif (file_exists($phpBotScript)) {
        $botScript = $phpBotScript;
        $botType = 'PHP';
    } else {
        echo json_encode(['success' => false, 'message' => 'No bot script found']);
        return;
    }
    
    // Execute bot in background
    if (PHP_OS_FAMILY === 'Windows') {
        // Windows
        if ($botType === 'Python Enhanced') {
            $command = "start /B python \"$botScript\" > nul 2>&1";
        } else {
            $command = "start /B php \"$botScript\" > nul 2>&1";
        }
        pclose(popen($command, 'r'));
    } else {
        // Linux/Unix
        if ($botType === 'Python Enhanced') {
            $command = "python3 \"$botScript\" > /dev/null 2>&1 &";
        } else {
            $command = "php \"$botScript\" > /dev/null 2>&1 &";
        }
        exec($command);
    }
    
    echo json_encode(['success' => true, 'message' => "$botType bot triggered successfully"]);
}

function getBotMetrics() {
    $metrics = [
        'total_syncs' => 0,
        'successful_syncs' => 0,
        'failed_syncs' => 0,
        'total_users_uploaded' => 0,
        'total_enrollments' => 0,
        'average_sync_time' => 0,
        'last_7_days' => []
    ];
    
    // Use PHP bot logs
    $phpLogFile = '../bot/logs/sync.log';
    $logFile = $phpLogFile;
    
    if (!file_exists($logFile)) {
        echo json_encode($metrics);
        return;
    }
    
    $lines = file($logFile, FILE_IGNORE_NEW_LINES);
    $syncTimes = [];
    $usersUploaded = 0;
    $enrollments = 0;
    
    foreach ($lines as $line) {
        // Count sync attempts (both PHP and Python bot messages)
        if (strpos($line, '=== Moodle Sync Bot Started ===') !== false) {
            $metrics['total_syncs']++;
        }
        
        // Count successful syncs (both PHP and Python bot messages)
        if (strpos($line, '=== Moodle Sync Bot Completed ===') !== false || 
            strpos($line, '=== Moodle Sync Bot Completed Successfully ===') !== false) {
            $metrics['successful_syncs']++;
        }
        
        // Count failed syncs (both PHP and Python bot messages)
        if (strpos($line, '=== Moodle Sync Bot Failed ===') !== false) {
            $metrics['failed_syncs']++;
        }
        
        // Extract sync time
        if (strpos($line, 'Sync completed in') !== false) {
            preg_match('/Sync completed in ([\d.]+) seconds/', $line, $matches);
            if ($matches) {
                $syncTimes[] = floatval($matches[1]);
            }
        }
        
        // Count users uploaded - get LATEST count, not accumulate
        if (strpos($line, 'Successfully uploaded') !== false && strpos($line, 'new users') !== false) {
            preg_match('/Successfully uploaded (\d+) new users/', $line, $matches);
            if ($matches) {
                $usersUploaded = intval($matches[1]); // Get latest, don't accumulate
            }
        }
        
        // Count enrollments - get LATEST count from enrollment summary
        if (strpos($line, 'Enrollment completed:') !== false) {
            preg_match('/Enrollment completed: (\d+) enrollments/', $line, $matches);
            if ($matches) {
                $enrollments = intval($matches[1]); // Get latest, don't accumulate
            }
        }
    }
    
    $metrics['total_users_uploaded'] = $usersUploaded;
    $metrics['total_enrollments'] = $enrollments;
    
    if (!empty($syncTimes)) {
        $metrics['average_sync_time'] = round(array_sum($syncTimes) / count($syncTimes), 2);
    }
    
    echo json_encode($metrics);
}

function extractTimestamp($line) {
    // Handle PHP bot format: [2025-10-16 11:52:53]
    preg_match('/\[(\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2})\]/', $line, $matches);
    return $matches[1] ?? null;
}

function extractLogLevel($line) {
    if (strpos($line, '[ERROR]') !== false) return 'error';
    if (strpos($line, '[WARNING]') !== false) return 'warning';
    if (strpos($line, '[INFO]') !== false) return 'info';
    return 'info';
}

function extractMessage($line) {
    // Remove timestamp and log level (both PHP and Python bot formats)
    $message = preg_replace('/\[\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2}(,\d+)?\] \[(ERROR|WARNING|INFO)\]/', '', $line);
    return trim($message);
}

function getPythonBotHealth() {
    // Check Python bot health status
    $healthFile = '../bot/temp/health_status.json';
    
    if (file_exists($healthFile)) {
        $healthData = json_decode(file_get_contents($healthFile), true);
        if ($healthData && isset($healthData['overall_health'])) {
            return $healthData['overall_health'];
        }
    }
    
    // Default health check by testing if Python bot can run
    $pythonScript = '../bot/moodle_sync_bot.py';
    if (file_exists($pythonScript)) {
        // Try to run a simple health check
        $command = "python -c \"import sys; sys.path.insert(0, '../bot'); from moodle_sync_bot import MoodleSyncBot; bot = MoodleSyncBot(); print('HEALTHY')\" 2>/dev/null";
        $output = shell_exec($command);
        if (strpos($output, 'HEALTHY') !== false) {
            return 'HEALTHY';
        }
    }
    
    return 'UNKNOWN';
}

function getPythonBotMetrics() {
    // Get Python bot performance metrics
    $metricsFile = '../bot/temp/metrics.json';
    
    if (file_exists($metricsFile)) {
        $metricsData = json_decode(file_get_contents($metricsFile), true);
        if ($metricsData) {
            return [
                'users_processed' => $metricsData['users_processed'] ?? 0,
                'users_created' => $metricsData['users_created'] ?? 0,
                'users_skipped' => $metricsData['users_skipped'] ?? 0,
                'enrollments_successful' => $metricsData['enrollments_successful'] ?? 0,
                'enrollments_failed' => $metricsData['enrollments_failed'] ?? 0,
                'api_calls_made' => $metricsData['api_calls_made'] ?? 0,
                'total_sync_time' => $metricsData['total_sync_time'] ?? 0,
                'performance_breakdown' => $metricsData['performance_breakdown'] ?? []
            ];
        }
    }
    
    return [];
}

function getBotLogs() {
    // Check both Python and PHP bot logs, prioritize Python
    $pythonLogFile = '../bot/logs/python_sync.log';
    $phpLogFile = '../bot/logs/sync.log';
    
    $logFile = file_exists($pythonLogFile) ? $pythonLogFile : $phpLogFile;
    $lines = 50; // Number of lines to return
    
    if (!file_exists($logFile)) {
        echo json_encode(['logs' => [], 'message' => 'No log file found']);
        return;
    }
    
    $logLines = file($logFile, FILE_IGNORE_NEW_LINES);
    $logLines = array_reverse($logLines); // Most recent first
    $logLines = array_slice($logLines, 0, $lines);
    
    $logs = [];
    foreach ($logLines as $line) {
        $logs[] = [
            'line' => $line,
            'timestamp' => extractTimestamp($line),
            'level' => extractLogLevel($line),
            'message' => extractMessage($line)
        ];
    }
    
    echo json_encode(['logs' => $logs]);
}
?>
