<?php
require_once '../includes/functions.php';

// Check admin authentication
if (!isAdminLoggedIn()) {
    header('Location: login.php');
    exit;
}

// Handle backup actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!validateCSRFToken($_POST['csrf_token'] ?? '')) {
        $error = 'Invalid security token';
    } else {
        $action = $_POST['action'] ?? '';
        
        switch ($action) {
            case 'create_backup':
                require_once '../cron/backup_system.php';
                $backup = new BackupSystem();
                $result = $backup->createFullBackup();
                
                if ($result['success']) {
                    $success = "Backup created successfully: " . $result['backup_name'] . " (" . $result['size'] . ")";
                } else {
                    $error = "Backup failed: " . $result['error'];
                }
                break;
                
            case 'restore_backup':
                $backup_file = $_POST['backup_file'] ?? '';
                if (empty($backup_file)) {
                    $error = 'No backup file selected';
                } else {
                    require_once '../cron/backup_system.php';
                    $backup = new BackupSystem();
                    $result = $backup->restoreFromBackup($backup_file);
                    
                    if ($result['success']) {
                        $success = "Backup restored successfully";
                    } else {
                        $error = "Restore failed: " . $result['error'];
                    }
                }
                break;
                
            case 'delete_backup':
                $backup_file = $_POST['backup_file'] ?? '';
                if (file_exists($backup_file)) {
                    if (unlink($backup_file)) {
                        $success = "Backup deleted successfully";
                    } else {
                        $error = "Failed to delete backup";
                    }
                } else {
                    $error = "Backup file not found";
                }
                break;
        }
    }
}

// Get backup information
$backup_dir = __DIR__ . '/../backups';
$backups = [];
$backup_logs = [];

if (is_dir($backup_dir)) {
    $backup_files = glob($backup_dir . '/skolo_kine_*_backup_*.tar.gz');
    
    foreach ($backup_files as $file) {
        $backups[] = [
            'file' => $file,
            'name' => basename($file),
            'size' => formatBytes(filesize($file)),
            'created' => date('Y-m-d H:i:s', filemtime($file))
        ];
    }
    
    // Sort by creation time (newest first)
    usort($backups, function($a, $b) {
        return strtotime($b['created']) - strtotime($a['created']);
    });
}

// Get backup logs
try {
    $pdo = getDB();
    $stmt = $pdo->query("
        SELECT * FROM backup_logs 
        ORDER BY created_at DESC 
        LIMIT 50
    ");
    $backup_logs = $stmt->fetchAll();
} catch (Exception $e) {
    $error = "Failed to load backup logs: " . $e->getMessage();
}

function formatBytes($bytes, $precision = 2) {
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    
    for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
        $bytes /= 1024;
    }
    
    return round($bytes, $precision) . ' ' . $units[$i];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Backup Management - <?php echo APP_NAME; ?></title>    
    <!-- Favicon -->
    <link rel="icon" type="image/svg+xml" href="../images/img-melanesys-logo.svg">
    <link rel="alternate icon" href="../images/img-melanesys-logo.svg">
    <link rel="apple-touch-icon" href="../images/img-melanesys-logo.svg">
    
    
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
</head>
<body class="bg-light">
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-3 col-lg-2 d-md-block bg-dark sidebar collapse">
                <div class="position-sticky pt-3">
                    <ul class="nav flex-column">
                        <li class="nav-item">
                            <a class="nav-link text-white" href="dashboard.php">
                                <i class="bi bi-speedometer2 me-2"></i>Dashboard
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="create.php">
                                <i class="bi bi-person-plus me-2"></i>Add Student
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="payments.php">
                                <i class="bi bi-credit-card me-2"></i>Payments
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white active" href="backup_management.php">
                                <i class="bi bi-archive me-2"></i>Backups
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="logout.php">
                                <i class="bi bi-box-arrow-right me-2"></i>Logout
                            </a>
                        </li>
                    </ul>
                </div>
            </div>

            <!-- Main content -->
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">
                        <i class="bi bi-archive me-2"></i>Backup Management
                    </h1>
                </div>

                <?php if (isset($success)): ?>
                    <div class="alert alert-success alert-dismissible fade show">
                        <i class="bi bi-check-circle me-2"></i><?php echo htmlspecialchars($success); ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <?php if (isset($error)): ?>
                    <div class="alert alert-danger alert-dismissible fade show">
                        <i class="bi bi-exclamation-triangle me-2"></i><?php echo htmlspecialchars($error); ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <!-- Backup Actions -->
                <div class="row mb-4">
                    <div class="col-md-6">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="card-title mb-0">
                                    <i class="bi bi-plus-circle me-2"></i>Create New Backup
                                </h5>
                            </div>
                            <div class="card-body">
                                <p class="card-text">Create a full system backup including database and files.</p>
                                <form method="POST" onsubmit="return confirm('This will create a new backup. Continue?')">
                                    <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
                                    <input type="hidden" name="action" value="create_backup">
                                    <button type="submit" class="btn btn-primary">
                                        <i class="bi bi-archive me-2"></i>Create Backup
                                    </button>
                                </form>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-md-6">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="card-title mb-0">
                                    <i class="bi bi-info-circle me-2"></i>Backup Information
                                </h5>
                            </div>
                            <div class="card-body">
                                <p><strong>Total Backups:</strong> <?php echo count($backups); ?></p>
                                <p><strong>Backup Directory:</strong> <?php echo htmlspecialchars($backup_dir); ?></p>
                                <p><strong>Max Backups:</strong> 30 (auto-cleanup)</p>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Existing Backups -->
                <div class="card">
                    <div class="card-header">
                        <h5 class="card-title mb-0">
                            <i class="bi bi-list me-2"></i>Existing Backups
                        </h5>
                    </div>
                    <div class="card-body">
                        <?php if (empty($backups)): ?>
                            <div class="text-center py-4">
                                <i class="bi bi-archive fs-1 text-muted"></i>
                                <p class="text-muted mt-2">No backups found</p>
                            </div>
                        <?php else: ?>
                            <div class="table-responsive">
                                <table class="table table-striped">
                                    <thead>
                                        <tr>
                                            <th>Backup Name</th>
                                            <th>Size</th>
                                            <th>Created</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($backups as $backup): ?>
                                            <tr>
                                                <td>
                                                    <i class="bi bi-file-earmark-zip me-2"></i>
                                                    <?php echo htmlspecialchars($backup['name']); ?>
                                                </td>
                                                <td><?php echo $backup['size']; ?></td>
                                                <td><?php echo $backup['created']; ?></td>
                                                <td>
                                                    <div class="btn-group btn-group-sm">
                                                        <button class="btn btn-outline-primary" onclick="downloadBackup('<?php echo htmlspecialchars($backup['file']); ?>')">
                                                            <i class="bi bi-download"></i>
                                                        </button>
                                                        <button class="btn btn-outline-warning" onclick="restoreBackup('<?php echo htmlspecialchars($backup['file']); ?>')">
                                                            <i class="bi bi-arrow-clockwise"></i>
                                                        </button>
                                                        <button class="btn btn-outline-danger" onclick="deleteBackup('<?php echo htmlspecialchars($backup['file']); ?>')">
                                                            <i class="bi bi-trash"></i>
                                                        </button>
                                                    </div>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Backup Logs -->
                <div class="card mt-4">
                    <div class="card-header">
                        <h5 class="card-title mb-0">
                            <i class="bi bi-clock-history me-2"></i>Backup Logs
                        </h5>
                    </div>
                    <div class="card-body">
                        <?php if (empty($backup_logs)): ?>
                            <div class="text-center py-4">
                                <i class="bi bi-clock-history fs-1 text-muted"></i>
                                <p class="text-muted mt-2">No backup logs found</p>
                            </div>
                        <?php else: ?>
                            <div class="table-responsive">
                                <table class="table table-sm">
                                    <thead>
                                        <tr>
                                            <th>Backup Name</th>
                                            <th>Type</th>
                                            <th>Status</th>
                                            <th>Created</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($backup_logs as $log): ?>
                                            <tr>
                                                <td><?php echo htmlspecialchars($log['backup_name']); ?></td>
                                                <td>
                                                    <span class="badge bg-info"><?php echo $log['type']; ?></span>
                                                </td>
                                                <td>
                                                    <?php if ($log['status'] === 'SUCCESS'): ?>
                                                        <span class="badge bg-success"><?php echo $log['status']; ?></span>
                                                    <?php elseif ($log['status'] === 'FAILED'): ?>
                                                        <span class="badge bg-danger"><?php echo $log['status']; ?></span>
                                                    <?php else: ?>
                                                        <span class="badge bg-warning"><?php echo $log['status']; ?></span>
                                                    <?php endif; ?>
                                                </td>
                                                <td><?php echo $log['created_at']; ?></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <!-- Hidden forms for actions -->
    <form id="restoreForm" method="POST" style="display: none;">
        <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
        <input type="hidden" name="action" value="restore_backup">
        <input type="hidden" name="backup_file" id="restoreBackupFile">
    </form>

    <form id="deleteForm" method="POST" style="display: none;">
        <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
        <input type="hidden" name="action" value="delete_backup">
        <input type="hidden" name="backup_file" id="deleteBackupFile">
    </form>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function downloadBackup(file) {
            window.open('download_backup.php?file=' + encodeURIComponent(file), '_blank');
        }

        function restoreBackup(file) {
            if (confirm('WARNING: This will restore the system from backup. All current data will be replaced. Continue?')) {
                document.getElementById('restoreBackupFile').value = file;
                document.getElementById('restoreForm').submit();
            }
        }

        function deleteBackup(file) {
            if (confirm('Are you sure you want to delete this backup?')) {
                document.getElementById('deleteBackupFile').value = file;
                document.getElementById('deleteForm').submit();
            }
        }
    </script>
</body>
</html>
