<?php
/**
 * View Student Application Details
 * Admin can view full details and approve/reject
 */

require_once '../../includes/functions.php';
require_once '../../includes/school_context.php';
requireAdminLogin();

$pdo = getDB();
$academy_reference = getCurrentSchool();

if (!$academy_reference) {
    die("Error: No school selected.");
}

$application_id = (int)($_GET['id'] ?? 0);

// Get application details
$stmt = $pdo->prepare("
    SELECT 
        sa.*,
        g.name as grade_name
    FROM student_applications sa
    JOIN grades g ON sa.grade_id = g.id
    WHERE sa.id = ? AND sa.academy_reference = ?
");
$stmt->execute([$application_id, $academy_reference]);
$app = $stmt->fetch();

if (!$app) {
    die("Application not found.");
}

// Get selected subjects
$stmt = $pdo->prepare("
    SELECT s.name, s.code 
    FROM subjects s
    JOIN student_application_subjects sas ON s.id = sas.subject_id
    WHERE sas.application_id = ?
");
$stmt->execute([$application_id]);
$subjects = $stmt->fetchAll();

//Get approver info if approved
$approver_name = null;
if ($app['approved_by']) {
    $stmt = $pdo->prepare("SELECT username FROM admins WHERE id = ?");
    $stmt->execute([$app['approved_by']]);
    $approver_name = $stmt->fetchColumn();
}

$pageTitle = "Application #" . $app['id'] . " - " . $app['full_name'];
require_once '../partials/header.php';
?>

<div class="container-fluid py-4">
    <?php if (isset($_SESSION['success'])): ?>
    <div class="alert alert-success alert-dismissible fade show">
        <i class="bi bi-check-circle me-2"></i>
        <?= $_SESSION['success'] ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
    <?php unset($_SESSION['success']); endif; ?>
    
    <?php if (isset($_SESSION['error'])): ?>
    <div class="alert alert-danger alert-dismissible fade show">
        <i class="bi bi-exclamation-triangle me-2"></i>
        <?= $_SESSION['error'] ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
    <?php unset($_SESSION['error']); endif; ?>

    <div class="row mb-4">
        <div class="col-12">
            <div class="d-flex justify-content-between align-items-center">
                <h2><i class="bi bi-file-earmark-text me-2"></i>Application Details</h2>
                <a href="list.php" class="btn btn-outline-secondary">
                    <i class="bi bi-arrow-left me-1"></i>Back to List
                </a>
            </div>
        </div>
    </div>

    <div class="row">
        <!-- Main Application Info -->
        <div class="col-md-8">
            <div class="card shadow-sm mb-4">
                <div class="card-header bg-primary text-white">
                    <h5 class="mb-0"><i class="bi bi-person me-2"></i>Student Information</h5>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="text-muted small">Full Name</label>
                            <div class="fs-5"><strong><?= htmlspecialchars($app['full_name']) ?></strong></div>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="text-muted small">Email</label>
                            <div>
                                <?= htmlspecialchars($app['email']) ?>
                                <?php if ($app['email_verified']): ?>
                                    <span class="badge bg-success ms-2"><i class="bi bi-check-circle"></i> Verified</span>
                                <?php else: ?>
                                    <span class="badge bg-warning ms-2"><i class="bi bi-exclamation-circle"></i> Not Verified</span>
                                <?php endif; ?>
                            </div>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="text-muted small">Phone</label>
                            <div><?= htmlspecialchars($app['phone'] ?? 'N/A') ?></div>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="text-muted small">Date of Birth</label>
                            <div><?= $app['date_of_birth'] ? date('d M Y', strtotime($app['date_of_birth'])) : 'N/A' ?></div>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="text-muted small">Gender</label>
                            <div><?= ucfirst($app['gender'] ?? 'N/A') ?></div>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="text-muted small">Grade</label>
                            <div><span class="badge bg-info"><?= htmlspecialchars($app['grade_name']) ?></span></div>
                        </div>
                    </div>
                </div>
            </div>

            <div class="card shadow-sm mb-4">
                <div class="card-header bg-success text-white">
                    <h5 class="mb-0"><i class="bi bi-book me-2"></i>Selected Subjects</h5>
                </div>
                <div class="card-body">
                    <div class="row">
                        <?php foreach ($subjects as $subject): ?>
                        <div class="col-md-6 mb-2">
                            <div class="p-2 border rounded">
                                <i class="bi bi-check-circle text-success me-2"></i>
                                <strong><?= htmlspecialchars($subject['name']) ?></strong>
                                <small class="text-muted">(<?= htmlspecialchars($subject['code']) ?>)</small>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                    <div class="mt-3">
                        <strong>Total Subjects:</strong> <?= count($subjects) ?>
                    </div>
                </div>
            </div>

            <div class="card shadow-sm mb-4">
                <div class="card-header bg-info text-white">
                    <h5 class="mb-0"><i class="bi bi-people me-2"></i>Parent/Guardian Information</h5>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="text-muted small">Parent Name</label>
                            <div><?= htmlspecialchars($app['parent_name'] ?? 'N/A') ?></div>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="text-muted small">Parent Email</label>
                            <div><?= htmlspecialchars($app['parent_email'] ?? 'N/A') ?></div>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="text-muted small">Parent Phone</label>
                            <div><?= htmlspecialchars($app['parent_phone'] ?? 'N/A') ?></div>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="text-muted small">Relationship</label>
                            <div><?= ucfirst($app['parent_relationship'] ?? 'N/A') ?></div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Status & Actions Sidebar -->
        <div class="col-md-4">
            <div class="card shadow-sm mb-4">
                <div class="card-header bg-dark text-white">
                    <h5 class="mb-0"><i class="bi bi-info-circle me-2"></i>Application Status</h5>
                </div>
                <div class="card-body">
                    <div class="mb-3">
                        <label class="text-muted small">Application Status</label>
                        <div>
                            <?php
                            $status_badges = [
                                'submitted' => 'warning',
                                'verified' => 'info',
                                'approved' => 'primary',
                                'rejected' => 'danger',
                                'completed' => 'success'
                            ];
                            $badge_color = $status_badges[$app['application_status']] ?? 'secondary';
                            ?>
                            <span class="badge bg-<?= $badge_color ?> fs-6">
                                <?= ucfirst($app['application_status']) ?>
                            </span>
                        </div>
                    </div>

                    <div class="mb-3">
                        <label class="text-muted small">Payment Status</label>
                        <div>
                            <?php if ($app['payment_status'] === 'paid'): ?>
                                <span class="badge bg-success fs-6"><i class="bi bi-check-circle"></i> Paid</span>
                                <div class="mt-1">
                                    <small>Amount: E<?= number_format($app['payment_amount'], 2) ?></small>
                                </div>
                                <?php if ($app['payment_date']): ?>
                                <div class="mt-1">
                                    <small>Paid: <?= date('d M Y', strtotime($app['payment_date'])) ?></small>
                                </div>
                                <?php endif; ?>
                                <?php if ($app['payment_method'] === 'manual'): ?>
                                <div class="mt-1">
                                    <small class="text-info"><i class="bi bi-check2-circle"></i> Manually Verified</small>
                                </div>
                                <?php endif; ?>
                            <?php elseif ($app['payment_status'] === 'pending'): ?>
                                <span class="badge bg-info fs-6"><i class="bi bi-clock"></i> Pending Verification</span>
                                <div class="mt-1">
                                    <small>Amount: E<?= number_format($app['payment_amount'], 2) ?></small>
                                </div>
                                <?php if ($app['payment_reference']): ?>
                                <div class="mt-1">
                                    <small>Ref: <?= htmlspecialchars($app['payment_reference']) ?></small>
                                </div>
                                <?php endif; ?>
                            <?php elseif ($app['payment_status'] === 'failed'): ?>
                                <span class="badge bg-danger fs-6"><i class="bi bi-x-circle"></i> Failed</span>
                                <div class="mt-1">
                                    <small>Amount: E<?= number_format($app['payment_amount'], 2) ?></small>
                                </div>
                            <?php else: ?>
                                <span class="badge bg-warning fs-6">Unpaid</span>
                                <div class="mt-1">
                                    <small>Amount Due: E<?= number_format($app['payment_amount'], 2) ?></small>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>

                    <div class="mb-3">
                        <label class="text-muted small">Submitted</label>
                        <div><?= date('d M Y H:i', strtotime($app['submitted_at'])) ?></div>
                    </div>

                    <?php if ($app['email_verified_at']): ?>
                    <div class="mb-3">
                        <label class="text-muted small">Email Verified</label>
                        <div><?= date('d M Y H:i', strtotime($app['email_verified_at'])) ?></div>
                    </div>
                    <?php endif; ?>

                    <?php if ($app['approved_at']): ?>
                    <div class="mb-3">
                        <label class="text-muted small">Approved</label>
                        <div><?= date('d M Y H:i', strtotime($app['approved_at'])) ?></div>
                        <?php if ($approver_name): ?>
                        <small class="text-muted">By: <?= htmlspecialchars($approver_name) ?></small>
                        <?php endif; ?>
                    </div>
                    <?php endif; ?>

                    <?php if ($app['student_id']): ?>
                    <div class="mb-3">
                        <label class="text-muted small">Student Account</label>
                        <div>
                            <span class="badge bg-success"><i class="bi bi-check-circle"></i> Created</span>
                        </div>
                        <?php if ($app['username']): ?>
                        <div class="mt-2 p-2 bg-light rounded">
                            <small><strong>Username:</strong> <?= htmlspecialchars($app['username']) ?></small>
                        </div>
                        <?php endif; ?>
                    </div>
                    <?php endif; ?>

                    <?php if ($app['rejection_reason']): ?>
                    <div class="mb-3">
                        <label class="text-muted small">Rejection Reason</label>
                        <div class="alert alert-danger mb-0">
                            <?= htmlspecialchars($app['rejection_reason']) ?>
                        </div>
                    </div>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Payment Verification Card -->
            <?php if ($app['payment_status'] === 'pending' && !$app['student_id']): ?>
            <div class="card shadow-sm mb-4">
                <div class="card-header bg-info text-white">
                    <h5 class="mb-0"><i class="bi bi-credit-card me-2"></i>Verify Payment</h5>
                </div>
                <div class="card-body">
                    <p class="mb-3">Manual payment submitted. Verify before approving application.</p>
                    
                    <form method="POST" action="verify_payment.php" onsubmit="return confirm('Approve this payment?');">
                        <input type="hidden" name="application_id" value="<?= $app['id'] ?>">
                        <button type="submit" name="action" value="approve_payment" class="btn btn-success w-100 mb-2">
                            <i class="bi bi-check-circle me-2"></i>Approve Payment
                        </button>
                    </form>

                    <button type="button" class="btn btn-outline-danger w-100" data-bs-toggle="modal" data-bs-target="#rejectPaymentModal">
                        <i class="bi bi-x-circle me-2"></i>Reject Payment
                    </button>
                </div>
            </div>
            <?php endif; ?>
            
            <!-- Actions Card -->
            <?php if ($app['application_status'] !== 'completed' && $app['application_status'] !== 'rejected'): ?>
            <div class="card shadow-sm mb-4">
                <div class="card-header bg-warning">
                    <h5 class="mb-0"><i class="bi bi-gear me-2"></i>Actions</h5>
                </div>
                <div class="card-body">
                    <?php if (!$app['student_id']): ?>
                    
                    <form method="POST" action="approve.php" onsubmit="return confirm('Approve this application and create student account?');">
                        <input type="hidden" name="application_id" value="<?= $app['id'] ?>">
                        <button type="submit" name="action" value="approve" class="btn btn-success w-100 mb-3">
                            <i class="bi bi-check-circle me-2"></i>Approve & Create Account
                        </button>
                    </form>

                    <button type="button" class="btn btn-danger w-100" data-bs-toggle="modal" data-bs-target="#rejectModal">
                        <i class="bi bi-x-circle me-2"></i>Reject Application
                    </button>

                    <div class="mt-3 p-2 bg-light rounded">
                        <small class="text-muted">
                            <i class="bi bi-info-circle me-1"></i>
                            <?php if ($app['payment_status'] !== 'paid'): ?>
                            Approving will bypass payment requirement and create the account.
                            <?php else: ?>
                            This will create the student account and send credentials.
                            <?php endif; ?>
                        </small>
                    </div>

                    <?php else: ?>
                    <div class="alert alert-info mb-0">
                        <i class="bi bi-check-circle me-2"></i>
                        Student account already created
                    </div>
                    <?php endif; ?>
                </div>
            </div>
            <?php endif; ?>

            <?php if ($app['admin_notes']): ?>
            <div class="card shadow-sm">
                <div class="card-header">
                    <h6 class="mb-0"><i class="bi bi-sticky me-2"></i>Admin Notes</h6>
                </div>
                <div class="card-body">
                    <?= nl2br(htmlspecialchars($app['admin_notes'])) ?>
                </div>
            </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<!-- Reject Application Modal -->
<div class="modal fade" id="rejectModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST" action="approve.php">
                <input type="hidden" name="application_id" value="<?= $app['id'] ?>">
                <div class="modal-header bg-danger text-white">
                    <h5 class="modal-title"><i class="bi bi-x-circle me-2"></i>Reject Application</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="alert alert-warning">
                        <i class="bi bi-exclamation-triangle me-2"></i>
                        This will reject the application and notify the applicant.
                    </div>
                    <div class="mb-3">
                        <label for="rejection_reason" class="form-label">Reason for Rejection *</label>
                        <textarea class="form-control" id="rejection_reason" name="rejection_reason" 
                                  rows="4" required placeholder="Explain why this application is being rejected..."></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" name="action" value="reject" class="btn btn-danger">
                        <i class="bi bi-x-circle me-2"></i>Reject Application
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Reject Payment Modal -->
<div class="modal fade" id="rejectPaymentModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST" action="verify_payment.php">
                <input type="hidden" name="application_id" value="<?= $app['id'] ?>">
                <div class="modal-header bg-danger text-white">
                    <h5 class="modal-title"><i class="bi bi-x-circle me-2"></i>Reject Payment</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="alert alert-warning">
                        <i class="bi bi-exclamation-triangle me-2"></i>
                        This will reject the payment and notify the applicant.
                    </div>
                    <div class="mb-3">
                        <label for="payment_rejection_reason" class="form-label">Reason for Rejection *</label>
                        <textarea class="form-control" id="payment_rejection_reason" name="payment_rejection_reason" 
                                  rows="4" required placeholder="Explain why the payment is being rejected (e.g., invalid receipt, wrong amount, etc.)"></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" name="action" value="reject_payment" class="btn btn-danger">
                        <i class="bi bi-x-circle me-2"></i>Reject Payment
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<?php require_once '../partials/footer.php'; ?>

