<?php
/**
 * Student Applications List
 * View and manage self-registration applications
 */

require_once '../../includes/functions.php';
require_once '../../includes/school_context.php';
requireAdminLogin();

$pdo = getDB();
$academy_reference = getCurrentSchool();

if (!$academy_reference) {
    die("Error: No school selected.");
}

// Get filter
$status_filter = $_GET['status'] ?? 'all';

// Build query
$sql = "
    SELECT 
        sa.*,
        g.name as grade_name,
        COUNT(sas.subject_id) as subject_count
    FROM student_applications sa
    JOIN grades g ON sa.grade_id = g.id
    LEFT JOIN student_application_subjects sas ON sa.id = sas.application_id
    WHERE sa.academy_reference = ?
";

$params = [$academy_reference];

if ($status_filter !== 'all') {
    $sql .= " AND sa.application_status = ?";
    $params[] = $status_filter;
}

$sql .= " GROUP BY sa.id ORDER BY sa.submitted_at DESC";

$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$applications = $stmt->fetchAll();

// Get counts for filters
$stmt = $pdo->prepare("
    SELECT application_status, COUNT(*) as count
    FROM student_applications
    WHERE academy_reference = ?
    GROUP BY application_status
");
$stmt->execute([$academy_reference]);
$status_counts = [];
while ($row = $stmt->fetch()) {
    $status_counts[$row['application_status']] = $row['count'];
}

$total_applications = array_sum($status_counts);

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Student Applications - Admin</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.min.css" rel="stylesheet">
</head>
<body class="bg-light">
    
    <!-- Header -->
    <?php require_once '../partials/header.php'; ?>
    
    <div class="container-fluid py-4">
        <div class="row mb-4">
            <div class="col-12">
                <h2><i class="bi bi-file-earmark-text me-2"></i>Student Applications</h2>
                <p class="text-muted">Manage self-registration applications</p>
            </div>
        </div>
        
        <!-- Statistics -->
        <div class="row mb-4">
            <div class="col-md-3">
                <div class="card shadow-sm">
                    <div class="card-body text-center">
                        <h3 class="text-primary"><?= $total_applications ?></h3>
                        <p class="mb-0">Total Applications</p>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card shadow-sm">
                    <div class="card-body text-center">
                        <h3 class="text-warning"><?= $status_counts['submitted'] ?? 0 ?></h3>
                        <p class="mb-0">Pending Verification</p>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card shadow-sm">
                    <div class="card-body text-center">
                        <h3 class="text-info"><?= $status_counts['verified'] ?? 0 ?></h3>
                        <p class="mb-0">Verified (Awaiting Payment)</p>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card shadow-sm">
                    <div class="card-body text-center">
                        <h3 class="text-success"><?= $status_counts['completed'] ?? 0 ?></h3>
                        <p class="mb-0">Completed</p>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Filters -->
        <div class="card shadow-sm mb-4">
            <div class="card-body">
                <div class="btn-group" role="group">
                    <a href="?status=all" class="btn btn-<?= $status_filter === 'all' ? 'primary' : 'outline-primary' ?>">
                        All (<?= $total_applications ?>)
                    </a>
                    <a href="?status=submitted" class="btn btn-<?= $status_filter === 'submitted' ? 'warning' : 'outline-warning' ?>">
                        Pending (<?= $status_counts['submitted'] ?? 0 ?>)
                    </a>
                    <a href="?status=verified" class="btn btn-<?= $status_filter === 'verified' ? 'info' : 'outline-info' ?>">
                        Verified (<?= $status_counts['verified'] ?? 0 ?>)
                    </a>
                    <a href="?status=completed" class="btn btn-<?= $status_filter === 'completed' ? 'success' : 'outline-success' ?>">
                        Completed (<?= $status_counts['completed'] ?? 0 ?>)
                    </a>
                    <a href="?status=rejected" class="btn btn-<?= $status_filter === 'rejected' ? 'danger' : 'outline-danger' ?>">
                        Rejected (<?= $status_counts['rejected'] ?? 0 ?>)
                    </a>
                </div>
            </div>
        </div>
        
        <!-- Applications Table -->
        <div class="card shadow-sm">
            <div class="card-body">
                <?php if (empty($applications)): ?>
                <div class="text-center py-5 text-muted">
                    <i class="bi bi-inbox display-4 d-block mb-3"></i>
                    <p>No applications found</p>
                </div>
                <?php else: ?>
                <div class="table-responsive">
                    <table class="table table-hover">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Student Name</th>
                                <th>Email</th>
                                <th>Grade</th>
                                <th>Subjects</th>
                                <th>Status</th>
                                <th>Payment</th>
                                <th>Submitted</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($applications as $app): ?>
                            <tr>
                                <td>#<?= $app['id'] ?></td>
                                <td>
                                    <strong><?= htmlspecialchars($app['full_name']) ?></strong><br>
                                    <small class="text-muted">Parent: <?= htmlspecialchars($app['parent_name']) ?></small>
                                </td>
                                <td>
                                    <small>
                                        <?= htmlspecialchars($app['email']) ?><br>
                                        <?php if ($app['email_verified']): ?>
                                        <span class="badge bg-success"><i class="bi bi-check"></i> Verified</span>
                                        <?php else: ?>
                                        <span class="badge bg-warning">Pending</span>
                                        <?php endif; ?>
                                    </small>
                                </td>
                                <td><?= htmlspecialchars($app['grade_name']) ?></td>
                                <td>
                                    <span class="badge bg-info"><?= $app['subject_count'] ?> subjects</span>
                                </td>
                                <td>
                                    <?php
                                    $status_badges = [
                                        'submitted' => 'warning',
                                        'verified' => 'info',
                                        'approved' => 'primary',
                                        'rejected' => 'danger',
                                        'completed' => 'success'
                                    ];
                                    $badge_color = $status_badges[$app['application_status']] ?? 'secondary';
                                    ?>
                                    <span class="badge bg-<?= $badge_color ?>">
                                        <?= htmlspecialchars($app['application_status']) ?>
                                    </span>
                                </td>
                                <td>
                                    <?php if ($app['payment_status'] === 'paid'): ?>
                                    <span class="badge bg-success">
                                        <i class="bi bi-check-circle"></i> Paid
                                    </span><br>
                                    <small><?= $currency_symbol ?? 'E' ?><?= number_format($app['payment_amount'], 2) ?></small>
                                    <?php else: ?>
                                    <span class="badge bg-warning">Unpaid</span><br>
                                    <small><?= $currency_symbol ?? 'E' ?><?= number_format($app['payment_amount'], 2) ?></small>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <small><?= date('Y-m-d H:i', strtotime($app['submitted_at'])) ?></small>
                                </td>
                                <td>
                                    <div class="btn-group">
                                        <a href="view.php?id=<?= $app['id'] ?>" class="btn btn-sm btn-outline-primary" title="View Details">
                                            <i class="bi bi-eye"></i>
                                        </a>
                                        <?php if ($app['application_status'] !== 'completed' && $app['application_status'] !== 'rejected' && !$app['student_id']): ?>
                                        <a href="view.php?id=<?= $app['id'] ?>#actions" class="btn btn-sm btn-success" title="Approve">
                                            <i class="bi bi-check-circle"></i>
                                        </a>
                                        <?php endif; ?>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                <?php endif; ?>
            </div>
        </div>
        
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

