<?php
/**
 * Approve or Reject Student Application
 * Creates student account and sends credentials
 */

session_start();
require_once '../../includes/functions.php';
require_once '../../includes/school_context.php';
require_once '../../includes/student_account_creator.php';
requireAdminLogin();

$pdo = getDB();
$academy_reference = getCurrentSchool();

if (!$academy_reference) {
    die("Error: No school selected.");
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: list.php');
    exit;
}

$application_id = (int)($_POST['application_id'] ?? 0);
$action = $_POST['action'] ?? '';

// Verify application belongs to this school
$stmt = $pdo->prepare("
    SELECT * FROM student_applications 
    WHERE id = ? AND academy_reference = ?
");
$stmt->execute([$application_id, $academy_reference]);
$app = $stmt->fetch();

if (!$app) {
    die("Application not found or access denied.");
}

$admin_id = $_SESSION['admin_id'] ?? null;

if ($action === 'approve') {
    // Approve and create student account
    
    if ($app['student_id']) {
        $_SESSION['error'] = 'Student account already created for this application.';
        header('Location: view.php?id=' . $application_id);
        exit;
    }
    
    try {
        // Use StudentAccountCreator to create the account
        $accountCreator = new StudentAccountCreator($pdo);
        
        // Pass true to bypass payment check (manual approval), and admin ID
        $result = $accountCreator->createFromApplication($application_id, true, $admin_id);
        
        if ($result['success']) {
            // Update application status to approved
            $stmt = $pdo->prepare("
                UPDATE student_applications 
                SET application_status = 'approved'
                WHERE id = ?
            ");
            $stmt->execute([$application_id]);
            
            $_SESSION['success'] = "Application approved successfully! Student account created.<br>
                                   <strong>Username:</strong> {$result['username']}<br>
                                   <strong>Password:</strong> {$result['password']}<br>
                                   <small>Credentials have been emailed to the student.</small>";
            header('Location: view.php?id=' . $application_id);
        } else {
            $_SESSION['error'] = "Error creating account: " . $result['error'];
            header('Location: view.php?id=' . $application_id);
        }
    } catch (Exception $e) {
        $_SESSION['error'] = "Error: " . $e->getMessage();
        header('Location: view.php?id=' . $application_id);
    }
    
} elseif ($action === 'reject') {
    // Reject application
    
    $rejection_reason = trim($_POST['rejection_reason'] ?? '');
    
    if (empty($rejection_reason)) {
        $_SESSION['error'] = 'Rejection reason is required.';
        header('Location: view.php?id=' . $application_id);
        exit;
    }
    
    try {
        $pdo->beginTransaction();
        
        // Update application to rejected
        $stmt = $pdo->prepare("
            UPDATE student_applications 
            SET application_status = 'rejected',
                rejection_reason = ?,
                approved_by = ?,
                approved_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute([$rejection_reason, $admin_id, $application_id]);
        
        // Send rejection email
        $subject = "Application Status Update";
        $message = "
            <!DOCTYPE html>
            <html>
            <head>
                <style>
                    body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
                    .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                    .header { background: #dc3545; color: white; padding: 20px; text-align: center; border-radius: 10px 10px 0 0; }
                    .content { background: #ffffff; padding: 30px; border: 1px solid #e0e0e0; }
                    .reason-box { background: #f8d7da; border-left: 4px solid #dc3545; padding: 15px; margin: 20px 0; border-radius: 5px; }
                </style>
            </head>
            <body>
                <div class='container'>
                    <div class='header'>
                        <h2 style='margin: 0;'>Application Status Update</h2>
                    </div>
                    <div class='content'>
                        <p>Dear {$app['full_name']},</p>
                        <p>Thank you for your interest in our school. After careful review, we regret to inform you that your application has not been approved at this time.</p>
                        
                        <div class='reason-box'>
                            <strong>Reason:</strong><br>
                            {$rejection_reason}
                        </div>
                        
                        <p>If you have any questions or would like to discuss this decision, please contact the school administration.</p>
                        
                        <p>Best regards,<br><strong>School Administration</strong></p>
                    </div>
                </div>
            </body>
            </html>
        ";
        
        $headers = "MIME-Version: 1.0\r\n";
        $headers .= "Content-type:text/html;charset=UTF-8\r\n";
        $headers .= "From: School Administration <noreply@melanegroup.com>\r\n";
        
        mail($app['email'], $subject, $message, $headers);
        
        $pdo->commit();
        
        $_SESSION['success'] = 'Application rejected and applicant notified.';
        header('Location: list.php');
    } catch (Exception $e) {
        $pdo->rollBack();
        $_SESSION['error'] = 'Error rejecting application: ' . $e->getMessage();
        header('Location: view.php?id=' . $application_id);
    }
    
} else {
    $_SESSION['error'] = 'Invalid action.';
    header('Location: view.php?id=' . $application_id);
}

exit;

