# 🗄️ Tenant Files Table - Migration Guide

## 📋 **Overview**

Complete database migration to track all files uploaded by tenants in your multi-tenant education platform.

---

## 📦 **Files Created**

```
✅ database/migrations/create_tenant_files_table.sql
   - Complete table creation
   - Indexes and constraints
   - Related tables (access_log, versions)
   - Views and triggers
   - Sample data

✅ database/migrations/rollback_tenant_files_table.sql
   - Complete rollback script
   - Backup creation
   - Safe restoration

✅ database/migrations/tenant_files_sample_data.sql
   - Sample data for testing
   - Multiple tenants
   - All file categories

✅ database/migrations/tenant_files_verification_queries.sql
   - 50+ verification queries
   - Data integrity checks
   - Performance analysis
   - Usage statistics

✅ TENANT_FILES_TABLE_MIGRATION_GUIDE.md
   - This complete guide
```

---

## 🗃️ **Table Structure**

### **Main Table: tenant_files**

```sql
CREATE TABLE tenant_files (
    id INT AUTO_INCREMENT PRIMARY KEY,
    tenant_id CHAR(36) NOT NULL,
    file_category ENUM('document', 'photo', 'receipt', 'report', 'backup', 'other'),
    original_filename VARCHAR(255) NOT NULL,
    stored_filename VARCHAR(255) NOT NULL UNIQUE,
    file_path VARCHAR(500) NOT NULL,
    file_size BIGINT NOT NULL,
    mime_type VARCHAR(100),
    file_hash VARCHAR(64),
    uploaded_by INT,
    related_id INT,
    related_type VARCHAR(50),
    description TEXT,
    tags VARCHAR(255),
    is_public TINYINT(1) DEFAULT 0,
    access_count INT DEFAULT 0,
    last_accessed_at DATETIME,
    upload_date DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    is_deleted TINYINT(1) DEFAULT 0,
    deleted_at DATETIME,
    deleted_by INT
);
```

### **Related Tables**

```
✅ tenant_file_access_log     - Tracks file access activity
✅ tenant_file_versions        - Version control for files
```

### **Views**

```
✅ v_active_tenant_files       - Non-deleted files only
✅ v_tenant_file_stats         - Statistics by tenant
```

---

## 🚀 **Migration Steps**

### **Step 1: Backup Current Database (CRITICAL)**

```bash
# Via command line
mysqldump -u username -p database_name > backup_before_migration.sql

# Or via phpMyAdmin
# Export tab → Quick export → Go
```

---

### **Step 2: Import Main Migration**

**Via phpMyAdmin:**
1. Go to **Import** tab
2. Choose file: `database/migrations/create_tenant_files_table.sql`
3. Format: SQL
4. Click **Go**

**Via Command Line:**
```bash
mysql -u username -p database_name < database/migrations/create_tenant_files_table.sql
```

**Expected Output:**
```
✅ Table created successfully
✅ Indexes created (8 indexes)
✅ Related tables created (2 tables)
✅ Views created (2 views)
✅ Trigger created (1 trigger)
✅ Sample data inserted (15+ records)
```

---

### **Step 3: Verify Migration**

**Run verification queries:**

```sql
-- Check table exists
DESCRIBE tenant_files;

-- Check data
SELECT COUNT(*) FROM tenant_files;

-- Check indexes
SHOW INDEXES FROM tenant_files;

-- Check views
SHOW FULL TABLES WHERE Table_type = 'VIEW';
```

---

### **Step 4: (Optional) Insert Sample Data**

If you want more test data:

```bash
# Import sample data
mysql -u username -p database_name < database/migrations/tenant_files_sample_data.sql
```

---

### **Step 5: Run Verification Queries**

```bash
# Run all verification queries
mysql -u username -p database_name < database/migrations/tenant_files_verification_queries.sql
```

---

## 📊 **What Gets Created**

### **Tables (3)**
```
✅ tenant_files              - Main file tracking table
✅ tenant_file_access_log    - Access activity log
✅ tenant_file_versions      - File version history
```

### **Indexes (8+)**
```
✅ idx_tenant_id
✅ idx_tenant_category
✅ idx_stored_filename
✅ idx_related
✅ idx_uploaded_by
✅ idx_upload_date
✅ idx_is_deleted
✅ idx_tenant_not_deleted
```

### **Views (2)**
```
✅ v_active_tenant_files     - Active files view
✅ v_tenant_file_stats       - Statistics view
```

### **Triggers (1)**
```
✅ before_tenant_file_access_insert  - Updates access count
```

---

## 🔧 **Configuration Options**

### **Foreign Keys (Optional)**

By default, foreign keys are **commented out** to avoid issues. To enable:

1. **Uncomment in migration file** (lines ~150-180)
2. **Ensure tables exist:**
   - `tenants` table with `academy_reference` column
   - `users` table with `id` column

```sql
-- Uncomment these lines in the migration file:
ALTER TABLE tenant_files
ADD CONSTRAINT fk_tenant_files_tenant 
FOREIGN KEY (tenant_id) 
REFERENCES tenants(academy_reference) 
ON DELETE CASCADE;
```

---

### **Full-Text Search (Optional)**

Enable full-text search on description and tags:

```sql
-- Uncomment in migration file (line ~145)
ALTER TABLE tenant_files 
ADD FULLTEXT INDEX ft_search (description, tags);
```

---

## 📝 **Usage Examples**

### **Insert New File**

```sql
INSERT INTO tenant_files (
    tenant_id, file_category, original_filename, stored_filename,
    file_path, file_size, mime_type, uploaded_by,
    related_id, related_type, description
) VALUES (
    'soshigh_demo', 'document', 'student_report.pdf', 'report_1706600000.pdf',
    'school_soshigh_demo/uploads/documents/report_1706600000.pdf',
    245678, 'application/pdf', 1,
    101, 'student', 'Term 1 report card'
);
```

---

### **Get Files for Tenant**

```sql
SELECT 
    id, file_category, original_filename,
    ROUND(file_size / 1024 / 1024, 2) AS size_mb,
    upload_date
FROM tenant_files
WHERE tenant_id = 'soshigh_demo'
AND is_deleted = 0
ORDER BY upload_date DESC;
```

---

### **Get Storage Usage**

```sql
SELECT 
    tenant_id,
    COUNT(*) AS total_files,
    ROUND(SUM(file_size) / 1024 / 1024 / 1024, 2) AS total_gb
FROM tenant_files
WHERE is_deleted = 0
GROUP BY tenant_id;
```

---

### **Soft Delete File**

```sql
UPDATE tenant_files
SET 
    is_deleted = 1,
    deleted_at = NOW(),
    deleted_by = 1  -- user_id
WHERE id = 123;
```

---

### **Track File Access**

```sql
INSERT INTO tenant_file_access_log (
    file_id, tenant_id, user_id, access_type, ip_address
) VALUES (
    123, 'soshigh_demo', 1, 'download', '192.168.1.100'
);
-- Trigger automatically updates access_count in tenant_files
```

---

## 🔄 **Rollback Instructions**

If something goes wrong:

### **Option 1: Run Rollback Script**

```bash
mysql -u username -p database_name < database/migrations/rollback_tenant_files_table.sql
```

**What it does:**
1. Creates backup of current data
2. Drops all tables, views, triggers
3. Restores from migration backup (if exists)

---

### **Option 2: Manual Rollback**

```sql
-- Drop everything
DROP VIEW IF EXISTS v_active_tenant_files;
DROP VIEW IF EXISTS v_tenant_file_stats;
DROP TRIGGER IF EXISTS before_tenant_file_access_insert;
DROP TABLE IF EXISTS tenant_file_versions;
DROP TABLE IF EXISTS tenant_file_access_log;
DROP TABLE IF EXISTS tenant_files;
```

---

## ✅ **Verification Checklist**

After migration, verify:

### **Tables**
- [ ] `tenant_files` exists
- [ ] `tenant_file_access_log` exists
- [ ] `tenant_file_versions` exists

### **Structure**
- [ ] All columns present
- [ ] Indexes created
- [ ] Constraints working

### **Views**
- [ ] `v_active_tenant_files` works
- [ ] `v_tenant_file_stats` works

### **Triggers**
- [ ] `before_tenant_file_access_insert` exists

### **Data**
- [ ] Sample data inserted (if chosen)
- [ ] Can insert new records
- [ ] Can query data

---

## 🔍 **Verification Queries**

### **Quick Check**

```sql
-- Tables exist
SHOW TABLES LIKE '%tenant_file%';

-- Row counts
SELECT 
    'tenant_files' AS table_name, 
    COUNT(*) AS rows 
FROM tenant_files
UNION ALL
SELECT 
    'tenant_file_access_log', 
    COUNT(*) 
FROM tenant_file_access_log
UNION ALL
SELECT 
    'tenant_file_versions', 
    COUNT(*) 
FROM tenant_file_versions;
```

---

### **Data Integrity**

```sql
-- Check for NULL values
SELECT 
    SUM(CASE WHEN tenant_id IS NULL THEN 1 ELSE 0 END) AS null_tenant_id,
    SUM(CASE WHEN stored_filename IS NULL THEN 1 ELSE 0 END) AS null_filename,
    SUM(CASE WHEN file_size IS NULL THEN 1 ELSE 0 END) AS null_size
FROM tenant_files;

-- Check for duplicates
SELECT 
    stored_filename, 
    COUNT(*) AS count
FROM tenant_files
GROUP BY stored_filename
HAVING COUNT(*) > 1;
```

---

## 📊 **Common Queries**

### **Storage by Tenant**

```sql
SELECT * FROM v_tenant_file_stats;
```

---

### **Recent Uploads**

```sql
SELECT 
    tenant_id,
    file_category,
    original_filename,
    upload_date
FROM v_active_tenant_files
WHERE upload_date >= DATE_SUB(NOW(), INTERVAL 7 DAY)
ORDER BY upload_date DESC;
```

---

### **Most Accessed Files**

```sql
SELECT 
    original_filename,
    access_count,
    last_accessed_at
FROM v_active_tenant_files
ORDER BY access_count DESC
LIMIT 10;
```

---

### **Files by Category**

```sql
SELECT 
    file_category,
    COUNT(*) AS count,
    ROUND(SUM(file_size) / 1024 / 1024, 2) AS total_mb
FROM v_active_tenant_files
GROUP BY file_category;
```

---

## ⚠️ **Troubleshooting**

### **Issue: Foreign Key Error**

```
Error #1005: Can't create table (errno: 150 "Foreign key constraint")
```

**Solution:**
- Foreign keys are commented out by default
- Only uncomment if you have the required tables
- Check table names and column types match

---

### **Issue: Table Already Exists**

```
Error #1050: Table 'tenant_files' already exists
```

**Solution:**
```sql
-- Drop existing table first
DROP TABLE IF EXISTS tenant_files;

-- Or use backup
CREATE TABLE tenant_files_old AS SELECT * FROM tenant_files;
DROP TABLE tenant_files;
```

---

### **Issue: Slow Performance**

**Solution:**
```sql
-- Analyze tables
ANALYZE TABLE tenant_files;
ANALYZE TABLE tenant_file_access_log;

-- Optimize tables
OPTIMIZE TABLE tenant_files;
```

---

## 🎯 **Best Practices**

1. ✅ **Always backup before migration**
2. ✅ **Test on development first**
3. ✅ **Run verification queries**
4. ✅ **Monitor disk space**
5. ✅ **Set up regular backups**
6. ✅ **Clean up old/deleted files periodically**
7. ✅ **Use soft deletes (don't actually delete rows)**
8. ✅ **Hash files to detect duplicates**

---

## 📚 **Additional Resources**

| File | Purpose |
|------|---------|
| `create_tenant_files_table.sql` | Main migration |
| `rollback_tenant_files_table.sql` | Rollback script |
| `tenant_files_sample_data.sql` | Sample data |
| `tenant_files_verification_queries.sql` | 50+ queries |

---

## ✅ **Migration Complete Checklist**

- [ ] Backup created
- [ ] Migration executed successfully
- [ ] Tables created (3 tables)
- [ ] Indexes created (8+ indexes)
- [ ] Views created (2 views)
- [ ] Trigger created
- [ ] Verification queries run
- [ ] Sample data inserted (optional)
- [ ] No errors in logs
- [ ] Can insert test record
- [ ] Can query data
- [ ] Rollback script tested (optional)

---

## 🎉 **Success!**

If all checks pass, your tenant files tracking system is ready!

**Next Steps:**
1. Integrate with file upload code
2. Set up automated cleanup jobs
3. Monitor storage usage
4. Configure backup strategy

---

*File: TENANT_FILES_TABLE_MIGRATION_GUIDE.md*  
*Date: 2025-01-30*  
*Version: 1.0*  
*Status: ✅ Complete Migration Package*

