# 📁 Multi-Tenant File System - Complete Guide

## 🎯 **Overview**

The Multi-Tenant File System provides **secure, isolated file storage** for each school (tenant) in your multi-tenant education platform. Each tenant has their own directory structure with proper access controls and security measures.

---

## ✨ **Features**

- ✅ **Isolated Storage** - Each tenant has separate directory
- ✅ **Security** - Path traversal prevention, file validation
- ✅ **File Management** - Upload, delete, list files
- ✅ **Access Control** - Role-based file access
- ✅ **Storage Quotas** - Configurable per-tenant limits
- ✅ **File Logging** - Track all operations
- ✅ **Auto Cleanup** - Remove old temporary files
- ✅ **Type Validation** - Restrict file types per category

---

## 📂 **Directory Structure**

```
tenants/
  {tenant_id}/                     # e.g., soshigh_demo
    ├── uploads/
    │   ├── documents/             # PDFs, Word, Excel, etc.
    │   ├── profile_photos/        # Student/staff photos
    │   ├── payment_receipts/      # Payment confirmations
    │   └── reports/               # Generated reports
    ├── backups/                   # Database/system backups
    ├── logs/                      # Tenant-specific logs
    ├── temp/                      # Temporary files (auto-cleanup)
    ├── config.json                # Tenant configuration
    └── .htaccess                  # Security rules
```

---

## 🚀 **Quick Start**

### **1. Import Database Tables**

```sql
-- Import the schema
SOURCE database/TENANT_FILESYSTEM_TABLES.sql;
```

### **2. Initialize File System**

```php
<?php
require_once 'config.php';
require_once 'includes/tenant_filesystem.php';

// Initialize
$fs = new TenantFileSystem($conn);

// Create tenant directory
$result = $fs->createTenantDirectory('soshigh_demo');
```

### **3. Upload a File**

```php
// Handle file upload from form
if (isset($_FILES['document'])) {
    $result = $fs->uploadFile(
        $_SESSION['academy_reference'],  // Tenant ID
        $_FILES['document'],              // File from form
        'documents'                       // File type
    );
    
    if ($result['success']) {
        echo "File uploaded: " . $result['url'];
    } else {
        echo "Error: " . $result['error'];
    }
}
```

---

## 📋 **File Type Categories**

### **Allowed File Types**

| Category | Extensions | Max Size | Use Case |
|----------|-----------|----------|----------|
| `documents` | pdf, doc, docx, xls, xlsx, txt, csv | 10 MB | Student documents, forms |
| `profile_photos` | jpg, jpeg, png, gif, webp | 5 MB | Profile pictures |
| `payment_receipts` | pdf, jpg, jpeg, png | 5 MB | Payment confirmations |
| `reports` | pdf, xls, xlsx, csv, html | 20 MB | Generated reports |
| `backups` | sql, zip, tar, gz | 100 MB | Database backups |
| `logs` | log, txt | 10 MB | System logs |
| `temp` | * (all) | 20 MB | Temporary files |

---

## 🔧 **Core Methods**

### **1. Create Tenant Directory**

```php
$result = $fs->createTenantDirectory($tenant_id);

// Returns:
[
    'success' => true,
    'message' => 'Tenant directory structure created',
    'path' => '/path/to/tenants/soshigh_demo'
]
```

### **2. Get Tenant Path**

```php
// Get base path
$basePath = $fs->getTenantPath('soshigh_demo');

// Get specific type path
$documentsPath = $fs->getTenantPath('soshigh_demo', 'documents');
```

### **3. Upload File**

```php
$result = $fs->uploadFile(
    $tenant_id,           // Tenant identifier
    $_FILES['myfile'],    // File from form
    'documents',          // File type category
    'custom_name'         // Optional custom filename
);

// Returns:
[
    'success' => true,
    'filename' => 'custom_name_1234567890.pdf',
    'original_name' => 'document.pdf',
    'path' => '/full/path/to/file',
    'relative_path' => 'tenant/uploads/documents/file.pdf',
    'size' => 1500000,
    'url' => '/serve_file.php?tenant=...&type=...&file=...'
]
```

### **4. List Files**

```php
// List all files
$result = $fs->getTenantFiles($tenant_id);

// List specific type
$result = $fs->getTenantFiles($tenant_id, 'documents');

// List with options
$result = $fs->getTenantFiles($tenant_id, 'documents', [
    'sort_by' => 'modified',    // name, size, modified
    'sort_order' => 'desc',     // asc, desc
    'limit' => 10,
    'extension' => 'pdf',
    'min_size' => 1024,
    'max_size' => 10485760
]);

// Returns:
[
    'success' => true,
    'files' => [
        [
            'name' => 'document_1234567890.pdf',
            'extension' => 'pdf',
            'size' => 1500000,
            'size_formatted' => '1.43 MB',
            'modified' => '2025-01-29 10:30:00',
            'modified_timestamp' => 1738147800,
            'type' => 'documents',
            'path' => '/full/path/to/file',
            'relative_path' => 'tenant/uploads/documents/file.pdf',
            'url' => '/serve_file.php?...',
            'is_image' => false
        ]
    ],
    'count' => 1
]
```

### **5. Delete File**

```php
// Delete by filename and type
$result = $fs->deleteFile($tenant_id, 'document.pdf', 'documents');

// Delete by relative path
$result = $fs->deleteFile($tenant_id, 'uploads/documents/document.pdf');

// Returns:
[
    'success' => true,
    'message' => 'File deleted successfully'
]
```

### **6. Validate Access**

```php
// Check if tenant can access a file
$canAccess = $fs->validateTenantAccess($tenant_id, $filepath);

if ($canAccess) {
    // Allow access
} else {
    // Deny access (path traversal blocked)
}
```

### **7. Storage Usage**

```php
$usage = $fs->getStorageUsage($tenant_id);

// Returns:
[
    'success' => true,
    'used' => 52428800,
    'used_formatted' => '50 MB',
    'quota' => 1073741824,
    'quota_formatted' => '1 GB',
    'percentage' => 4.88,
    'available' => 1021313024,
    'available_formatted' => '974 MB'
]
```

### **8. Cleanup Temp Files**

```php
// Delete temp files older than 7 days
$result = $fs->cleanupTempFiles($tenant_id, 7);

// Returns:
[
    'success' => true,
    'deleted' => 15,
    'size_freed' => 31457280,
    'size_freed_formatted' => '30 MB',
    'message' => 'Deleted 15 temporary files older than 7 days'
]
```

---

## 🔒 **Security Features**

### **1. Path Traversal Prevention**

```php
// ✅ Valid paths
$fs->validateTenantAccess('soshigh', 'uploads/documents/file.pdf');

// ❌ Blocked paths
$fs->validateTenantAccess('soshigh', '../../../etc/passwd');
$fs->validateTenantAccess('soshigh', 'uploads/../../other_tenant/file.pdf');
```

### **2. File Type Validation**

```php
// Only allowed extensions per category
// Trying to upload .exe to 'documents' will fail
$result = $fs->uploadFile($tenant_id, $maliciousFile, 'documents');
// Returns: ['success' => false, 'error' => 'File type .exe not allowed']
```

### **3. File Size Limits**

```php
// Each category has max size
// Uploading 15MB file to 'documents' (10MB max) will fail
$result = $fs->uploadFile($tenant_id, $largeFile, 'documents');
// Returns: ['success' => false, 'error' => 'File size exceeds maximum allowed (10 MB)']
```

### **4. Storage Quotas**

```php
// Tenant storage quota enforced
// Uploading file that exceeds quota will fail
$result = $fs->uploadFile($tenant_id, $file, 'documents');
// Returns: ['success' => false, 'error' => 'Storage quota exceeded']
```

### **5. Access Control in serve_file.php**

```php
// File serving script checks:
// - User is authenticated
// - User belongs to correct tenant
// - User role has permission for file type
// - File path is valid (no traversal)
```

---

## 🛠️ **Practical Examples**

### **Example 1: Student Document Upload**

```php
<?php
session_start();
require_once 'config.php';
require_once 'includes/tenant_filesystem.php';

$fs = new TenantFileSystem($conn);

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['document'])) {
    $tenant_id = $_SESSION['academy_reference'];
    $student_id = $_SESSION['student_id'];
    
    // Upload with custom name
    $customName = "student_{$student_id}_document";
    $result = $fs->uploadFile($tenant_id, $_FILES['document'], 'documents', $customName);
    
    if ($result['success']) {
        // Save to database
        $stmt = $conn->prepare("
            INSERT INTO student_documents 
            (student_id, filename, file_path, file_size, uploaded_at)
            VALUES (?, ?, ?, ?, NOW())
        ");
        $stmt->execute([
            $student_id,
            $result['filename'],
            $result['relative_path'],
            $result['size']
        ]);
        
        echo json_encode([
            'success' => true,
            'message' => 'Document uploaded successfully',
            'url' => $result['url']
        ]);
    } else {
        echo json_encode(['success' => false, 'error' => $result['error']]);
    }
}
?>

<!-- HTML Form -->
<form method="post" enctype="multipart/form-data">
    <input type="file" name="document" accept=".pdf,.doc,.docx" required>
    <button type="submit">Upload Document</button>
</form>
```

### **Example 2: Profile Photo Upload with Validation**

```php
<?php
function uploadProfilePhoto($user_id, $tenant_id, $user_type = 'student') {
    global $fs, $conn;
    
    if (!isset($_FILES['profile_photo'])) {
        return ['success' => false, 'error' => 'No photo uploaded'];
    }
    
    $file = $_FILES['profile_photo'];
    
    // Validate image
    $imageInfo = @getimagesize($file['tmp_name']);
    if ($imageInfo === false) {
        return ['success' => false, 'error' => 'Invalid image file'];
    }
    
    // Check dimensions
    list($width, $height) = $imageInfo;
    if ($width < 100 || $height < 100) {
        return ['success' => false, 'error' => 'Image too small (minimum 100x100px)'];
    }
    
    if ($width > 4000 || $height > 4000) {
        return ['success' => false, 'error' => 'Image too large (maximum 4000x4000px)'];
    }
    
    // Delete old photo if exists
    $stmt = $conn->prepare("SELECT profile_photo FROM {$user_type}s WHERE id = ?");
    $stmt->execute([$user_id]);
    $oldPhoto = $stmt->fetchColumn();
    
    if ($oldPhoto) {
        $fs->deleteFile($tenant_id, $oldPhoto, 'profile_photos');
    }
    
    // Upload new photo
    $customName = "{$user_type}_{$user_id}_profile";
    $result = $fs->uploadFile($tenant_id, $file, 'profile_photos', $customName);
    
    if ($result['success']) {
        // Update database
        $stmt = $conn->prepare("UPDATE {$user_type}s SET profile_photo = ? WHERE id = ?");
        $stmt->execute([$result['relative_path'], $user_id]);
        
        return [
            'success' => true,
            'photo_url' => $result['url']
        ];
    }
    
    return $result;
}
?>
```

### **Example 3: Admin File Manager**

```php
<?php
// Admin page to view all tenant files
session_start();
require_once 'config.php';
require_once 'includes/tenant_filesystem.php';

$fs = new TenantFileSystem($conn);
$tenant_id = $_SESSION['academy_reference'];

// Get storage usage
$storage = $fs->getStorageUsage($tenant_id);

// Get files by category
$categories = ['documents', 'profile_photos', 'payment_receipts', 'reports'];
$filesByCategory = [];

foreach ($categories as $category) {
    $files = $fs->getTenantFiles($tenant_id, $category, [
        'sort_by' => 'modified',
        'sort_order' => 'desc',
        'limit' => 50
    ]);
    
    if ($files['success']) {
        $filesByCategory[$category] = $files;
    }
}
?>

<!DOCTYPE html>
<html>
<head>
    <title>File Manager</title>
    <style>
        .storage-bar {
            width: 100%;
            height: 30px;
            background: #e0e0e0;
            border-radius: 5px;
            overflow: hidden;
        }
        .storage-used {
            height: 100%;
            background: <?= $storage['percentage'] > 80 ? '#f44336' : '#4caf50' ?>;
            width: <?= $storage['percentage'] ?>%;
        }
        .file-list { margin: 20px 0; }
        .file-item { 
            padding: 10px; 
            border: 1px solid #ddd; 
            margin: 5px 0;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        .category { 
            margin: 30px 0; 
            padding: 20px;
            background: #f9f9f9;
            border-radius: 8px;
        }
    </style>
</head>
<body>
    <h1>File Manager - <?= htmlspecialchars($tenant_id) ?></h1>
    
    <!-- Storage Usage -->
    <div class="storage">
        <h2>Storage Usage</h2>
        <p><?= $storage['used_formatted'] ?> of <?= $storage['quota_formatted'] ?> (<?= $storage['percentage'] ?>%)</p>
        <div class="storage-bar">
            <div class="storage-used"></div>
        </div>
    </div>
    
    <!-- Files by Category -->
    <?php foreach ($filesByCategory as $category => $files): ?>
    <div class="category">
        <h2><?= ucwords(str_replace('_', ' ', $category)) ?> (<?= $files['count'] ?>)</h2>
        
        <div class="file-list">
            <?php foreach ($files['files'] as $file): ?>
            <div class="file-item">
                <div>
                    <strong><?= htmlspecialchars($file['name']) ?></strong><br>
                    <small><?= $file['size_formatted'] ?> - <?= $file['modified'] ?></small>
                </div>
                <div>
                    <a href="<?= htmlspecialchars($file['url']) ?>" target="_blank">View</a>
                    <a href="?delete=<?= urlencode($file['name']) ?>&type=<?= $category ?>" 
                       onclick="return confirm('Delete this file?')">Delete</a>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
    </div>
    <?php endforeach; ?>
</body>
</html>
```

### **Example 4: Batch Upload**

```php
<?php
// Upload multiple files at once
function uploadMultipleFiles($tenant_id, $files_array, $type) {
    global $fs;
    
    $results = [
        'success' => [],
        'failed' => []
    ];
    
    foreach ($files_array['tmp_name'] as $index => $tmpName) {
        $file = [
            'name' => $files_array['name'][$index],
            'type' => $files_array['type'][$index],
            'tmp_name' => $tmpName,
            'error' => $files_array['error'][$index],
            'size' => $files_array['size'][$index]
        ];
        
        $result = $fs->uploadFile($tenant_id, $file, $type);
        
        if ($result['success']) {
            $results['success'][] = $result;
        } else {
            $results['failed'][] = [
                'filename' => $file['name'],
                'error' => $result['error']
            ];
        }
    }
    
    return $results;
}

// Usage
if (isset($_FILES['documents'])) {
    $results = uploadMultipleFiles(
        $_SESSION['academy_reference'],
        $_FILES['documents'],
        'documents'
    );
    
    echo "Uploaded: " . count($results['success']) . "\n";
    echo "Failed: " . count($results['failed']) . "\n";
}
?>

<!-- HTML Form for Multiple Files -->
<form method="post" enctype="multipart/form-data">
    <input type="file" name="documents[]" multiple>
    <button type="submit">Upload Multiple</button>
</form>
```

---

## 🔧 **Configuration**

### **Modify File Type Restrictions**

Edit `includes/tenant_filesystem.php`:

```php
private $allowedFileTypes = [
    'documents' => ['pdf', 'doc', 'docx', 'xls', 'xlsx', 'txt', 'csv', 'ppt', 'pptx'],
    // Add more types...
];
```

### **Modify Size Limits**

```php
private $maxFileSizes = [
    'documents' => 10485760, // 10MB
    'profile_photos' => 5242880, // 5MB
    // Modify as needed...
];
```

### **Change Default Storage Quota**

```php
// In tenant config.json
{
    "storage_quota": 2147483648  // 2GB
}

// Or programmatically:
$config = json_decode(file_get_contents($configPath), true);
$config['storage_quota'] = 2147483648;
file_put_contents($configPath, json_encode($config, JSON_PRETTY_PRINT));
```

---

## 📊 **Database Tables**

### **file_operations_log**
Logs all file operations (upload, delete, etc.)

### **file_access_log**
Logs when files are accessed/downloaded

### **tenant_files** (Optional)
Registry of all files with metadata

### **file_shares** (Optional)
For file sharing between users

### **tenant_storage_quotas**
Track storage usage and quotas

See `database/TENANT_FILESYSTEM_TABLES.sql` for complete schema.

---

## 🧹 **Maintenance**

### **Auto-Cleanup Temp Files**

```php
// Add to cron job (daily)
$fs->cleanupTempFiles($tenant_id, 7); // Delete files older than 7 days
```

### **Recalculate Storage Usage**

```sql
UPDATE tenant_storage_quotas
SET storage_used = (
    SELECT COALESCE(SUM(file_size), 0)
    FROM tenant_files
    WHERE tenant_id = tenant_storage_quotas.tenant_id
    AND deleted_at IS NULL
),
last_calculated_at = NOW()
WHERE tenant_id = 'your_tenant_id';
```

### **Clean Old Logs**

```sql
-- Keep last 90 days
DELETE FROM file_access_log WHERE accessed_at < DATE_SUB(NOW(), INTERVAL 90 DAY);
DELETE FROM file_operations_log WHERE created_at < DATE_SUB(NOW(), INTERVAL 90 DAY);
```

---

## ⚠️ **Common Issues**

### **Issue: "Failed to create directory"**

```
Error: Failed to create directory: uploads/documents
```

**Solution:**
```bash
# Check permissions
chmod -R 755 tenants/
chown -R www-data:www-data tenants/

# Or for specific tenant
chmod -R 755 tenants/soshigh_demo/
```

### **Issue: "Storage quota exceeded"**

**Solution:**
```php
// Increase quota in config.json
$fs->updateStorageQuota($tenant_id, 2147483648); // 2GB

// Or clean up old files
$fs->cleanupTempFiles($tenant_id, 7);
```

### **Issue: "File not found" when accessing via serve_file.php**

**Solution:**
- Check file actually exists
- Verify tenant ID is correct
- Check user has proper role/permissions
- Validate file path

---

## 🎯 **Best Practices**

1. ✅ **Always validate user input**
2. ✅ **Use custom filenames to prevent conflicts**
3. ✅ **Delete old files when replacing (e.g., profile photos)**
4. ✅ **Implement storage quotas**
5. ✅ **Run cleanup jobs regularly**
6. ✅ **Log important operations**
7. ✅ **Use role-based access control**
8. ✅ **Validate file types and sizes**

---

## 📚 **Additional Resources**

- **Complete Examples:** `TENANT_FILESYSTEM_USAGE_EXAMPLES.php`
- **Database Schema:** `database/TENANT_FILESYSTEM_TABLES.sql`
- **File Serving:** `serve_file.php`
- **Core Class:** `includes/tenant_filesystem.php`

---

## ✅ **Summary**

This multi-tenant file system provides:

- ✅ **Complete isolation** between tenants
- ✅ **Secure file handling** with validation
- ✅ **Flexible file management** (upload, delete, list)
- ✅ **Storage quotas** and usage tracking
- ✅ **Comprehensive logging**
- ✅ **Role-based access control**
- ✅ **Path traversal prevention**
- ✅ **Auto cleanup** for temporary files

**All files are organized, secure, and production-ready!** 🚀

---

*Last Updated: 2025*  
*File: TENANT_FILESYSTEM_GUIDE.md*

