# ⚡ Tenant Directory Manager - Quick Reference

## 🚀 **Quick Start (1 Minute)**

```php
<?php
require_once 'includes/tenant_directory_manager.php';
$manager = new TenantDirectoryManager($conn);

// Initialize new tenant
$result = $manager->initializeTenant('school_id', 'School Name');
// ✅ Creates 10 directories automatically
?>
```

---

## 📋 **Common Operations**

### **Initialize Tenant**
```php
$manager->initializeTenant('soshigh_demo', 'SOS SOSHIGH School');
```

### **Get Paths**
```php
$root = $manager->getTenantRootPath('soshigh_demo');
$docs = $manager->getTenantUploadPath('soshigh_demo', 'documents');
$backup = $manager->getTenantBackupPath('soshigh_demo', 'database');
$logs = $manager->getTenantLogPath('soshigh_demo', 'errors');
```

### **Check Storage**
```php
$storage = $manager->calculateTenantStorageUsage('soshigh_demo');
echo "Used: {$storage['total_formatted']} ({$storage['percentage']}%)";
```

### **Verify Directories**
```php
$verify = $manager->verifyTenantDirectories('soshigh_demo');
if (!$verify['success']) {
    echo "Issues: {$verify['directories_missing']} missing";
}
```

### **Update Quota**
```php
$manager->updateTenantQuota('soshigh_demo', 2147483648); // 2GB
```

---

## 📂 **Directory Structure**

```
tenants/school_{id}/
├── uploads/ (documents, profile_photos, payment_receipts, reports)
├── backups/ (database, files)
├── logs/ (access, errors, payments)
├── temp/
├── config.json
└── .htaccess
```

---

## 🔧 **Integration Points**

### **School Creation**
```php
$manager->initializeTenant($academy_reference, $school_name);
```

### **File Upload**
```php
$path = $manager->getTenantUploadPath($_SESSION['academy_reference'], 'documents');
move_uploaded_file($_FILES['file']['tmp_name'], $path . '/' . $filename);
```

### **Storage Widget**
```php
$storage = $manager->calculateTenantStorageUsage($_SESSION['academy_reference']);
// Display: {$storage['total_formatted']} / {$storage['quota_formatted']}
```

---

## ⚙️ **Configuration**

```php
// Custom settings
$manager->createTenantConfig('school_id', [
    'storage_quota' => 5368709120,  // 5GB
    'enable_notifications' => true,
    'notification_email' => 'admin@school.edu'
]);
```

---

## 🐛 **Common Issues**

**Permission Denied:**
```bash
chmod 755 tenants/
chown www-data:www-data tenants/
```

**Invalid Tenant ID:**
- ✅ Valid: `school_001`, `soshigh-demo`
- ❌ Invalid: `school@123`, `ab`

---

## 📚 **Full Documentation**

- Complete Guide: `TENANT_DIRECTORY_MANAGER_GUIDE.md`
- Examples: `TENANT_DIRECTORY_MANAGER_EXAMPLES.php`
- Integration: `TENANT_DIRECTORY_MANAGER_INTEGRATION.md`

---

## 🧪 **Test**

```
Visit: test_tenant_directory_manager.php
Expected: ✅ All 10 tests pass
```

---

**Ready in 5 minutes! 🚀**

