# 📱 Tenant-Specific SMS Configuration Guide

## Overview

Your multi-tenant platform now supports **tenant-specific SMS gateway configuration**! Each school can:
- Use their own SMS gateway credentials
- Or use the super admin's default gateway
- Enable/disable SMS independently

---

## 🎯 Key Features

### **1. Three Configuration Levels**
```
Super Admin Default Gateway
        ↓ (fallback)
Tenant Custom Gateway
        ↓ (option)
SMS Disabled Per Tenant
```

### **2. Flexible Control**
- **Super Admin**: Sets default gateway for all schools
- **Tenant Admin**: Can override with custom gateway or disable SMS
- **Automatic Fallback**: If tenant gateway fails, tries super admin default

### **3. Cost Control**
- Each school can use their own SMS credits
- Or share super admin's gateway
- Track SMS usage per tenant

---

## 🚀 Setup Guide

### **Step 1: Super Admin - Set Default Gateway**

1. Log in as Super Admin
2. Navigate to **Settings** → **SMS Gateway**
   - Or visit: `/super_admin/settings/sms_gateway.php`
3. Toggle **"Enable Default SMS Gateway"**
4. Select Provider:
   - **Africa's Talking** (Recommended for Africa)
   - **Twilio** (Global)
5. Enter your credentials
6. Click **"Save Default Gateway Settings"**

#### **Example Configuration:**
```
Enable Default SMS Gateway: ✅ ON
SMS Provider: Africa's Talking
Username: sandbox
API Key: atsk_abc123...
Sender ID: SKOLO
```

---

### **Step 2: Tenant Admin - Configure SMS**

#### **Option A: Use Default Gateway** (Easiest)
1. Log in as Tenant Admin
2. Go to **Settings**
3. Scroll to **"SMS Gateway Configuration"**
4. Leave **"Use Custom SMS Gateway"** toggled **OFF**
5. SMS will use super admin's default gateway automatically

#### **Option B: Use Custom Gateway** (Own Credentials)
1. Log in as Tenant Admin
2. Go to **Settings**
3. Scroll to **"SMS Gateway Configuration"**
4. Toggle **"Use Custom SMS Gateway"** to **ON**
5. Toggle **"Enable SMS for this School"** to **ON**
6. Select Provider and enter your credentials
7. Click **"Save Settings"**

#### **Option C: Disable SMS** (No SMS at all)
1. Toggle **"Use Custom SMS Gateway"** to **ON**
2. Toggle **"Enable SMS for this School"** to **OFF**
3. No SMS will be sent for this school

---

## 📊 How It Works

### **Decision Flow:**
```
Student registers at School A
        ↓
Check: Does School A have custom SMS enabled?
        ↓
    ├─ YES → Use School A's gateway
    │         ├─ Success ✅
    │         └─ Failure → Try super admin default
    │
    └─ NO → Use super admin default gateway
              ├─ Success ✅
              └─ Failure → Fall back to email
```

### **Configuration Priority:**
1. **Tenant Custom Gateway** (if `use_custom_sms` = ON)
2. **Super Admin Default** (if tenant not configured)
3. **Disabled** (if neither enabled)

---

## 💰 Cost Management

### **Scenario 1: Shared Gateway (Default)**
- Super admin pays for all SMS
- Schools use shared credentials
- Easier setup, one bill

### **Scenario 2: Custom Gateway Per School**
- Each school has own Africa's Talking account
- Each school pays for their own SMS
- Better cost tracking per school

### **Scenario 3: Mixed**
- Some schools use custom gateway (own credits)
- Some schools use default gateway (shared credits)
- Flexible billing

---

## 🗄️ Database Schema

### **Super Admin Settings**
```sql
CREATE TABLE super_admin_settings (
  setting_key VARCHAR(100),      -- e.g., 'default_sms_enabled'
  setting_value TEXT             -- e.g., '1'
);
```

### **Tenant Settings**
```sql
CREATE TABLE tenant_settings (
  academy_reference VARCHAR(50),  -- e.g., 'KINE'
  setting_key VARCHAR(100),       -- e.g., 'use_custom_sms'
  setting_value TEXT              -- e.g., '1'
);
```

### **Settings Keys:**

#### Super Admin (Global Default):
- `default_sms_enabled` - Enable/disable default gateway
- `default_sms_provider` - 'africastalking' or 'twilio'
- `default_at_username` - Africa's Talking username
- `default_at_api_key` - Africa's Talking API key
- `default_at_sender_id` - Sender ID
- `default_twilio_account_sid` - Twilio Account SID
- `default_twilio_auth_token` - Twilio Auth Token
- `default_twilio_from_number` - Twilio from number

#### Tenant (School-Specific):
- `use_custom_sms` - Use custom gateway (1) or default (0)
- `sms_enabled` - Enable SMS for this school
- `sms_provider` - 'africastalking' or 'twilio'
- `sms_username` - Africa's Talking username
- `sms_api_key` - Africa's Talking API key
- `sms_sender_id` - Sender ID
- `sms_twilio_sid` - Twilio Account SID
- `sms_twilio_token` - Twilio Auth Token
- `sms_twilio_from` - Twilio from number

---

## 📁 Files Modified

### **New Files:**
1. **super_admin/settings/sms_gateway.php**
   - Super admin UI for default gateway configuration
   - Shows statistics (schools using default vs custom)

2. **database/migrations/create_super_admin_settings_table.sql**
   - Creates `super_admin_settings` table
   - Initializes default SMS settings

3. **SMS_TENANT_CONFIGURATION_GUIDE.md** (this file)
   - Complete documentation

### **Updated Files:**
1. **admin/settings.php**
   - Added SMS Gateway Configuration section
   - Tenant can configure custom gateway or use default

2. **includes/sms_functions.php**
   - Added `getSMSConfig()` - Retrieves tenant or default config
   - Updated `sendSMS()` - Uses tenant-specific config
   - Updated `sendSMS_AfricasTalking()` - Accepts config array
   - Updated `sendSMS_Twilio()` - Accepts config array

---

## 🎯 Usage Examples

### **Example 1: All Schools Use Default**
```php
// Super Admin configures default
Africa's Talking:
  Username: production_user
  API Key: atsk_prod_key_123

// All schools automatically use this
School A → Uses super admin gateway ✅
School B → Uses super admin gateway ✅
School C → Uses super admin gateway ✅
```

### **Example 2: Mixed Configuration**
```php
// Super Admin has default configured
// School A uses custom gateway
School A Settings:
  use_custom_sms: ON
  sms_username: school_a_user
  sms_api_key: atsk_school_a_key

// School B uses default
School B Settings:
  use_custom_sms: OFF
  → Automatically uses super admin default ✅

// School C disables SMS
School C Settings:
  use_custom_sms: ON
  sms_enabled: OFF
  → No SMS sent ❌
```

### **Example 3: Automatic Fallback**
```php
// School A has custom gateway but API key is invalid
School A Registration:
  1. Try School A's gateway
  2. Fails (invalid credentials)
  3. Automatically try super admin default
  4. Success ✅
  5. Email also sent as backup
```

---

## 🔧 Testing

### **Test 1: Super Admin Default Gateway**
1. Configure super admin default gateway
2. Visit `/check_sms_config.php`
3. Send test SMS
4. Verify SMS received

### **Test 2: Tenant Custom Gateway**
1. Log in as tenant admin
2. Enable custom SMS gateway
3. Enter credentials
4. Test student registration
5. Verify SMS code received

### **Test 3: Fallback Behavior**
1. Configure tenant with invalid credentials
2. Register student
3. Verify system falls back to default gateway
4. Check logs for fallback message

---

## 📊 Monitoring

### **Super Admin Dashboard**
Visit: `/super_admin/settings/sms_gateway.php`

Shows:
- **Schools Using Default Gateway**: Count
- **Schools Using Custom Gateway**: Count
- **Schools with SMS Disabled**: Count

### **SMS Logs**
Query database:
```sql
SELECT 
    sl.*,
    ts.setting_value as tenant_gateway
FROM sms_logs sl
LEFT JOIN tenant_settings ts 
    ON ts.setting_key = 'use_custom_sms'
ORDER BY sent_at DESC;
```

### **Per-Tenant Usage**
```sql
SELECT 
    COUNT(*) as sms_count,
    SUM(CASE WHEN status='SUCCESS' THEN 1 ELSE 0 END) as success_count,
    academy_reference
FROM sms_logs
GROUP BY academy_reference;
```

---

## 🐛 Troubleshooting

### **Problem: SMS not sending for tenant**
**Check:**
1. Does tenant have `use_custom_sms` = 1?
2. If yes, is `sms_enabled` = 1?
3. Is API key valid?
4. Is super admin default configured as fallback?

### **Problem: All tenants not receiving SMS**
**Check:**
1. Is super admin default gateway enabled?
2. Are credentials correct?
3. Check `super_admin_settings` table

### **Problem: Can't see SMS settings in admin**
**Solution:** Run the database migration:
```bash
mysql -u root -p multi_tenant_skoloi < database/migrations/create_super_admin_settings_table.sql
```

---

## 🎓 Best Practices

### **1. For Super Admins:**
- ✅ Always configure a default gateway as fallback
- ✅ Monitor SMS usage across all tenants
- ✅ Set up alerts for failed SMS
- ✅ Regularly review gateway statistics

### **2. For Tenant Admins:**
- ✅ Use default gateway initially (easiest)
- ✅ Switch to custom gateway if you need separate billing
- ✅ Test SMS after configuration changes
- ✅ Keep API keys secure (never share)

### **3. For Developers:**
- ✅ Always pass `academy_reference` when sending SMS
- ✅ Log SMS failures for debugging
- ✅ Implement graceful fallback (SMS → Email → Direct link)

---

## 🔐 Security

### **API Key Protection:**
- ✅ Stored as text (consider encryption in production)
- ✅ Not displayed in forms (password fields)
- ✅ Only updated if new value provided
- ✅ Not included in exports or logs

### **Access Control:**
- ✅ Super admin: Full access to all gateways
- ✅ Tenant admin: Only their school's settings
- ✅ Students/Parents: No access to SMS settings

---

## 📈 Future Enhancements

### **Potential Improvements:**
1. **SMS Credit Tracking** - Track remaining credits per gateway
2. **Usage Reports** - Detailed SMS usage per tenant
3. **Cost Allocation** - Automatic billing for shared gateway
4. **Provider Failover** - Try multiple providers automatically
5. **Bulk SMS** - Send announcements to all parents
6. **SMS Templates** - Customizable message templates
7. **Delivery Reports** - Real-time delivery status
8. **Rate Limiting** - Prevent SMS spam

---

## ✅ Quick Reference

### **Default Gateway (Super Admin)**
```
Path: /super_admin/settings/sms_gateway.php
Purpose: Set global default for all schools
Required: Yes (recommended)
```

### **Custom Gateway (Tenant)**
```
Path: /admin/settings.php → SMS Gateway Configuration
Purpose: Override default with custom credentials
Required: No (optional)
```

### **SMS Functions**
```php
// Automatically uses tenant or default config
sendSMS($phone, $message, $type, $academy_reference);

// Get config for specific tenant
$config = getSMSConfig($academy_reference);

// Check if SMS enabled
if ($config['sms_enabled']) {
    // Send SMS
}
```

---

## 🎉 Summary

You now have a **flexible, tenant-aware SMS system**:

✅ **Super Admin**: Sets default gateway for all schools  
✅ **Tenant Admin**: Can use default or configure custom gateway  
✅ **Automatic Fallback**: Tenant → Default → Email  
✅ **Cost Control**: Separate billing per tenant (optional)  
✅ **Easy Setup**: Works out of the box with default gateway  
✅ **Monitoring**: Track usage per tenant  
✅ **Secure**: API keys protected  

**Perfect for multi-tenant SaaS platforms!** 🚀

---

**Last Updated:** <?= date('Y-m-d') ?>  
**Version:** 2.0 (Tenant-Specific Configuration)

