# 📱 SMS Gateway Setup Guide

## Overview

Your multi-tenant platform now supports SMS verification and credential delivery using **Africa's Talking** (primary) or **Twilio** (backup).

---

## ✅ Features Implemented

### 1. SMS Verification
- **Primary verification method** instead of email
- 6-digit verification codes
- 15-minute expiry
- Automatic fallback to email if SMS fails

### 2. Credential Delivery
- Login credentials sent via SMS after approval
- Includes username, password, and login URL
- Also sent via email as backup

### 3. Payment Notifications
- Payment confirmations via SMS
- Payment reminders (future enhancement)

### 4. Graceful Fallback
```
SMS → Email → Direct Link
```
Students can always proceed even if SMS/email fails!

---

## 🚀 Installation Steps

### Step 1: Install Dependencies

Run this command in your project root:

```bash
composer require africastalking/africastalking
composer require twilio/sdk
```

Or manually create/update `composer.json`:

```json
{
    "require": {
        "phpmailer/phpmailer": "^6.8",
        "africastalking/africastalking": "^3.0",
        "twilio/sdk": "^6.0"
    }
}
```

Then run:
```bash
composer install
```

### Step 2: Sign Up for Africa's Talking

1. Go to: https://africastalking.com/
2. Click **Sign Up** (Free account)
3. Verify your email and phone
4. **Get free sandbox credits** for testing

### Step 3: Get Your API Credentials

1. Log into Africa's Talking dashboard
2. Go to **Settings** → **API Key**
3. Copy your **Username** (e.g., "sandbox" or your custom username)
4. Copy your **API Key**
5. (Optional) Register a **Sender ID** for branding (e.g., "SKOLO")

### Step 4: Configure Your System

Open `config.php` and update these lines:

```php
// SMS Gateway Configuration
define('SMS_ENABLED', true); // Set to true to enable
define('SMS_PROVIDER', 'africastalking'); // Use Africa's Talking

// Africa's Talking Configuration
define('AT_USERNAME', 'your-username-here'); // From dashboard
define('AT_API_KEY', 'your-api-key-here'); // From dashboard
define('AT_SENDER_ID', 'SKOLO'); // Your sender ID (or leave as SKOLO)
define('AT_ENVIRONMENT', 'sandbox'); // Change to 'production' when ready

// SMS Settings
define('SMS_COUNTRY_CODE', '+268'); // Eswatini (+268)
define('SMS_VERIFICATION_EXPIRY', 15); // Code expires in 15 minutes
```

### Step 5: Create Database Tables

Run the SQL migration:

```bash
mysql -u root -p multi_tenant_skoloi < database/migrations/create_sms_tables.sql
```

Or manually run:

```sql
-- SMS Logs Table
CREATE TABLE IF NOT EXISTS `sms_logs` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `phone_number` varchar(20) NOT NULL,
  `message` text NOT NULL,
  `message_type` enum('VERIFICATION','CREDENTIALS','PAYMENT_REMINDER','PAYMENT_CONFIRMATION','GENERAL','TEST') NOT NULL DEFAULT 'GENERAL',
  `status` enum('SUCCESS','FAILED','PENDING') NOT NULL DEFAULT 'PENDING',
  `error_message` text DEFAULT NULL,
  `provider` varchar(50) DEFAULT NULL,
  `message_id` varchar(255) DEFAULT NULL,
  `cost` decimal(10,4) DEFAULT NULL,
  `sent_at` timestamp NOT NULL DEFAULT current_timestamp(),
  PRIMARY KEY (`id`),
  KEY `phone_number` (`phone_number`),
  KEY `status` (`status`),
  KEY `sent_at` (`sent_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- SMS Verifications Table
CREATE TABLE IF NOT EXISTS `sms_verifications` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `phone_number` varchar(20) NOT NULL,
  `verification_code` varchar(10) NOT NULL,
  `expires_at` datetime NOT NULL,
  `used` tinyint(1) NOT NULL DEFAULT 0,
  `verified_at` datetime DEFAULT NULL,
  `attempts` int(11) NOT NULL DEFAULT 0,
  `created_at` timestamp NOT NULL DEFAULT current_timestamp(),
  PRIMARY KEY (`id`),
  KEY `phone_number` (`phone_number`),
  KEY `verification_code` (`verification_code`),
  KEY `expires_at` (`expires_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;
```

### Step 6: Test Your Configuration

1. Visit: `https://yourdomain.com/check_sms_config.php`
2. Verify all settings show ✅
3. Enter your phone number (+268XXXXXXXX)
4. Click **Send Test SMS**
5. Check if you receive the SMS

---

## 📊 How It Works

### Registration Flow

```
Student fills registration form
    ↓
Generate 6-digit verification code
    ↓
Try to send SMS with code
    ↓
├─ SMS Success → Show code input form
│
└─ SMS Failed → Try email
    ↓
    ├─ Email Success → Show "Check email" message
    │
    └─ Email Failed → Show direct verification button
```

### Approval Flow

```
Admin approves application
    ↓
Create student account
    ↓
Generate username & password
    ↓
Send credentials via SMS
    ↓
Also send via email (backup)
    ↓
Student receives both
```

---

## 💰 Pricing

### Africa's Talking (Eswatini)
- **Verification SMS**: ~E0.20 - E0.30 per SMS
- **Credentials SMS**: ~E0.20 - E0.30 per SMS

### Example Costs:
- **100 students/month**: ~E40 - E60
- **1000 students/month**: ~E400 - E600

**ROI:** Much more reliable than email, especially for mobile users!

---

## 🔧 Configuration Options

### Disable SMS Temporarily

In `config.php`:
```php
define('SMS_ENABLED', false); // Disable SMS, use email only
```

### Switch to Production

After testing with sandbox:

1. Add credits to your Africa's Talking account
2. Update `config.php`:
```php
define('AT_ENVIRONMENT', 'production');
define('AT_USERNAME', 'your-production-username');
```

### Use Twilio Instead

If you prefer Twilio:

1. Sign up at: https://www.twilio.com/
2. Get your Account SID and Auth Token
3. Update `config.php`:
```php
define('SMS_PROVIDER', 'twilio');
define('TWILIO_ACCOUNT_SID', 'your-sid');
define('TWILIO_AUTH_TOKEN', 'your-token');
define('TWILIO_FROM_NUMBER', '+268XXXXXXXX');
```

---

## 📁 Files Created/Updated

### New Files:
- ✅ `includes/sms_functions.php` - Core SMS functions
- ✅ `verify_sms_code.php` - SMS verification handler
- ✅ `resend_verification.php` - Resend code handler
- ✅ `check_sms_config.php` - Configuration tester
- ✅ `database/migrations/create_sms_tables.sql` - Database tables
- ✅ `SMS_SETUP_GUIDE.md` - This guide

### Updated Files:
- ✅ `config.php` - Added SMS configuration
- ✅ `register_student.php` - Added SMS verification
- ✅ `includes/student_account_creator.php` - Send credentials via SMS

---

## 🐛 Troubleshooting

### SMS Not Sending?

1. **Check API Key**: Visit `check_sms_config.php`
2. **Verify Credits**: Check Africa's Talking dashboard
3. **Phone Format**: Must be +268XXXXXXXX (8 digits after +268)
4. **Check Logs**: Look in PHP error logs for details

### Common Errors

#### "SMS not configured"
- API key is still placeholder (`your-api-key-here`)
- Solution: Update `AT_API_KEY` in `config.php`

#### "Invalid phone number"
- Phone number format is wrong
- Solution: Use +268 followed by 8 digits

#### "Connection timeout"
- Server can't reach Africa's Talking API
- Solution: Check firewall, or use sandbox mode

### View SMS Logs

Check sent SMS history:
```sql
SELECT * FROM sms_logs ORDER BY sent_at DESC LIMIT 50;
```

Check verification attempts:
```sql
SELECT * FROM sms_verifications ORDER BY created_at DESC LIMIT 50;
```

---

## 📈 SMS Statistics

Visit `/check_sms_config.php` to see:
- Total SMS sent
- Success rate
- Failed SMS
- Today's SMS count

Or query directly:
```sql
SELECT 
    COUNT(*) as total,
    SUM(CASE WHEN status = 'SUCCESS' THEN 1 ELSE 0 END) as success,
    SUM(CASE WHEN status = 'FAILED' THEN 1 ELSE 0 END) as failed
FROM sms_logs;
```

---

## 🎯 Future Enhancements

### Already Built Into System:
- ✅ SMS verification codes
- ✅ Credential delivery
- ✅ Graceful fallbacks
- ✅ Multi-provider support
- ✅ Complete logging

### Easy to Add:
- Payment reminder SMS (function already exists)
- Payment confirmation SMS (function already exists)
- Bulk SMS to parents
- Exam result notifications
- Attendance alerts

---

## 💡 Best Practices

### 1. Test with Sandbox First
- Use sandbox credentials
- Test all flows
- Verify SMS format and content
- Check costs per SMS

### 2. Monitor Usage
- Check SMS logs regularly
- Monitor success rate
- Track costs
- Set up alerts for failures

### 3. Keep Email Backup
- Never disable email completely
- Use SMS as primary, email as fallback
- Provides redundancy

### 4. Phone Number Validation
- Always validate format before sending
- Store in international format (+268...)
- Handle missing/invalid numbers gracefully

---

## 🆘 Support

### Africa's Talking Support:
- Email: support@africastalking.com
- Docs: https://developers.africastalking.com/
- Slack: https://slackin-africastalking.now.sh/

### System Issues:
- Check `check_sms_config.php` first
- Review PHP error logs
- Check `sms_logs` table for details

---

## ✅ Quick Start Checklist

- [ ] Run `composer install` to get SDKs
- [ ] Sign up for Africa's Talking
- [ ] Get API credentials
- [ ] Update `config.php` with credentials
- [ ] Run database migrations
- [ ] Test with `check_sms_config.php`
- [ ] Try registering a test student
- [ ] Verify SMS is received
- [ ] Test the verification flow
- [ ] Check SMS statistics

---

## 🎉 You're Done!

Your platform now has professional SMS capabilities:
- ✅ Instant verification codes
- ✅ Automatic credential delivery
- ✅ Better user experience
- ✅ Higher conversion rates
- ✅ More reliable than email

**Next:** Add credits to your Africa's Talking account and go live!

---

**Last Updated:** <?= date('Y-m-d') ?>  
**Version:** 1.0

