# 🎓 Moodle Auto-Enrollment System - Complete Implementation Guide

## 📋 Overview

This system automatically enrolls students in Moodle courses when:
1. Admin assigns subjects to students
2. Parents pay for subjects
3. Students register and select subjects

**Key Feature:** Uses subject-to-course mapping to avoid course code conflicts!

---

## 🏗️ Architecture

```
┌─────────────────┐
│   Admin Panel   │
│  Assign Subject │
└────────┬────────┘
         │
         ↓
┌──────────────────────────────┐
│ Subject-to-Course Mapping    │
│ (subject_moodle_mapping)     │
│ - Internal Subject ID        │
│ - Moodle Course ID/Shortname │
│ - Auto-enroll Settings       │
└────────┬──────────────────────┘
         │
         ↓
┌──────────────────────────────┐
│ Enrollment Handler           │
│ - Check mapping exists       │
│ - Ensure user in Moodle      │
│ - Enroll in correct course   │
│ - Log result                 │
└────────┬──────────────────────┘
         │
         ↓
┌──────────────────────────────┐
│   Moodle LMS                 │
│   Student enrolled in course │
└──────────────────────────────┘
```

---

## 🗄️ Database Setup

### **Step 1: Run SQL Migration**

```bash
mysql -u root -p your_database < database/migrations/create_moodle_mapping_table.sql
```

This creates 3 tables:
1. **`subject_moodle_mapping`** - Maps subjects to courses
2. **`moodle_enrollment_log`** - Logs all enrollment attempts
3. **`moodle_sync_queue`** - Queue for background processing

---

## ⚙️ Configuration Steps

### **Step 1: Configure Moodle Settings**

In your school settings, add these configuration options:

```php
// In tenant_settings table or config file
moodle_enabled = 1
moodle_url = "https://yourschool.moodle.com"
moodle_token = "your_webservice_token"
moodle_service = "moodle_mobile_app" // or your custom service
```

### **Step 2: Map Subjects to Courses**

Go to: `admin/moodle/configure_mapping.php`

For each subject:
1. **Select Internal Subject** (e.g., Grade 7 - Mathematics)
2. **Select Moodle Course** (from dropdown) OR enter manually:
   - Course Shortname: `G7-MATH`
   - Course ID Number: `G7-MAT-003`
3. **Enable Auto-enroll** ✓
4. **Enable Sync** ✓
5. **Save Mapping**

**Example Mappings:**

| Internal Subject | Moodle Course Shortname | Auto-Enroll |
|------------------|-------------------------|-------------|
| Grade 7 Mathematics | G7-MATH | ✓ |
| Grade 7 English | G7-ENG | ✓ |
| Grade 7 SiSwati | G7-SIS | ✓ |
| Form 1 Science | F1-SCI | ✓ |

---

## 🔧 Integration Points

### **1. Admin Assigns Subjects**

When admin assigns subjects to student:

```php
// In admin/students/assign_subjects.php or similar

require_once '../../includes/moodle_enrollment_handler.php';

// After saving to student_subject table
foreach ($selected_subjects as $subject_id) {
    // Insert into student_subject
    $stmt = $pdo->prepare("
        INSERT INTO student_subject (student_id, subject_id, enrolled_at)
        VALUES (?, ?, NOW())
    ");
    $stmt->execute([$student_id, $subject_id]);
    
    // AUTO-ENROLL IN MOODLE
    $enrollmentHandler = new MoodleEnrollmentHandler($academy_reference, $pdo);
    $result = $enrollmentHandler->enrollStudentInSubject($student_id, $subject_id);
    
    if ($result['success']) {
        echo "✓ Enrolled in Moodle: " . $subject_name;
    } else {
        echo "⚠ Moodle enrollment failed: " . $result['error'];
        // Subject still assigned in your system, just not in Moodle
    }
}
```

### **2. Parent Payment Triggers Enrollment**

Already implemented in `parent/process_payment.php`:

```php
// Around line 110-129
foreach ($payment_data['subjects'] as $subject_id) {
    // Check if already enrolled
    $stmt = $pdo->prepare("
        SELECT id FROM student_subject 
        WHERE student_id = ? AND subject_id = ?
    ");
    $stmt->execute([$student_id, $subject_id]);
    
    if (!$stmt->fetch()) {
        // Enroll in subject (your system)
        $stmt = $pdo->prepare("
            INSERT INTO student_subject (student_id, subject_id, enrolled_at)
            VALUES (?, ?, NOW())
        ");
        $stmt->execute([$student_id, $subject_id]);
        
        // AUTO-ENROLL IN MOODLE
        require_once '../includes/moodle_enrollment_handler.php';
        $enrollmentHandler = new MoodleEnrollmentHandler($student['academy_reference'], $pdo);
        $enrollmentHandler->enrollStudentInSubject($student_id, $subject_id);
    }
}
```

### **3. Student Registration**

In `register.php` (student self-registration):

```php
// After creating student and subjects
require_once 'includes/moodle_enrollment_handler.php';

$enrollmentHandler = new MoodleEnrollmentHandler($academy_reference, $pdo);
$result = $enrollmentHandler->enrollStudentInMultipleSubjects(
    $studentId, 
    $data['subject_ids']
);

if ($result['success']) {
    $_SESSION['moodle_enrolled'] = $result['enrolled'] . ' courses';
}
```

---

## 🚀 Usage Examples

### **Example 1: Enroll Single Student**

```php
require_once 'includes/moodle_enrollment_handler.php';

$handler = new MoodleEnrollmentHandler('KINE', $pdo);
$result = $handler->enrollStudentInSubject($student_id, $subject_id);

if ($result['success']) {
    echo "Student enrolled successfully!";
} else {
    echo "Error: " . $result['error'];
}
```

### **Example 2: Enroll in Multiple Subjects**

```php
$subject_ids = [1, 2, 3, 4]; // Math, English, Science, SiSwati

$result = $handler->enrollStudentInMultipleSubjects($student_id, $subject_ids);

echo "Enrolled: {$result['enrolled']} / {$result['total']} subjects";
echo "Failed: {$result['failed']}";
```

### **Example 3: Sync Existing Students**

```php
// Sync all subjects for one student
$result = $handler->syncStudentEnrollments($student_id);

// Sync all students in one subject
$result = $handler->syncSubjectEnrollments($subject_id);
```

### **Example 4: Queue for Background Processing**

```php
// For bulk operations, use queue
foreach ($students as $student) {
    $handler->queueEnrollment($student['id'], $subject_id, $priority = 5);
}

// Process queue (in cron job)
$handler->processQueue($limit = 50);
```

---

## 🔄 How Conflict Resolution Works

### **The Problem:**
Multiple schools use same subjects but different Moodle instances/courses.

**Example:**
- **School A:** Grade 7 Math → Moodle Course "G7-MATH-A" (ID: 101)
- **School B:** Grade 7 Math → Moodle Course "G7-MATH-B" (ID: 201)

### **The Solution:**
Use `subject_moodle_mapping` table with `academy_reference`:

```sql
-- School A mapping
INSERT INTO subject_moodle_mapping 
(academy_reference, subject_id, moodle_course_shortname)
VALUES ('SCHOOL_A', 1, 'G7-MATH-A');

-- School B mapping
INSERT INTO subject_moodle_mapping 
(academy_reference, subject_id, moodle_course_shortname)
VALUES ('SCHOOL_B', 1, 'G7-MATH-B');
```

**System automatically:**
1. Looks up mapping for current school
2. Uses that school's Moodle course
3. No conflicts! ✅

---

## 📊 Monitoring & Logging

### **View Enrollment Log**

```sql
SELECT 
    el.*, 
    s.full_name as student_name,
    sub.name as subject_name
FROM moodle_enrollment_log el
JOIN students s ON el.student_id = s.id
JOIN subjects sub ON el.subject_id = sub.id
WHERE el.academy_reference = 'YOUR_SCHOOL'
ORDER BY el.attempted_at DESC
LIMIT 50;
```

### **Check Success Rate**

```sql
SELECT 
    status,
    COUNT(*) as count,
    ROUND(COUNT(*) * 100.0 / (SELECT COUNT(*) FROM moodle_enrollment_log), 2) as percentage
FROM moodle_enrollment_log
WHERE academy_reference = 'YOUR_SCHOOL'
GROUP BY status;
```

### **Find Failed Enrollments**

```sql
SELECT 
    s.full_name,
    sub.name as subject,
    el.error_message,
    el.attempted_at
FROM moodle_enrollment_log el
JOIN students s ON el.student_id = s.id
JOIN subjects sub ON el.subject_id = sub.id
WHERE el.status = 'failed'
AND el.academy_reference = 'YOUR_SCHOOL'
ORDER BY el.attempted_at DESC;
```

---

## 🛠️ Admin Tools

### **1. Configure Mappings**
```
URL: admin/moodle/configure_mapping.php
```
- Map subjects to Moodle courses
- Enable/disable auto-enrollment
- Test connections

### **2. Sync All Students**
```
URL: admin/moodle/sync_all_students.php
```
- Bulk sync all enrolled students
- Re-sync failed enrollments
- View sync progress

### **3. Enrollment Log**
```
URL: admin/moodle/enrollment_log.php
```
- View all enrollment attempts
- Filter by status (success/failed)
- Export logs

### **4. Test Connection**
```
URL: admin/moodle/test_connection.php
```
- Verify Moodle credentials
- Test API connectivity
- View available courses

---

## 🔐 Security Considerations

### **1. Moodle Token**
```php
// Store securely in tenant_settings or config
$moodle_token = getSchoolSetting('moodle_token');

// Never expose in frontend
// Never log in plain text
```

### **2. User Validation**
```php
// Always verify student belongs to school
if ($student['academy_reference'] !== $current_school) {
    throw new Exception("Unauthorized");
}
```

### **3. API Error Handling**
```php
try {
    $result = $handler->enrollStudentInSubject($student_id, $subject_id);
} catch (Exception $e) {
    // Log error
    error_log("Moodle enrollment error: " . $e->getMessage());
    
    // Don't expose API details to user
    $user_message = "Could not complete enrollment. Please try again later.";
}
```

---

## ⏰ Cron Jobs

### **Process Enrollment Queue**

```bash
# /etc/crontab
*/5 * * * * php /path/to/Multi-Tanent/cron/process_moodle_queue.php
```

```php
// cron/process_moodle_queue.php
<?php
require_once '../includes/functions.php';
require_once '../includes/moodle_enrollment_handler.php';

$pdo = getDB();

// Get all active schools
$stmt = $pdo->query("SELECT reference_code FROM academy_references WHERE is_active = 1");
$schools = $stmt->fetchAll();

foreach ($schools as $school) {
    $handler = new MoodleEnrollmentHandler($school['reference_code'], $pdo);
    $result = $handler->processQueue(50); // Process 50 items per school
    
    echo "{$school['reference_code']}: Processed {$result['processed']} items\n";
}
?>
```

---

## 🐛 Troubleshooting

### **Issue 1: "No Moodle mapping configured"**

**Solution:**
1. Go to `admin/moodle/configure_mapping.php`
2. Create mapping for that subject
3. Try enrollment again

---

### **Issue 2: "Could not create user in Moodle"**

**Possible causes:**
- Email already exists in Moodle
- Username conflicts
- Invalid characters in name

**Solution:**
```php
// Check user exists first
$existing_user = $moodle->getUserByUsername($username);
if ($existing_user) {
    $moodle_user_id = $existing_user['id'];
    // Update student record
    $pdo->prepare("UPDATE students SET moodle_user_id = ? WHERE id = ?")
        ->execute([$moodle_user_id, $student_id]);
}
```

---

### **Issue 3: "Course not found"**

**Solution:**
- Verify Moodle course exists
- Check course shortname spelling
- Update mapping with correct shortname

---

### **Issue 4: Enrollments not happening**

**Checklist:**
- [ ] Moodle enabled for school?
- [ ] Subject mapped to course?
- [ ] Auto-enroll enabled in mapping?
- [ ] Sync enabled in mapping?
- [ ] Valid Moodle token?
- [ ] Moodle webservice active?

---

## 📝 Testing Checklist

- [ ] Run SQL migration successfully
- [ ] Configure Moodle settings (URL, token)
- [ ] Map at least one subject to Moodle course
- [ ] Test connection to Moodle
- [ ] Admin assigns subject → Student enrolled in Moodle
- [ ] Parent pays for subject → Student enrolled in Moodle
- [ ] Check enrollment log shows success
- [ ] Try unenroll → Student removed from Moodle
- [ ] Test with unmapped subject → Graceful failure
- [ ] Test bulk sync → All students enrolled
- [ ] Check queue processing works
- [ ] Verify multi-school isolation (no conflicts)

---

## 🎯 Benefits

✅ **Automatic Enrollment** - No manual work in Moodle
✅ **Conflict-Free** - Per-school course mapping
✅ **Scalable** - Queue system for bulk operations
✅ **Auditable** - Complete enrollment log
✅ **Recoverable** - Failed enrollments can be retried
✅ **Multi-Tenant** - Each school has own mappings
✅ **Flexible** - Can disable per subject if needed

---

## 📚 Files Created

1. `database/migrations/create_moodle_mapping_table.sql` - Database schema
2. `admin/moodle/configure_mapping.php` - Admin UI for mappings
3. `includes/moodle_enrollment_handler.php` - Enrollment logic
4. `MOODLE_AUTO_ENROLLMENT_GUIDE.md` - This guide

---

## 🚀 Next Steps

1. ✅ Run SQL migration
2. ✅ Configure Moodle settings in admin panel
3. ✅ Map subjects to courses
4. ✅ Test with one student
5. ✅ Integrate into existing workflows
6. ✅ Set up cron job for queue processing
7. ✅ Monitor enrollment log
8. ✅ Train admins on mapping interface

---

**Your system is now ready for automatic Moodle enrollment!** 🎉

Questions? Check the troubleshooting section or review the enrollment log for details.

