-- ================================================================
-- CREATE SUPER ADMIN CREDENTIALS
-- ================================================================
-- This script creates a proper super admin account with database-based authentication
-- Run this in cPanel → phpMyAdmin → SQL tab
-- ================================================================

-- ================================================================
-- PART 1: Add missing columns to admins table
-- ================================================================

-- Add email column (ignore error if exists)
ALTER TABLE admins ADD COLUMN email VARCHAR(255) NULL AFTER username;

-- Add is_active column (ignore error if exists)
ALTER TABLE admins ADD COLUMN is_active TINYINT(1) DEFAULT 1 AFTER academy_reference;

-- Add created_at column (ignore error if exists)
ALTER TABLE admins ADD COLUMN created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP AFTER is_active;

-- Note: If you see "Duplicate column name" errors above, that's OK! 
-- It means the columns already exist.

-- Show current admins table structure
DESCRIBE admins;

-- Show existing admins
SELECT id, username, 
       COALESCE(email, 'N/A') as email, 
       COALESCE(academy_reference, 'NULL') as academy_reference, 
       COALESCE(is_active, 1) as is_active
FROM admins 
ORDER BY id;

-- ================================================================
-- PART 2: Create Super Admin Account
-- ================================================================

-- Insert super admin with bcrypt hashed password
-- Default password: SuperAdmin123! (CHANGE THIS AFTER FIRST LOGIN!)
-- Password hash generated with bcrypt cost 10
INSERT INTO admins (
    username, 
    email,
    password_hash, 
    academy_reference, 
    is_active, 
    created_at
) VALUES (
    'superadmin',
    'superadmin@melanegroup.com',
    '$2y$10$YPKvAWqN5xJZ3PZE8.4hXOGxB6y4p4jQZzN8fKH5yDnq7x.FQh3Sm',  -- Password: SuperAdmin123!
    NULL,  -- NULL = Super Admin (can access all schools)
    1,     -- Active
    NOW()
)
ON DUPLICATE KEY UPDATE
    password_hash = '$2y$10$YPKvAWqN5xJZ3PZE8.4hXOGxB6y4p4jQZzN8fKH5yDnq7x.FQh3Sm',
    academy_reference = NULL,
    is_active = 1;

-- ================================================================
-- PART 3: Also update existing 'admin' user to be super admin
-- ================================================================

-- Make the default 'admin' user a super admin too
UPDATE admins 
SET 
    academy_reference = NULL,  -- NULL = Super Admin
    is_active = 1,
    email = COALESCE(email, 'admin@melanegroup.com')
WHERE username = 'admin';

-- ================================================================
-- PART 4: Verify Super Admin Accounts
-- ================================================================

-- Show all super admin accounts (academy_reference = NULL)
SELECT 
    id,
    username,
    email,
    academy_reference,
    is_active,
    created_at,
    CASE 
        WHEN academy_reference IS NULL THEN '✅ SUPER ADMIN'
        ELSE '❌ School Admin'
    END AS admin_type
FROM admins
ORDER BY 
    CASE WHEN academy_reference IS NULL THEN 0 ELSE 1 END,
    id;

-- ================================================================
-- PART 5: Create alternative super admin (optional backup)
-- ================================================================

-- Create a backup super admin account
INSERT INTO admins (
    username, 
    email,
    password_hash, 
    academy_reference, 
    is_active, 
    created_at
) VALUES (
    'platform_admin',
    'platform@melanegroup.com',
    '$2y$10$YPKvAWqN5xJZ3PZE8.4hXOGxB6y4p4jQZzN8fKH5yDnq7x.FQh3Sm',  -- Password: SuperAdmin123!
    NULL,  -- Super Admin
    1,
    NOW()
)
ON DUPLICATE KEY UPDATE
    password_hash = '$2y$10$YPKvAWqN5xJZ3PZE8.4hXOGxB6y4p4jQZzN8fKH5yDnq7x.FQh3Sm',
    academy_reference = NULL;

-- ================================================================
-- PART 6: Final Verification
-- ================================================================

-- Count super admins
SELECT 
    COUNT(*) as total_super_admins
FROM admins 
WHERE academy_reference IS NULL AND is_active = 1;

-- Show all super admins with details
SELECT 
    '🛡️ SUPER ADMIN ACCOUNTS' as info,
    username,
    email,
    'Password: SuperAdmin123!' as default_password,
    '⚠️ CHANGE PASSWORD IMMEDIATELY!' as warning,
    created_at
FROM admins 
WHERE academy_reference IS NULL AND is_active = 1;

-- ================================================================
-- DONE! Super Admin Accounts Created
-- ================================================================

-- CREDENTIALS CREATED:
-- ═══════════════════════════════════════════════════════════════
-- 
-- Account 1 (Primary):
--   Username: superadmin
--   Email: superadmin@melanegroup.com
--   Password: SuperAdmin123!
--   Access: ALL schools + Super Admin dashboard
-- 
-- Account 2 (Existing):
--   Username: admin
--   Email: admin@melanegroup.com
--   Password: admin123 (or your current password)
--   Access: ALL schools + Super Admin dashboard
-- 
-- Account 3 (Backup):
--   Username: platform_admin
--   Email: platform@melanegroup.com
--   Password: SuperAdmin123!
--   Access: ALL schools + Super Admin dashboard
-- 
-- ═══════════════════════════════════════════════════════════════
-- 
-- ⚠️ IMPORTANT SECURITY STEPS:
-- 1. Login with new credentials immediately
-- 2. Change all passwords from defaults
-- 3. Update email addresses to your real emails
-- 4. Test super admin access
-- 5. Delete/disable any unused admin accounts
-- 
-- ═══════════════════════════════════════════════════════════════
