-- ================================================================
-- COMPLETE DATABASE FIX - ALL REMAINING ISSUES
-- ================================================================
-- This script fixes ALL remaining database issues in one go
-- Run this in cPanel → phpMyAdmin → SQL tab
-- Time: 3-5 minutes
-- ================================================================

-- ================================================================
-- PART 1: Create missing tables
-- ================================================================

-- Create grade_subject table (renamed from grade_subjects)
CREATE TABLE IF NOT EXISTS `grade_subject` (
  `grade_id` int(11) NOT NULL,
  `subject_id` int(11) NOT NULL,
  PRIMARY KEY (`grade_id`,`subject_id`),
  KEY `idx_grade_subject_subject_id` (`subject_id`),
  KEY `subject_id` (`subject_id`),
  CONSTRAINT `fk_grade_subject_grade` FOREIGN KEY (`grade_id`) REFERENCES `grades` (`id`) ON DELETE CASCADE ON UPDATE CASCADE,
  CONSTRAINT `fk_grade_subject_subject` FOREIGN KEY (`subject_id`) REFERENCES `subjects` (`id`) ON DELETE CASCADE ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- Create student_subject table alias (in addition to student_subjects)
CREATE TABLE IF NOT EXISTS `student_subject` (
  `student_id` int(11) NOT NULL,
  `subject_id` int(11) NOT NULL,
  PRIMARY KEY (`student_id`,`subject_id`),
  KEY `idx_student_subject_subject_id` (`subject_id`),
  KEY `subject_id` (`subject_id`),
  CONSTRAINT `fk_student_subject_student` FOREIGN KEY (`student_id`) REFERENCES `students` (`id`) ON DELETE CASCADE ON UPDATE CASCADE,
  CONSTRAINT `fk_student_subject_subject` FOREIGN KEY (`subject_id`) REFERENCES `subjects` (`id`) ON DELETE CASCADE ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- ================================================================
-- PART 2: Populate grade_subject from grade_subjects
-- ================================================================

-- Copy data from grade_subjects to grade_subject
INSERT IGNORE INTO `grade_subject` (grade_id, subject_id)
SELECT grade_id, subject_id FROM `grade_subjects`
WHERE EXISTS (SELECT 1 FROM information_schema.tables 
              WHERE table_schema = DATABASE() 
              AND table_name = 'grade_subjects');

-- If grade_subject is empty, populate with default data for all grades
INSERT IGNORE INTO `grade_subject` (grade_id, subject_id)
SELECT g.id, s.id
FROM grades g
CROSS JOIN subjects s
WHERE NOT EXISTS (SELECT 1 FROM grade_subject LIMIT 1);

-- ================================================================
-- PART 3: Sync student_subject with student_subjects
-- ================================================================

-- Copy data from student_subjects to student_subject
INSERT IGNORE INTO `student_subject` (student_id, subject_id)
SELECT student_id, subject_id FROM `student_subjects`
WHERE EXISTS (SELECT 1 FROM information_schema.tables 
              WHERE table_schema = DATABASE() 
              AND table_name = 'student_subjects');

-- Sync back from student_subject to student_subjects
INSERT IGNORE INTO `student_subjects` (student_id, subject_id)
SELECT student_id, subject_id FROM `student_subject`
WHERE EXISTS (SELECT 1 FROM information_schema.tables 
              WHERE table_schema = DATABASE() 
              AND table_name = 'student_subjects');

-- ================================================================
-- PART 4: Add missing columns to students table
-- ================================================================

-- Add first_name if missing
ALTER TABLE `students` 
ADD COLUMN IF NOT EXISTS `first_name` VARCHAR(100) NULL AFTER `id`;

-- Add last_name if missing  
ALTER TABLE `students` 
ADD COLUMN IF NOT EXISTS `last_name` VARCHAR(100) NULL AFTER `first_name`;

-- Add full_name if missing
ALTER TABLE `students` 
ADD COLUMN IF NOT EXISTS `full_name` VARCHAR(200) NULL AFTER `last_name`;

-- Add status column if missing
ALTER TABLE `students` 
ADD COLUMN IF NOT EXISTS `status` ENUM('active','suspended','deleted') DEFAULT 'active' AFTER `payment_status`;

-- Add code column to subjects table if missing
ALTER TABLE `subjects` 
ADD COLUMN IF NOT EXISTS `code` VARCHAR(20) NULL AFTER `name`;

-- Add created_at column to payments table if missing
ALTER TABLE `payments` 
ADD COLUMN IF NOT EXISTS `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP AFTER `academy_reference`;

-- Add currency column to payments table if missing
ALTER TABLE `payments` 
ADD COLUMN IF NOT EXISTS `currency` VARCHAR(8) DEFAULT 'SZL' AFTER `amount`;

-- Add msisdn column to payments table if missing
ALTER TABLE `payments` 
ADD COLUMN IF NOT EXISTS `msisdn` VARCHAR(32) NULL AFTER `currency`;

-- ================================================================
-- PART 5: Populate name fields and codes
-- ================================================================

-- If full_name exists but first_name/last_name don't, split full_name
UPDATE `students` 
SET 
    `first_name` = TRIM(SUBSTRING_INDEX(`full_name`, ' ', 1)),
    `last_name` = TRIM(SUBSTRING_INDEX(`full_name`, ' ', -1))
WHERE (`first_name` IS NULL OR `first_name` = '') 
AND (`last_name` IS NULL OR `last_name` = '')
AND `full_name` IS NOT NULL 
AND `full_name` != '';

-- If first_name/last_name exist but full_name doesn't, combine them
UPDATE `students` 
SET `full_name` = CONCAT(TRIM(COALESCE(`first_name`, '')), ' ', TRIM(COALESCE(`last_name`, '')))
WHERE (`full_name` IS NULL OR `full_name` = '')
AND (`first_name` IS NOT NULL OR `last_name` IS NOT NULL);

-- Ensure all name fields are populated
UPDATE `students` 
SET 
    `first_name` = COALESCE(NULLIF(`first_name`, ''), SUBSTRING_INDEX(`full_name`, ' ', 1), `username`),
    `last_name` = COALESCE(NULLIF(`last_name`, ''), SUBSTRING_INDEX(`full_name`, ' ', -1), ''),
    `full_name` = COALESCE(NULLIF(`full_name`, ''), CONCAT(`first_name`, ' ', `last_name`), `username`)
WHERE `full_name` IS NULL OR `full_name` = '' OR `first_name` IS NULL OR `first_name` = '';

-- Populate subject codes from names (first 3 letters uppercase)
UPDATE `subjects` 
SET `code` = UPPER(LEFT(`name`, 3))
WHERE `code` IS NULL OR `code` = '';

-- ================================================================
-- PART 6: Update status field based on payment_status
-- ================================================================

-- Check if deleted_at column exists before using it
UPDATE `students` 
SET `status` = CASE 
    WHEN `payment_status` = 'suspended' THEN 'suspended'
    ELSE 'active'
END
WHERE `status` IS NULL OR `status` = '';

-- ================================================================
-- PART 7: Add indexes for performance
-- ================================================================

-- Add index on status if not exists
ALTER TABLE `students` ADD INDEX IF NOT EXISTS `idx_students_status` (`status`);

-- Add index on grade_id if not exists
ALTER TABLE `students` ADD INDEX IF NOT EXISTS `idx_students_grade_id` (`grade_id`);

-- ================================================================
-- PART 8: Verify all fixes
-- ================================================================

-- Check tables exist
SELECT 
    'Tables Check' as verification,
    (SELECT COUNT(*) FROM information_schema.tables WHERE table_schema = DATABASE() AND table_name = 'grade_subject') as has_grade_subject,
    (SELECT COUNT(*) FROM information_schema.tables WHERE table_schema = DATABASE() AND table_name = 'student_subject') as has_student_subject,
    (SELECT COUNT(*) FROM information_schema.tables WHERE table_schema = DATABASE() AND table_name = 'student_subjects') as has_student_subjects;

-- Check columns exist
SELECT 
    'Columns Check' as verification,
    (SELECT COUNT(*) FROM information_schema.columns WHERE table_schema = DATABASE() AND table_name = 'students' AND column_name = 'first_name') as has_first_name,
    (SELECT COUNT(*) FROM information_schema.columns WHERE table_schema = DATABASE() AND table_name = 'students' AND column_name = 'last_name') as has_last_name,
    (SELECT COUNT(*) FROM information_schema.columns WHERE table_schema = DATABASE() AND table_name = 'students' AND column_name = 'full_name') as has_full_name,
    (SELECT COUNT(*) FROM information_schema.columns WHERE table_schema = DATABASE() AND table_name = 'students' AND column_name = 'status') as has_status;

-- Show data counts
SELECT 'grade_subject' as table_name, COUNT(*) as count FROM grade_subject
UNION ALL
SELECT 'student_subject', COUNT(*) FROM student_subject
UNION ALL
SELECT 'student_subjects', COUNT(*) FROM student_subjects
UNION ALL
SELECT 'students', COUNT(*) FROM students;

-- Show sample student data
SELECT 
    'Sample Student Data' as info,
    id,
    username,
    first_name,
    last_name,
    full_name,
    status,
    payment_status
FROM students 
ORDER BY id 
LIMIT 5;

-- ================================================================
-- PART 9: Create views for backward compatibility (optional)
-- ================================================================

-- Create view for grade_subjects (if original table exists)
CREATE OR REPLACE VIEW `grade_subjects_view` AS
SELECT * FROM `grade_subject`;

-- ================================================================
-- DONE! All database issues fixed!
-- ================================================================

SELECT 
    '✅ DATABASE FIX COMPLETE!' as status,
    'All tables created' as tables_status,
    'All columns added' as columns_status,
    'All data populated' as data_status,
    'Ready to use!' as result;

