# Card Payment Integration - MTN MoMo

## 🎯 Overview
Add credit/debit card payment option alongside Mobile Money payment for students who don't have MoMo or prefer to pay by card.

## 💳 Payment Options Available

### Current:
- ✅ MTN Mobile Money (MoMo)

### Adding:
- 🆕 **Credit/Debit Card Payment** (Visa, Mastercard, etc.)

## 🔧 Implementation Options

### **Option 1: MTN MoMo Card Payment API** (Recommended)
MTN provides card payment gateway that integrates with their Collection API.

**Pros:**
- Same provider as MoMo
- Unified payment tracking
- Same settlement account
- No additional merchant account needed

**Cons:**
- Requires separate API subscription
- Higher transaction fees (~3-5%)

### **Option 2: PayGate / DPO PayGate** (Popular in Eswatini)
PayGate is a popular payment gateway in Southern Africa.

**Pros:**
- Supports multiple card types
- Well-established in Eswatini
- Direct bank settlement
- Good documentation

**Cons:**
- Separate merchant account required
- Additional integration
- Monthly fees may apply

### **Option 3: Stripe** (International)
Global payment processor supporting cards worldwide.

**Pros:**
- Best developer experience
- Excellent documentation
- No monthly fees (pay per transaction)
- Supports multiple currencies

**Cons:**
- May have issues with Eswatini
- Higher fees for international transactions
- Payout delays

## 🚀 Recommended: MTN MoMo + PayGate Hybrid

I recommend implementing **both MTN MoMo and PayGate** to give students maximum flexibility:

1. **MTN MoMo** - for local mobile money users
2. **PayGate Card Payment** - for card users

## 📋 Implementation Steps

### Step 1: Register with PayGate
1. Visit: https://www.paygate.co.za
2. Apply for merchant account
3. Get credentials:
   - PayGate ID
   - Encryption Key
   - Test credentials for sandbox

### Step 2: Update Payment Page
Add card payment option to registration payment step.

### Step 3: Implement PayGate Integration
Create card payment handler similar to MoMo handler.

## 💻 Code Implementation

### 1. Update Config.php
```php
// PayGate Configuration
define('PAYGATE_ID', 'your-paygate-id');
define('PAYGATE_SECRET', 'your-paygate-secret');
define('PAYGATE_URL', 'https://secure.paygate.co.za/payweb3/process.trans');
define('PAYGATE_TEST_MODE', true); // Set to false for production
```

### 2. Create Payment Selection Page
Update the registration flow to show payment options:

```php
// After student registration, show payment options
?>
<div class="payment-options">
    <h3>Choose Payment Method</h3>
    
    <div class="row">
        <!-- MoMo Payment -->
        <div class="col-md-6">
            <div class="payment-card" onclick="selectPaymentMethod('momo')">
                <i class="bi bi-phone"></i>
                <h4>Mobile Money</h4>
                <p>Pay with MTN MoMo</p>
                <span class="badge">Instant</span>
            </div>
        </div>
        
        <!-- Card Payment -->
        <div class="col-md-6">
            <div class="payment-card" onclick="selectPaymentMethod('card')">
                <i class="bi bi-credit-card"></i>
                <h4>Credit/Debit Card</h4>
                <p>Visa, Mastercard, etc.</p>
                <span class="badge">Secure</span>
            </div>
        </div>
    </div>
</div>
```

### 3. Create PayGate Payment Handler
```php
// payments/paygate/request.php
<?php
require_once '../../config.php';
require_once '../../includes/functions.php';

session_start();

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    die('Invalid request');
}

$student_id = intval($_POST['student_id']);
$amount = floatval($_POST['amount']);

// Get student details
$pdo = getDB();
$stmt = $pdo->prepare("SELECT * FROM students WHERE id = ?");
$stmt->execute([$student_id]);
$student = $stmt->fetch();

if (!$student) {
    die('Student not found');
}

// Generate unique reference
$reference = 'KINE_' . $student_id . '_' . time();

// Create payment record
$stmt = $pdo->prepare("
    INSERT INTO payments 
    (student_id, amount, currency, reference_id, external_id, status, payment_method, academy_reference, created_at) 
    VALUES (?, ?, 'SZL', ?, ?, 'PENDING', 'CARD', 'KINE', NOW())
");
$stmt->execute([
    $student_id,
    $amount,
    $reference,
    $reference
]);
$payment_id = $pdo->lastInsertId();

// Prepare PayGate request
$return_url = PUBLIC_BASE_URL . '/payments/paygate/return.php';
$notify_url = PUBLIC_BASE_URL . '/payments/paygate/notify.php';

$paygate_data = [
    'PAYGATE_ID' => PAYGATE_ID,
    'REFERENCE' => $reference,
    'AMOUNT' => number_format($amount * 100, 0, '', ''), // Amount in cents
    'CURRENCY' => 'SZL',
    'RETURN_URL' => $return_url,
    'TRANSACTION_DATE' => date('Y-m-d H:i:s'),
    'LOCALE' => 'en',
    'COUNTRY' => 'SZ',
    'EMAIL' => $student['email'],
    'NOTIFY_URL' => $notify_url
];

// Generate checksum
$checksum_string = '';
foreach ($paygate_data as $key => $value) {
    $checksum_string .= $value;
}
$checksum_string .= PAYGATE_SECRET;
$paygate_data['CHECKSUM'] = md5($checksum_string);

// Initialize PayGate transaction
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, PAYGATE_URL);
curl_setopt($ch, CURLOPT_POST, 1);
curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($paygate_data));
curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);

$response = curl_exec($ch);
curl_close($ch);

// Parse response
parse_str($response, $response_data);

if (isset($response_data['PAY_REQUEST_ID'])) {
    // Redirect to PayGate payment page
    $payment_url = 'https://secure.paygate.co.za/payweb3/process.trans?PAY_REQUEST_ID=' . $response_data['PAY_REQUEST_ID'];
    
    // Update payment with PayGate reference
    $stmt = $pdo->prepare("
        UPDATE payments 
        SET momo_reference_id = ? 
        WHERE id = ?
    ");
    $stmt->execute([$response_data['PAY_REQUEST_ID'], $payment_id]);
    
    // Redirect to payment page
    header('Location: ' . $payment_url);
    exit;
} else {
    // Error handling
    die('Payment initialization failed. Please try again.');
}
?>
```

### 4. Create PayGate Return Handler
```php
// payments/paygate/return.php
<?php
require_once '../../config.php';
require_once '../../includes/functions.php';

// Get PayGate response
$pay_request_id = $_GET['PAY_REQUEST_ID'] ?? null;

if (!$pay_request_id) {
    die('Invalid payment response');
}

// Query payment status
$query_data = [
    'PAYGATE_ID' => PAYGATE_ID,
    'PAY_REQUEST_ID' => $pay_request_id,
    'REFERENCE' => $_GET['REFERENCE'] ?? ''
];

$checksum_string = '';
foreach ($query_data as $value) {
    $checksum_string .= $value;
}
$checksum_string .= PAYGATE_SECRET;
$query_data['CHECKSUM'] = md5($checksum_string);

// Query PayGate
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, 'https://secure.paygate.co.za/payweb3/query.trans');
curl_setopt($ch, CURLOPT_POST, 1);
curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($query_data));
curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);

$response = curl_exec($ch);
curl_close($ch);

parse_str($response, $result);

// Update payment status
$pdo = getDB();
$status = $result['TRANSACTION_STATUS'] == '1' ? 'SUCCESSFUL' : 'FAILED';

$stmt = $pdo->prepare("
    UPDATE payments 
    SET status = ?,
        raw_response = ?,
        updated_at = NOW()
    WHERE reference_id = ?
");
$stmt->execute([
    $status,
    json_encode($result),
    $_GET['REFERENCE']
]);

if ($status === 'SUCCESSFUL') {
    // Get payment details
    $stmt = $pdo->prepare("SELECT * FROM payments WHERE reference_id = ?");
    $stmt->execute([$_GET['REFERENCE']]);
    $payment = $stmt->fetch();
    
    // Update student payment status
    $stmt = $pdo->prepare("
        UPDATE students 
        SET payment_status = 'PAID',
            payment_date = NOW()
        WHERE id = ?
    ");
    $stmt->execute([$payment['student_id']]);
    
    // Credit admin wallet
    require_once '../../includes/wallet_functions.php';
    creditAdminWallet($payment['id'], PLATFORM_ADMIN_ID, 'KINE');
    
    // Redirect to success page
    header('Location: ../../success.php?payment=card&reference=' . $_GET['REFERENCE']);
} else {
    // Redirect to failure page
    header('Location: ../../register.php?payment_error=1');
}
?>
```

### 5. Update Payment Selection UI
```html
<!-- In register.php or after successful registration -->
<div class="payment-method-selector">
    <h3>Choose Payment Method</h3>
    
    <div class="payment-options">
        <!-- Mobile Money Option -->
        <div class="payment-option" id="momo-option">
            <div class="option-header">
                <i class="bi bi-phone-fill"></i>
                <h4>Mobile Money</h4>
            </div>
            <p>Pay with MTN Mobile Money</p>
            <ul>
                <li>✅ Instant payment</li>
                <li>✅ Pay from phone</li>
                <li>✅ No extra fees</li>
            </ul>
            <button onclick="showMoMoForm()" class="btn btn-primary">
                Continue with MoMo
            </button>
        </div>
        
        <!-- Card Payment Option -->
        <div class="payment-option" id="card-option">
            <div class="option-header">
                <i class="bi bi-credit-card-fill"></i>
                <h4>Card Payment</h4>
            </div>
            <p>Pay with Visa, Mastercard, etc.</p>
            <ul>
                <li>✅ Secure payment</li>
                <li>✅ All cards accepted</li>
                <li>✅ International cards</li>
            </ul>
            <button onclick="showCardForm()" class="btn btn-primary">
                Continue with Card
            </button>
        </div>
    </div>
</div>

<style>
.payment-options {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
    gap: 2rem;
    margin: 2rem 0;
}

.payment-option {
    border: 2px solid #e0e0e0;
    border-radius: 12px;
    padding: 2rem;
    text-align: center;
    transition: all 0.3s;
}

.payment-option:hover {
    border-color: #667eea;
    box-shadow: 0 8px 20px rgba(102, 126, 234, 0.2);
    transform: translateY(-5px);
}

.option-header {
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 1rem;
    margin-bottom: 1rem;
}

.option-header i {
    font-size: 3rem;
    color: #667eea;
}

.payment-option ul {
    list-style: none;
    padding: 0;
    margin: 1rem 0;
    text-align: left;
}

.payment-option li {
    padding: 0.5rem 0;
}
</style>
```

## 💰 Revenue Sharing with Card Payments

Card payments will follow the same model:
- **Student pays**: SZL 3.00
- **Card processing fee**: ~3% (SZL 0.09) - paid by you or student
- **Platform fee**: SZL 0.50 (your fee)
- **Admin gets**: SZL 2.50 - card fee

### Option A: You Absorb Card Fees
- Admin gets: SZL 2.41 (SZL 2.50 - SZL 0.09)
- You get: SZL 0.50

### Option B: Student Pays Card Fee
- Student pays: SZL 3.09 (SZL 3.00 + 3%)
- Admin gets: SZL 2.50
- You get: SZL 0.50
- Card fee: SZL 0.09 (to payment processor)

## 📊 Payment Flow Comparison

### Mobile Money Flow:
```
Student → Enters MoMo number → Push notification → Approves → Done
```

### Card Payment Flow:
```
Student → Enters card details → Redirect to PayGate → Card verification → Done
```

## 🎯 Benefits of Adding Card Payment

1. **More Payment Options**
   - Students without MoMo can pay
   - International students can pay
   - Parents can use their cards

2. **Higher Conversion**
   - More students complete registration
   - Reduces payment barriers
   - Professional payment experience

3. **Backup Option**
   - If MoMo is down, cards work
   - Multiple payment channels
   - Better reliability

## 🚨 Important Notes

### Card Payment Fees:
- **PayGate**: ~2.5-3.5% per transaction
- **Stripe**: ~3.5% + SZL 2.00 per transaction
- **MTN Card**: ~3-4% per transaction

### Settlement Time:
- **MoMo**: Instant
- **Cards**: 1-3 business days

### Requirements:
1. Business registration documents
2. Bank account
3. Verification process (1-2 weeks)
4. Initial setup fees may apply

## ✅ Recommended Next Steps

1. **Register with PayGate**
   - Apply online
   - Get test credentials
   - Test integration

2. **Implement Card Payment**
   - Add payment selection UI
   - Integrate PayGate API
   - Test with test cards

3. **Update Payment Tracking**
   - Add `payment_method` column to payments table
   - Track MoMo vs Card separately
   - Show payment method in admin dashboard

4. **Test Both Methods**
   - Test MoMo payment
   - Test card payment
   - Verify wallet crediting works for both

Would you like me to implement the card payment integration now?
