<?php
/**
 * System Health Monitoring Cron Job
 * Monitors system resources and performance
 */

require_once __DIR__ . '/../config.php';

// Set execution time limit
set_time_limit(60); // 1 minute

// Log file
$logFile = __DIR__ . '/../logs/system_health.log';

// Ensure logs directory exists
if (!is_dir(__DIR__ . '/../logs')) {
    mkdir(__DIR__ . '/../logs', 0755, true);
}

function logMessage($message) {
    global $logFile;
    $timestamp = date('Y-m-d H:i:s');
    $logEntry = "[$timestamp] $message" . PHP_EOL;
    file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
    echo $logEntry;
}

function getSystemInfo() {
    $info = [];
    
    // Memory usage
    $info['memory_usage'] = memory_get_usage(true);
    $info['memory_peak'] = memory_get_peak_usage(true);
    $info['memory_limit'] = ini_get('memory_limit');
    
    // Disk space
    $info['disk_free'] = disk_free_space(__DIR__ . '/../');
    $info['disk_total'] = disk_total_space(__DIR__ . '/../');
    $info['disk_used'] = $info['disk_total'] - $info['disk_free'];
    $info['disk_percent'] = round(($info['disk_used'] / $info['disk_total']) * 100, 2);
    
    // Load average (if available)
    if (function_exists('sys_getloadavg')) {
        $load = sys_getloadavg();
        $info['load_1min'] = $load[0];
        $info['load_5min'] = $load[1];
        $info['load_15min'] = $load[2];
    }
    
    // PHP version and extensions
    $info['php_version'] = PHP_VERSION;
    $info['php_sapi'] = PHP_SAPI;
    
    // Server info
    $info['server_software'] = $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown';
    $info['server_name'] = $_SERVER['SERVER_NAME'] ?? 'Unknown';
    
    return $info;
}

logMessage("=== SYSTEM HEALTH CHECK STARTED ===");

try {
    $systemInfo = getSystemInfo();
    
    // Memory usage check
    $memoryUsageMB = round($systemInfo['memory_usage'] / 1024 / 1024, 2);
    $memoryPeakMB = round($systemInfo['memory_peak'] / 1024 / 1024, 2);
    $memoryLimit = $systemInfo['memory_limit'];
    
    logMessage("Memory usage: {$memoryUsageMB}MB (Peak: {$memoryPeakMB}MB, Limit: {$memoryLimit})");
    
    if ($memoryPeakMB > 128) {
        logMessage("WARNING: High memory usage detected");
    }
    
    // Disk space check
    $diskFreeGB = round($systemInfo['disk_free'] / 1024 / 1024 / 1024, 2);
    $diskTotalGB = round($systemInfo['disk_total'] / 1024 / 1024 / 1024, 2);
    $diskUsedGB = round($systemInfo['disk_used'] / 1024 / 1024 / 1024, 2);
    $diskPercent = $systemInfo['disk_percent'];
    
    logMessage("Disk usage: {$diskUsedGB}GB / {$diskTotalGB}GB ({$diskPercent}%)");
    
    if ($diskPercent > 90) {
        logMessage("CRITICAL: Disk usage is above 90%");
    } elseif ($diskPercent > 80) {
        logMessage("WARNING: Disk usage is above 80%");
    }
    
    // Load average check
    if (isset($systemInfo['load_1min'])) {
        $load1min = $systemInfo['load_1min'];
        $load5min = $systemInfo['load_5min'];
        $load15min = $systemInfo['load_15min'];
        
        logMessage("Load average: 1min={$load1min}, 5min={$load5min}, 15min={$load15min}");
        
        if ($load1min > 5) {
            logMessage("WARNING: High system load detected");
        }
    }
    
    // Database connection check
    try {
        $pdo = getDB();
        $startTime = microtime(true);
        $stmt = $pdo->query("SELECT 1");
        $endTime = microtime(true);
        $dbResponseTime = round(($endTime - $startTime) * 1000, 2);
        
        logMessage("Database response time: {$dbResponseTime}ms");
        
        if ($dbResponseTime > 1000) {
            logMessage("WARNING: Slow database response time");
        }
        
    } catch (Exception $e) {
        logMessage("ERROR: Database connection failed: " . $e->getMessage());
    }
    
    // Check for stuck processes
    $processes = [];
    if (function_exists('exec')) {
        exec('ps aux | grep -E "(php|python)" | grep -v grep | wc -l', $processes);
        $processCount = intval($processes[0]);
        logMessage("Active PHP/Python processes: $processCount");
        
        if ($processCount > 20) {
            logMessage("WARNING: High number of processes running");
        }
    }
    
    // Check log file sizes
    $logsDir = __DIR__ . '/../logs';
    if (is_dir($logsDir)) {
        $logFiles = glob($logsDir . '/*.log');
        $totalLogSize = 0;
        $largeLogs = [];
        
        foreach ($logFiles as $logFile) {
            $size = filesize($logFile);
            $totalLogSize += $size;
            
            if ($size > 10485760) { // 10MB
                $largeLogs[] = basename($logFile) . ' (' . round($size/1024/1024, 2) . 'MB)';
            }
        }
        
        $totalLogSizeMB = round($totalLogSize / 1024 / 1024, 2);
        logMessage("Total log files size: {$totalLogSizeMB}MB");
        
        if (count($largeLogs) > 0) {
            logMessage("Large log files: " . implode(', ', $largeLogs));
        }
    }
    
    // Check for error logs
    $errorLog = ini_get('error_log');
    if ($errorLog && file_exists($errorLog)) {
        $errorLogSize = filesize($errorLog);
        if ($errorLogSize > 0) {
            logMessage("PHP error log size: " . round($errorLogSize / 1024, 2) . "KB");
        }
    }
    
    // System uptime
    if (function_exists('exec')) {
        exec('uptime', $uptime);
        if (!empty($uptime[0])) {
            logMessage("System uptime: " . $uptime[0]);
        }
    }
    
    // PHP configuration check
    $maxExecutionTime = ini_get('max_execution_time');
    $maxInputTime = ini_get('max_input_time');
    $postMaxSize = ini_get('post_max_size');
    $uploadMaxFilesize = ini_get('upload_max_filesize');
    
    logMessage("PHP configuration:");
    logMessage("- Max execution time: {$maxExecutionTime}s");
    logMessage("- Max input time: {$maxInputTime}s");
    logMessage("- Post max size: {$postMaxSize}");
    logMessage("- Upload max filesize: {$uploadMaxFilesize}");
    
    // Check if we're in a web server environment
    if (php_sapi_name() === 'cli') {
        logMessage("Running in CLI mode");
    } else {
        logMessage("Running in web server mode");
    }
    
    logMessage("=== SYSTEM HEALTH CHECK COMPLETED ===");
    
} catch (Exception $e) {
    logMessage("ERROR: System health check failed: " . $e->getMessage());
    exit(1);
}

logMessage("System health check completed successfully");
?>
