<?php
/**
 * Email Queue Processing Cron Job
 * Processes pending email notifications
 */

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../includes/functions.php';

// Set execution time limit
set_time_limit(300); // 5 minutes

// Log file
$logFile = __DIR__ . '/../logs/email_queue.log';

// Ensure logs directory exists
if (!is_dir(__DIR__ . '/../logs')) {
    mkdir(__DIR__ . '/../logs', 0755, true);
}

function logMessage($message) {
    global $logFile;
    $timestamp = date('Y-m-d H:i:s');
    $logEntry = "[$timestamp] $message" . PHP_EOL;
    file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
    echo $logEntry;
}

function sendEmail($to, $subject, $message, $from = null) {
    if (!$from) {
        $from = 'noreply@' . ($_SERVER['SERVER_NAME'] ?? 'localhost');
    }
    
    $headers = [
        'From: ' . $from,
        'Reply-To: ' . $from,
        'X-Mailer: PHP/' . phpversion(),
        'Content-Type: text/html; charset=UTF-8'
    ];
    
    return mail($to, $subject, $message, implode("\r\n", $headers));
}

logMessage("=== EMAIL QUEUE PROCESSING STARTED ===");

try {
    $pdo = getDB();
    
    // Get pending email verifications (older than 1 hour, not sent)
    $stmt = $pdo->prepare("
        SELECT id, first_name, last_name, email, email_verification_token, created_at
        FROM students 
        WHERE email_verified = 0 
        AND email_verification_token IS NOT NULL
        AND created_at < NOW() - INTERVAL 1 HOUR
        AND created_at > NOW() - INTERVAL 24 HOURS
        ORDER BY created_at ASC
        LIMIT 50
    ");
    $stmt->execute();
    $pendingVerifications = $stmt->fetchAll();
    
    $verificationEmailsSent = 0;
    $verificationEmailsFailed = 0;
    
    foreach ($pendingVerifications as $student) {
        try {
            $verificationLink = "https://" . ($_SERVER['SERVER_NAME'] ?? 'localhost') . 
                               "/verify-email.php?token=" . $student['email_verification_token'];
            
            $subject = "Verify Your Email - " . APP_NAME;
            $message = "
                <html>
                <body>
                    <h2>Welcome to " . APP_NAME . "!</h2>
                    <p>Hello {$student['first_name']} {$student['last_name']},</p>
                    <p>Please verify your email address by clicking the link below:</p>
                    <p><a href='{$verificationLink}' style='background-color: #007bff; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px;'>Verify Email</a></p>
                    <p>If the button doesn't work, copy and paste this link into your browser:</p>
                    <p>{$verificationLink}</p>
                    <p>This link will expire in 24 hours.</p>
                    <p>Best regards,<br>The " . APP_NAME . " Team</p>
                </body>
                </html>
            ";
            
            if (sendEmail($student['email'], $subject, $message)) {
                $verificationEmailsSent++;
                logMessage("Sent verification email to: {$student['email']}");
            } else {
                $verificationEmailsFailed++;
                logMessage("Failed to send verification email to: {$student['email']}");
            }
            
        } catch (Exception $e) {
            $verificationEmailsFailed++;
            logMessage("ERROR: Failed to process verification for {$student['email']}: " . $e->getMessage());
        }
    }
    
    // Get students who need password reminders (verified but no recent activity)
    $stmt = $pdo->prepare("
        SELECT id, first_name, last_name, email, last_name as password_hint
        FROM students 
        WHERE email_verified = 1 
        AND created_at < NOW() - INTERVAL 7 DAY
        AND created_at > NOW() - INTERVAL 30 DAY
        ORDER BY created_at ASC
        LIMIT 20
    ");
    $stmt->execute();
    $passwordReminders = $stmt->fetchAll();
    
    $reminderEmailsSent = 0;
    $reminderEmailsFailed = 0;
    
    foreach ($passwordReminders as $student) {
        try {
            $loginUrl = "https://" . ($_SERVER['SERVER_NAME'] ?? 'localhost') . "/login.php";
            
            $subject = "Your " . APP_NAME . " Login Credentials";
            $message = "
                <html>
                <body>
                    <h2>Your " . APP_NAME . " Login Information</h2>
                    <p>Hello {$student['first_name']} {$student['last_name']},</p>
                    <p>Here are your login credentials for " . APP_NAME . ":</p>
                    <div style='background-color: #f8f9fa; padding: 15px; border-radius: 5px; margin: 10px 0;'>
                        <p><strong>Username:</strong> {$student['email']}</p>
                        <p><strong>Password:</strong> {$student['password_hint']}</p>
                    </div>
                    <p>You can log in at: <a href='{$loginUrl}'>{$loginUrl}</a></p>
                    <p>If you have any questions, please contact support.</p>
                    <p>Best regards,<br>The " . APP_NAME . " Team</p>
                </body>
                </html>
            ";
            
            if (sendEmail($student['email'], $subject, $message)) {
                $reminderEmailsSent++;
                logMessage("Sent password reminder to: {$student['email']}");
            } else {
                $reminderEmailsFailed++;
                logMessage("Failed to send password reminder to: {$student['email']}");
            }
            
        } catch (Exception $e) {
            $reminderEmailsFailed++;
            logMessage("ERROR: Failed to process reminder for {$student['email']}: " . $e->getMessage());
        }
    }
    
    // Clean up old email verification tokens (older than 24 hours)
    $stmt = $pdo->prepare("
        UPDATE students 
        SET email_verification_token = NULL, email_verification_expires = NULL
        WHERE email_verified = 0 
        AND email_verification_expires < NOW()
    ");
    $stmt->execute();
    $cleanedTokens = $stmt->rowCount();
    
    if ($cleanedTokens > 0) {
        logMessage("Cleaned up expired verification tokens: $cleanedTokens");
    }
    
    // Get email statistics
    $stmt = $pdo->query("
        SELECT 
            COUNT(*) as total_students,
            SUM(CASE WHEN email_verified = 1 THEN 1 ELSE 0 END) as verified_students,
            SUM(CASE WHEN email_verified = 0 THEN 1 ELSE 0 END) as unverified_students
        FROM students
    ");
    $stats = $stmt->fetch();
    
    logMessage("Email queue processing completed:");
    logMessage("- Verification emails sent: $verificationEmailsSent");
    logMessage("- Verification emails failed: $verificationEmailsFailed");
    logMessage("- Password reminders sent: $reminderEmailsSent");
    logMessage("- Password reminders failed: $reminderEmailsFailed");
    logMessage("- Expired tokens cleaned: $cleanedTokens");
    logMessage("- Total students: {$stats['total_students']}");
    logMessage("- Verified students: {$stats['verified_students']}");
    logMessage("- Unverified students: {$stats['unverified_students']}");
    
} catch (Exception $e) {
    logMessage("ERROR: Email queue processing failed: " . $e->getMessage());
    exit(1);
}

logMessage("=== EMAIL QUEUE PROCESSING COMPLETED ===");
?>
