<?php
/**
 * Performance Monitoring Cron Job
 * Monitors application performance and response times
 */

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../includes/functions.php';

// Set execution time limit
set_time_limit(60); // 1 minute

// Log file
$logFile = __DIR__ . '/../logs/performance.log';

// Ensure logs directory exists
if (!is_dir(__DIR__ . '/../logs')) {
    mkdir(__DIR__ . '/../logs', 0755, true);
}

function logMessage($message) {
    global $logFile;
    $timestamp = date('Y-m-d H:i:s');
    $logEntry = "[$timestamp] $message" . PHP_EOL;
    file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
    echo $logEntry;
}

function measureResponseTime($url) {
    $startTime = microtime(true);
    $context = stream_context_create([
        'http' => [
            'timeout' => 10,
            'method' => 'GET'
        ]
    ]);
    
    $response = @file_get_contents($url, false, $context);
    $endTime = microtime(true);
    
    return [
        'time' => round(($endTime - $startTime) * 1000, 2),
        'success' => $response !== false,
        'size' => $response ? strlen($response) : 0
    ];
}

logMessage("=== PERFORMANCE MONITORING STARTED ===");

try {
    $performanceData = [];
    
    // Memory usage
    $memoryUsage = memory_get_usage(true);
    $memoryPeak = memory_get_peak_usage(true);
    $memoryLimit = ini_get('memory_limit');
    
    $performanceData['memory_usage_mb'] = round($memoryUsage / 1024 / 1024, 2);
    $performanceData['memory_peak_mb'] = round($memoryPeak / 1024 / 1024, 2);
    $performanceData['memory_limit'] = $memoryLimit;
    
    logMessage("Memory usage: {$performanceData['memory_usage_mb']}MB (Peak: {$performanceData['memory_peak_mb']}MB)");
    
    // Database performance
    $dbStartTime = microtime(true);
    try {
        $pdo = getDB();
        
        // Test simple query
        $stmt = $pdo->query("SELECT COUNT(*) as count FROM students");
        $result = $stmt->fetch();
        $studentCount = $result['count'];
        
        // Test complex query
        $stmt = $pdo->query("
            SELECT s.*, g.name as grade_name 
            FROM students s 
            LEFT JOIN grades g ON s.grade_id = g.id 
            WHERE s.email_verified = 1 
            ORDER BY s.created_at DESC 
            LIMIT 10
        ");
        $students = $stmt->fetchAll();
        
        $dbEndTime = microtime(true);
        $dbResponseTime = round(($dbEndTime - $dbStartTime) * 1000, 2);
        
        $performanceData['db_response_time_ms'] = $dbResponseTime;
        $performanceData['student_count'] = $studentCount;
        
        logMessage("Database response time: {$dbResponseTime}ms");
        logMessage("Total students: $studentCount");
        
        if ($dbResponseTime > 1000) {
            logMessage("WARNING: Slow database response time");
        }
        
    } catch (Exception $e) {
        logMessage("ERROR: Database performance test failed: " . $e->getMessage());
        $performanceData['db_response_time_ms'] = -1;
    }
    
    // Moodle API performance
    $moodleStartTime = microtime(true);
    try {
        $moodleInfo = callMoodleAPI('core_webservice_get_site_info');
        $moodleEndTime = microtime(true);
        $moodleResponseTime = round(($moodleEndTime - $moodleStartTime) * 1000, 2);
        
        $performanceData['moodle_response_time_ms'] = $moodleResponseTime;
        
        logMessage("Moodle API response time: {$moodleResponseTime}ms");
        
        if ($moodleResponseTime > 5000) {
            logMessage("WARNING: Slow Moodle API response time");
        }
        
    } catch (Exception $e) {
        logMessage("ERROR: Moodle API performance test failed: " . $e->getMessage());
        $performanceData['moodle_response_time_ms'] = -1;
    }
    
    // File system performance
    $fsStartTime = microtime(true);
    $testFile = __DIR__ . '/../logs/performance_test_' . time() . '.tmp';
    file_put_contents($testFile, str_repeat('test', 1000));
    $fsEndTime = microtime(true);
    $fsResponseTime = round(($fsEndTime - $fsStartTime) * 1000, 2);
    unlink($testFile);
    
    $performanceData['filesystem_response_time_ms'] = $fsResponseTime;
    logMessage("File system response time: {$fsResponseTime}ms");
    
    // Disk I/O performance
    $diskStartTime = microtime(true);
    $logsDir = __DIR__ . '/../logs';
    $logFiles = glob($logsDir . '/*.log');
    $totalLogSize = 0;
    foreach ($logFiles as $file) {
        $totalLogSize += filesize($file);
    }
    $diskEndTime = microtime(true);
    $diskResponseTime = round(($diskEndTime - $diskStartTime) * 1000, 2);
    
    $performanceData['disk_io_response_time_ms'] = $diskResponseTime;
    $performanceData['total_log_size_mb'] = round($totalLogSize / 1024 / 1024, 2);
    
    logMessage("Disk I/O response time: {$diskResponseTime}ms");
    logMessage("Total log size: {$performanceData['total_log_size_mb']}MB");
    
    // CPU usage (if available)
    if (function_exists('sys_getloadavg')) {
        $load = sys_getloadavg();
        $performanceData['load_1min'] = $load[0];
        $performanceData['load_5min'] = $load[1];
        $performanceData['load_15min'] = $load[2];
        
        logMessage("System load: 1min={$load[0]}, 5min={$load[1]}, 15min={$load[2]}");
        
        if ($load[0] > 5) {
            logMessage("WARNING: High system load detected");
        }
    }
    
    // Check for performance bottlenecks
    $bottlenecks = [];
    
    if ($performanceData['db_response_time_ms'] > 1000) {
        $bottlenecks[] = "Database response time is slow";
    }
    
    if ($performanceData['moodle_response_time_ms'] > 5000) {
        $bottlenecks[] = "Moodle API response time is slow";
    }
    
    if ($performanceData['filesystem_response_time_ms'] > 100) {
        $bottlenecks[] = "File system response time is slow";
    }
    
    if (isset($performanceData['load_1min']) && $performanceData['load_1min'] > 5) {
        $bottlenecks[] = "High system load detected";
    }
    
    if (count($bottlenecks) > 0) {
        logMessage("Performance bottlenecks detected:");
        foreach ($bottlenecks as $bottleneck) {
            logMessage("- $bottleneck");
        }
    } else {
        logMessage("No performance bottlenecks detected");
    }
    
    // Check for memory leaks
    $memoryAfter = memory_get_usage(true);
    $memoryIncrease = $memoryAfter - $memoryUsage;
    
    if ($memoryIncrease > 1048576) { // 1MB
        logMessage("WARNING: Potential memory leak detected (increase: " . round($memoryIncrease/1024, 2) . "KB)");
    } else {
        logMessage("Memory usage stable");
    }
    
    // Check for long-running processes
    if (function_exists('exec')) {
        exec('ps aux | grep -E "(php|python)" | grep -v grep | wc -l', $processes);
        $processCount = intval($processes[0]);
        $performanceData['process_count'] = $processCount;
        
        logMessage("Active processes: $processCount");
        
        if ($processCount > 20) {
            logMessage("WARNING: High number of processes running");
        }
    }
    
    // Check disk space
    $diskFree = disk_free_space(__DIR__ . '/../');
    $diskTotal = disk_total_space(__DIR__ . '/../');
    $diskUsed = $diskTotal - $diskFree;
    $diskPercent = round(($diskUsed / $diskTotal) * 100, 2);
    
    $performanceData['disk_usage_percent'] = $diskPercent;
    $performanceData['disk_free_gb'] = round($diskFree / 1024 / 1024 / 1024, 2);
    
    logMessage("Disk usage: {$diskPercent}% (Free: {$performanceData['disk_free_gb']}GB)");
    
    if ($diskPercent > 90) {
        logMessage("WARNING: Disk usage is above 90%");
    }
    
    // Performance summary
    logMessage("Performance monitoring completed:");
    logMessage("- Database response: {$performanceData['db_response_time_ms']}ms");
    logMessage("- Moodle API response: {$performanceData['moodle_response_time_ms']}ms");
    logMessage("- File system response: {$performanceData['filesystem_response_time_ms']}ms");
    logMessage("- Disk I/O response: {$diskResponseTime}ms");
    logMessage("- Memory usage: {$performanceData['memory_usage_mb']}MB");
    logMessage("- Disk usage: {$diskPercent}%");
    logMessage("- Bottlenecks: " . count($bottlenecks));
    
    // Save performance data to file for analysis
    $performanceFile = __DIR__ . '/../logs/performance_data.json';
    $performanceData['timestamp'] = date('Y-m-d H:i:s');
    $performanceData['bottlenecks'] = $bottlenecks;
    
    $existingData = [];
    if (file_exists($performanceFile)) {
        $existingData = json_decode(file_get_contents($performanceFile), true) ?: [];
    }
    
    $existingData[] = $performanceData;
    
    // Keep only last 100 entries
    if (count($existingData) > 100) {
        $existingData = array_slice($existingData, -100);
    }
    
    file_put_contents($performanceFile, json_encode($existingData, JSON_PRETTY_PRINT));
    
} catch (Exception $e) {
    logMessage("ERROR: Performance monitoring failed: " . $e->getMessage());
    exit(1);
}

logMessage("=== PERFORMANCE MONITORING COMPLETED ===");
?>
